<?php

namespace Inside\Http\Controllers\Admin;

use Illuminate\Http\JsonResponse;
use Inside\Facades\RouteFormatter;
use Inside\Http\Controllers\AdminController;
use OpenApi\Annotations as OA;

class RouteController extends AdminController
{
    /**
     * @OA\Schema(
     *     schema="InsideRoute",
     *     title="Route inside",
     *     type="object",
     *     description="Les informations d'un endpoint d'inside",
     *     @OA\Property(
     *         property="domain",
     *         title="Le domaine",
     *         type="string",
     *         nullable=true,
     *         description="Le sous-domaine ou s'applique la route. Si null la route s'applique au domaine principal.",
     *         default=null,
     *         example=null
     *     ),
     *     @OA\Property(
     *         property="method",
     *         title="La méthode",
     *         type="string",
     *         description="La méthode de la route (get, post, put, patch, delete ou options) en capital",
     *         default="GET",
     *         example="GET"
     *     ),
     *     @OA\Property(
     *         property="uri",
     *         title="L'URI",
     *         type="string",
     *         description="L'uri de la route en relative.",
     *         example="/api/v1/users"
     *     ),
     *     @OA\Property(
     *         property="name",
     *         title="Le nom machine",
     *         type="string",
     *         nullable=true,
     *         description="Le nom machine de la route.",
     *         default=null,
     *         example="users.list"
     *     ),
     *     @OA\Property(
     *         property="action",
     *         title="L'action",
     *         type="string",
     *         description="Ce qui est déclenché en back sur cette route.",
     *         example="UserController@list"
     *     ),
     *     @OA\Property(
     *         property="middleware",
     *         title="Les middlewares",
     *         type="array",
     *         description="Les middlewares appliqués à cette route.",
     *         items=@OA\Items(type="string", description="Le nom du middleware", example="auth.api")
     *     )
     * )
     *
     * @OA\Get(
     *    path="/admin/routes/index",
     *     operationId="routeList",
     *     tags={"Admin"},
     *     summary="Récupère la liste de tous les endpoints",
     *     @OA\Response(
     *         response=200,
     *         description="Liste des routes",
     *         @OA\JsonContent(
     *             type="array",
     *             @OA\Items(ref="#/components/schemas/InsideRoute")
     *        )
     *    )
     * )
     *
     * @return JsonResponse
     */
    public function index(): JsonResponse
    {
        return response()->json(
            collect(RouteFormatter::getRoutes())
                ->map(function ($route) {
                    $route['middleware'] = empty($route['middleware']) ?
                        [] :
                        explode(',', $route['middleware']);

                    return $route;
                })
                ->values()
        );
    }

    /**
     * @OA\Post(
     *    path="/admin/routes/clear",
     *     operationId="routeClear",
     *     tags={"Admin"},
     *     summary="Suppression du cache des routes ( attention, perte de performance )",
     *     @OA\Response(
     *         response=200,
     *         @OA\JsonContent(ref="#/components/schemas/AdminResponse")
     *    )
     * )
     *
     * @return JsonResponse
     */
    public function clear(): JsonResponse
    {
        return response()->json($this->callArtisan('inside:route:clear'));
    }

    /**
     * @OA\Post(
     *    path="/admin/routes/cache",
     *     operationId="routeCache",
     *     tags={"Admin"},
     *     summary="Mise encache des routes inside.",
     *     @OA\Response(
     *         response=200,
     *         @OA\JsonContent(ref="#/components/schemas/AdminResponse")
     *    )
     * )
     * @return JsonResponse
     */
    public function cache(): JsonResponse
    {
        return response()->json($this->callArtisan('inside:route:cache'));
    }
}
