<?php

namespace Inside\Http\Controllers\Admin;

use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Contracts\Queue\Factory as QueueFactory;
use Illuminate\Pagination\AbstractPaginator;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Queue;
use Inside\Authentication\Models\User;
use InvalidArgumentException;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Process\Process;

/**
 * Class QueueController
 *
 * All admin queue actions
 */
class QueueController
{
    /**
     * @var array map for worker / queue
     */
    protected $workerQueueMap = [];

    /**
     * @var string[]
     */
    protected $supportedQueues = [];

    /**
     * QueueController constructor.
     */
    public function __construct()
    {
        // Getting default driver start queue manager and load config
        $driver = Queue::getDefaultDriver();

        $this->supportedQueues = ['default', 'indexation', 'images', 'lts'];

        $this->workerQueueMap = [
            'default' => [
                get_high_priority_queue_name(),
                get_default_queue_name(),
                get_low_priority_queue_name(),
            ],
            'indexation' => [
                config('scout.queue.queue'),
            ],
            'images' => [
                get_images_queue_name(),
            ],
        ];
        if (config('queue.connections.lts', false)) {
            $this->workerQueueMap['lts'] = [get_lts_queue_name()];
        }

        // Add custom workers
        foreach (config('workers', []) as $worker => $options) {
            if (! array_key_exists('name', $options)) {
                continue;
            }

            $this->supportedQueues[] = $worker;
            $this->workerQueueMap[$worker] = [$options['name']];
        }
    }

    /**
     * Get the list of available queues
     *
     * @return array
     */
    public function list()
    {
        return [
            'status' => true,
            'data' => $this->supportedQueues,
        ];
    }

    /**
     * Get some infos/stats on a queue using the workername
     *
     * @param string $workerName
     * @return array
     * @throws Exception
     */
    public function queueStats(string $workerName): array
    {
        if (! in_array($workerName, $this->supportedQueues)) {
            throw new InvalidArgumentException('Wrong workername ['.$workerName.']');
        }

        if (($queues = $this->workerQueueMap[$workerName]) === null) {
            throw new InvalidArgumentException('Can\'t find queue for ['.$workerName.']');
        }
        if (Queue::getConnectionName() != 'redis') {
            throw new Exception('Queue driver ['.Queue::getConnectionName().'] Not yet supported');
        }
        $connection = app(QueueFactory::class)->connection(Queue::getConnectionName());
        $redis = $connection->getRedis()->connection(config('queue.connections.redis.connection'));

        $stats = [];
        foreach ($queues as $queue) {
            $stats[$queue] = [
                'size' => Queue::size($queue),
                'primary' => $redis->llen(Queue::getQueue($queue)),
                'delayed' => $redis->zcount(Queue::getQueue($queue).':delayed', '-inf', '+inf'),
                'reserved' => $redis->zcount(Queue::getQueue($queue).':reserved', '-inf', '+inf'),
            ];
        }

        return $stats;
    }

    /**
     * Get infos on failed jobs for $workerName
     *
     * @param string $workerName
     * @return AbstractPaginator
     * @throws Exception
     */
    public function queueGetFailed(string $workerName): AbstractPaginator
    {
        if (! in_array($workerName, $this->supportedQueues)) {
            throw new InvalidArgumentException('Wrong workername ['.$workerName.']');
        }

        if (($queues = $this->workerQueueMap[$workerName]) === null) {
            throw new InvalidArgumentException('Can\'t find queue for ['.$workerName.']');
        }
        if (Queue::getConnectionName() != 'redis') {
            throw new Exception('Queue driver ['.Queue::getConnectionName().'] Not yet supported');
        }

        /** @var AbstractPaginator $jobs */
        $jobs =
            DB::table(config('queue.failed.table'))->whereIn('queue', $queues)->orderByDesc('failed_at')->paginate();
        $jobs->getCollection()->map( // TODO check si le map est bien utile / utilisé
            function ($job) {
                return $this->decode($job);
            }
        );

        return $jobs;
    }

    /**
     * Force to restart workers
     *
     * NOTE: take care, this will nicely kill EVERY running workers
     *
     * @return array
     */
    public function queueRestart()
    {
        return ['success' => Artisan::call('queue:restart') === 0, 'data' => Artisan::output()];
    }

    /**
     * Flush failed jobs
     *
     * @return array
     */
    public function queueFlushFailed()
    {
        return ['success' => Artisan::call('queue:flush') === 0, 'data' => Artisan::output()];
    }

    /**
     * Try to get pm2 status
     *
     * @return array
     */
    public function getPm2Status()
    {
        // Try to exec pm2 status from shell
        // /usr/local/lib/nodejs/bin
        $process = new Process(['pm2', 'jlist']);
        try {
            $process->run();
        } catch (\Throwable $e) {
            throw new ProcessFailedException($process);
        }

        if (! $process->isSuccessful()) {
            throw new ProcessFailedException($process);
        }

        $status = collect(json_decode($process->getOutput(), true));

        return [
            'data' => $status->transform(
                function ($informations) {
                    return [
                        'id' => $informations['pm_id'],
                        'name' => $informations['name'],
                        'pid' => $informations['pid'],
                        'uptime' => Carbon::createFromTimestampMs($informations['pm2_env']['pm_uptime'])
                            ->diffForHumans(),
                        'restarted' => $informations['pm2_env']['restart_time'],
                        'status' => $informations['pm2_env']['status'],
                        'mem' => inside_bytes_to_human($informations['monit']['memory']),
                        'cpu' => $informations['monit']['cpu'],
                        'user' => $informations['pm2_env']['username'],
                    ];
                }
            ),
        ];
    }

    /**
     * Decode job informations
     *
     * @param mixed $job
     * @return mixed
     */
    protected function decode($job)
    {
        $job->payload = json_decode($job->payload);
        $job->retried_by =
            isset($job->retried_by) ? collect(json_decode($job->retried_by))->sortByDesc('retried_at')->values() : [];

        return $job;
    }
}
