<?php

namespace Inside\Http\Controllers\Admin;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Facades\ExportImport;
use Inside\I18n\Facades\Translation;
use Inside\Layout\Models\Layout;
use Inside\Layout\Models\LayoutPath;
use Inside\Settings\Facades\Settings;
use InvalidArgumentException;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use ZipArchive;

/**
 * Class ExportImportController
 *
 * Export & import inside stuff
 */
class ExportImportController
{
    /**
     * Export all layouts in a zip file
     *
     * @return BinaryFileResponse
     */
    public function exportLayoutsToZip(): BinaryFileResponse
    {
        // Export layouts as a big json data
        $data = Layout::with('paths')->get()->toJson(JSON_PRETTY_PRINT);

        // Prepare a zip
        $zipFilenameParts = [
            'layouts',
            config('app.code'),
            inside_version(),
            Carbon::now()->timestamp,
            Str::random(64),
        ];
        $zipFilename = implode('_', $zipFilenameParts).'.zip';
        $zipPath = str_replace('/', DIRECTORY_SEPARATOR, storage_path('app/tmp/'.$zipFilename));
        $zipFile = new ZipArchive();
        $zipFile->open($zipPath, ZipArchive::CREATE);
        $zipFile->addFromString('layouts.json', $data);
        $zipFile->close();

        // Download zip
        $response = response()->download(
            $zipPath,
            $zipFilename,
            ['Content-Type: application/octet-stream', 'Content-Length: '.filesize($zipPath)]
        );

        if (! windows_os()) {
            $response->deleteFileAfterSend(true);
        }

        return $response;
    }

    /**
     * @return BinaryFileResponse
     */
    public function exportLayoutsToJson()
    {
        $layouts = Layout::with('paths')->get()->toJson();
        File::put(cms_base_path('layouts.json'), $layouts);

        return response()->download(cms_base_path('layouts.json'));
    }

    /**
     * Import layouts to this instance
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function importLayoutsFromZip(Request $request): JsonResponse
    {
        $zipPath = $request->get('layouts');
        if (! $zipPath) {
            throw new InvalidArgumentException('Missing layouts file path');
        }
        if (! Storage::disk('local')->exists($zipPath)) {
            throw new InvalidArgumentException('File path does not exist');
        }

        /** @phpstan-ignore-next-line */
        $zipPath = Storage::disk('local')->path($zipPath);
        $zipFile = new ZipArchive();
        $zipFile->open($zipPath, ZipArchive::CREATE);
        $data = $zipFile->getFromName('layouts.json');
        $zipFile->close();
        if (! $data) {
            throw new InvalidArgumentException('Wrong layouts file');
        }
        $data = json_decode($data);
        if ($data === null) {
            throw new InvalidArgumentException('Wrong layouts file');
        }
        $data = collect($data);

        return response()->json(
            [
                'success' => (($count = ExportImport::importLayouts($data)) > 0) && Artisan::call('cache:clear'),
                'data'    => ['imported' => $count],
            ]
        );
    }

    /**
     * Export all exportable information from this inside
     *
     * @return BinaryFileResponse
     */
    public function exportInsideToZip(): BinaryFileResponse
    {
        $insideData = collect(
            [
                'layouts'  => Layout::with('paths')->get()->toJson(JSON_PRETTY_PRINT),
                'settings' => Settings::getExportableSettings()->toJson(JSON_PRETTY_PRINT),
                'i18n'     => $data = Translation::getOverrideItems(['state' => 'translated'])->each->append(
                    'locale'
                )->each->makeHidden(
                    [
                        'id',
                        'language_id',
                        'language',
                        'draft',
                        'locked',
                        'exportable',
                        'created_at',
                        'updated_at',
                    ]
                )->toJson(
                    JSON_PRETTY_PRINT
                ),
            ]
        );

        // Prepare a zip
        $zipFilenameParts = [
            'inside',
            config('app.code'),
            inside_version(),
            Carbon::now()->timestamp,
            Str::random(64),
        ];
        $zipFilename = implode('_', $zipFilenameParts).'.zip';
        $zipPath = str_replace('/', DIRECTORY_SEPARATOR, storage_path('app/tmp/'.$zipFilename));
        $zipFile = new ZipArchive();
        $zipFile->open($zipPath, ZipArchive::CREATE);
        foreach ($insideData as $fileName => $data) {
            $zipFile->addFromString($fileName.'.json', $data);
        }
        $zipFile->close();

        // Download zip
        $response = response()->download(
            $zipPath,
            $zipFilename,
            ['Content-Type: application/octet-stream', 'Content-Length: '.filesize($zipPath)]
        );

        if (! windows_os()) {
            $response->deleteFileAfterSend(true);
        }

        return $response;
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function importInsideFromZip(Request $request): JsonResponse
    {
        // Get archive
        $zipPath = $request->get('inside');
        if ($zipPath === null) {
            throw new InvalidArgumentException('Missing inside file path');
        }
        if (! Storage::disk('local')->exists($zipPath)) {
            throw new InvalidArgumentException('File path does not exist');
        }

        $insideData = [
            'layouts' => null,
            'settings' => null,
            'i18n' => null,
        ];

        /** @phpstan-ignore-next-line */
        $zipPath = Storage::disk('local')->path($zipPath);
        $zipFile = new ZipArchive();
        $zipFile->open($zipPath, ZipArchive::CREATE);
        foreach ($insideData as $name => $null) {
            try {
                if (method_exists($this, Str::camel('import_'.$name))) {
                    $data = $zipFile->getFromName($name.'.json');
                    if (empty($data)) {
                        throw new InvalidArgumentException('Wrong '.$name.' file');
                    }
                    $data = json_decode($data);
                    if ($data === null) {
                        throw new InvalidArgumentException('Wrong '.$name.' file');
                    } else {
                        $data = collect($data);
                        $insideData[$name] = $this->{Str::camel('import_'.$name)}($data);
                    }
                } else {
                    throw new InvalidArgumentException('Wrong '.$name.' file');
                }
            } catch (Exception $e) {
                $insideData[$name] = false;
            }
        }
        $zipFile->close();
        Artisan::call('cache:clear');

        return response()->json(
            [
                'success' => $insideData,
            ]
        );
    }

    /**
     * Import i18n
     *
     * @param $data
     * @return bool
     */
    protected function importI18n(Collection $data): bool
    {
        $result = Translation::importTranslationsFromData($data->toArray());

        return is_array($result['failed']) && empty($result['failed']);
    }

    /**
     * Import Settings
     *
     * @param $data
     * @return bool
     */
    protected function importSettings(Collection $data): bool
    {
        return  Settings::importSettings($data);
    }
}
