<?php

namespace Inside\Http\Controllers\Admin;

use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Inside\Authentication\Models\User;
use Inside\Content\Events\BirthdayWishesSendEvent;
use Inside\Content\Events\ExportCompletedEvent;
use Inside\Notify\Events\NotificationCreatedEvent;
use Inside\Notify\Models\Notification;
use Inside\Reaction\Events\ReactionSubscribedEvent;
use Inside\Reaction\Events\ReactionUnsubscribedEvent;
use Inside\Reaction\Models\Reaction;

class BroadcastingController
{
    public function triggerBirthday(): JsonResponse
    {
        /** @var User|null $to */
        $to = Auth::user();
        /** @var User|null $from */
        $from = User::inRandomOrder()->first();

        $success = $to !== null && $from !== null;

        if ($success) {
            event(new BirthdayWishesSendEvent($from, $to));
        }

        return response()->json(
            [
                'success' => $success,
            ]
        );
    }

    public function triggerExportCompleted(): JsonResponse
    {
        /** @var User|null $user */
        $user = Auth::user();

        if (($user !== null) && $files = Storage::disk('protected')->allFiles()) {
            event(new ExportCompletedEvent('protected', array_rand($files), $user));
        }

        return response()->json(
            [
                'success' => true,
            ]
        );
    }

    public function triggerReaction(): JsonResponse
    {
        /** @var Reaction|null $reaction */
        $reaction = Reaction::inRandomOrder()->first();

        if ($success = ($reaction !== null)) {
            if (rand() % 2 == 1) {
                event(new ReactionSubscribedEvent($reaction));
            } else {
                event(new ReactionUnsubscribedEvent($reaction));
            }
        }

        return response()->json(
            [
                'success' => $success,
            ]
        );
    }

    public function triggerNotification(): JsonResponse
    {
        /** @var User $user */
        $user = Auth::user();

        /** @var Notification|null $notification */
        $notification = $user->notifications()->inRandomOrder()->first();

        if ($success = ($notification !== null)) {
            event(new NotificationCreatedEvent($notification));
        }

        return response()->json(
            [
                'success' => $success,
            ]
        );
    }
}
