<?php

namespace Inside\Exceptions;

use Exception;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Contracts\Container\Container;
use Illuminate\Contracts\Debug\ExceptionHandler as ExceptionHandlerContract;
use Illuminate\Contracts\Routing\UrlGenerator;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Session\TokenMismatchException;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\View;
use Illuminate\Support\ViewErrorBag;
use Illuminate\Validation\ValidationException;
use Laravel\Lumen\Http\ResponseFactory;
use Psr\Log\LoggerInterface;
use Psr\Log\LogLevel;
use Symfony\Component\Console\Application as ConsoleApplication;
use Symfony\Component\Console\Exception\CommandNotFoundException;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Debug\Exception\FlattenException;
use Symfony\Component\Debug\ExceptionHandler as SymfonyExceptionHandler;
use Symfony\Component\ErrorHandler\ErrorRenderer\HtmlErrorRenderer;
use Symfony\Component\HttpFoundation\Exception\SuspiciousOperationException;
use Symfony\Component\HttpFoundation\RedirectResponse as SymfonyRedirectResponse;
use Symfony\Component\HttpFoundation\Response as SymfonyResponse;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Symfony\Component\HttpKernel\Exception\HttpExceptionInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Throwable;
use Whoops\Handler\HandlerInterface;
use Whoops\Run as Whoops;

/**
 * Exception handler.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class Handler implements ExceptionHandlerContract
{
    /**
     * A list of the exception types that should not be reported
     */
    protected array $dontReport = [
        CommandNotFoundException::class,
    ];

    /**
     * Internal exception that should not be reported!
     */
    protected array $internalDontReport = [
        AuthenticationException::class,
        AuthorizationException::class,
        HttpException::class,
        NotFoundHttpException::class,
        ModelNotFoundException::class,
        SuspiciousOperationException::class,
        TokenMismatchException::class,
        ValidationException::class,
    ];

    /*
     * @var array<class-string<\Throwable>, \Psr\Log\LogLevel::*>
     */
    protected array $levels = [];

    /**
     * don't flash on validation
     */
    protected array $dontFlash = [
        'password',
        'password_confirmation',
    ];

    protected array $severityReport = [
        E_ERROR,
    ];

    public function __construct(
        protected Container $container
    ) {
    }

    /**
     * Report or log an exception.
     *
     * This is a great spot to send exceptions to Sentry, Bugsnag, etc.
     * @throws Exception
     * @throws Throwable
     */
    public function report(Throwable $e): void
    {
        if ($this->shouldntReport($e)) {
            return;
        }

        // In debug mode, we log with default
        if (config('app.debug', false)) {
            if (is_callable($reportCallable = [$e, 'report'])) {
                $this->container->call($reportCallable);

                return;
            }

            try {
                $logger = $this->container->make(LoggerInterface::class);
            } catch (Throwable) {
                throw $e;
            }

            $level = Arr::first(
                $this->levels,
                fn ($level, $type) => $e instanceof $type,
                LogLevel::ERROR
            );

            $context = array_merge(
                $this->exceptionContext($e),
                $this->context(),
                ['exception' => $e]
            );

            method_exists($logger, $level)
                ? $logger->{$level}($e->getMessage(), $context)
                : $logger->log($level, $e->getMessage(), $context);

            return;
        }

        // On non debug mode, only log
        $severity = method_exists($e, 'getSeverity') ? $e->getSeverity() : E_ERROR;

        try {
            $logger = $this->container->make(LoggerInterface::class);
        } catch (Exception $exception) {
            throw $e;
        }

        // log with trace if error is really bad
        if (in_array($severity, $this->severityReport) && isset($e->getTrace()[0]['line'])) {
            $logger->error(
                '['.$e->getCode().'] "'.$e->getMessage().'" on line '.$e->getTrace()[0]['line'].' of file '
                .($e->getTrace()[0]['file'] ?? '<unkown>')
            );
            foreach ($e->getTrace() as $trace) {
                if (array_key_exists('file', $trace) && array_key_exists('line', $trace)) {
                    $logger->error(
                        'File ['.$trace['file'].'] line ['.$trace['line'].'] in function {'.$trace['function']
                        .'}'
                    );
                }
            }

            return;
        }
        // Simply log error code and message by default
        $logger->error('['.$e->getCode().'] "'.$e->getMessage());
    }

    /**
     * Get exception context
     */
    protected function exceptionContext(Exception $exception): array
    {
        return [];
    }

    /**
     * Get context
     */
    protected function context(): array
    {
        try {
            return array_filter(
                [
                    'userUuid' => Auth::id(),
                ]
            );
        } catch (Throwable) {
            return [];
        }
    }

    /**
     * Exception $e should not report !
     */
    protected function shouldntReport(Exception $e): bool
    {
        $dontReport = array_merge($this->dontReport, $this->internalDontReport);

        return ! is_null(
            Arr::first(
                $dontReport,
                function ($type) use ($e) {
                    return $e instanceof $type;
                }
            )
        );
    }

    /**
     * Render an exception into an HTTP response.
     *
     * @param  Request  $request
     * @param  Exception  $exception
     *
     * @return SymfonyResponse
     */
    public function render($request, Exception $exception): SymfonyResponse
    {
        $exception = $this->prepareException($exception);

        if ($exception instanceof HttpResponseException) {
            return $exception->getResponse();
        } elseif ($exception instanceof ValidationException) {
            return $this->convertValidationExceptionToResponse($exception);
        } elseif ($exception instanceof AuthenticationException) {
            return $this->unauthenticated($request, $exception);
        }

        if (! $exception instanceof Exception) {
            $exception = new FatalThrowableError($exception);
        }

        if (! $this->isApiCall($request)) {
            return $this->prepareResponse($request, $exception);
        }

        return $this->getJsonResponse($exception);
    }

    /**
     * Prepare an internal exception to its http equivalent
     */
    protected function prepareException(Throwable $exception): Throwable
    {
        return match (true) {
            $exception instanceof ModelNotFoundException => new NotFoundHttpException(
                $exception->getMessage(),
                $exception
            ),
            $exception instanceof AuthorizationException => new AccessDeniedHttpException($exception->getMessage(), $exception),
            $exception instanceof TokenMismatchException => new HttpException(419, $exception->getMessage(), $exception),
            $exception instanceof SuspiciousOperationException => new NotFoundHttpException('Bad hostname provided.', $exception),
            default => $exception,
        };
    }

    protected function prepareResponse(Request $request, Exception $e): SymfonyResponse
    {
        if (! $this->isHttpException($e) && Config::get('app.debug')) {
            return $this->toIlluminateResponse($this->convertExceptionToResponse($e), $e);
        }

        if (! $this->isHttpException($e)) {
            $e = new HttpException(500, $e->getMessage());
        }

        /** @var HttpException $e */
        return $this->toIlluminateResponse(
            $this->renderHttpException($e),
            $e
        );
    }

    protected function toIlluminateResponse(SymfonyResponse $response, Exception $e): Response|RedirectResponse
    {
        if ($response instanceof SymfonyRedirectResponse) {
            $response = new RedirectResponse(
                $response->getTargetUrl(),
                $response->getStatusCode(),
                $response->headers->all()
            );
        } else {
            $response = new Response(
                $response->getContent(),
                $response->getStatusCode(),
                $response->headers->all()
            );
        }

        return $response->withException($e);
    }

    /**
     * Determines if request is an api call.
     *
     * If the request URI contains '/api/v'.
     *
     * @param  Request  $request
     *
     * @return bool
     */
    protected function isApiCall(Request $request): bool
    {
        return (php_sapi_name() != 'cli' && str_contains($request->getUri(), '/api/v') !== false)
            || $request->expectsJson();
    }

    /**
     * Creates a new JSON response.
     *
     * @param  Exception  $exception
     *
     * @return JsonResponse
     */
    protected function getJsonResponse(Exception $exception): JsonResponse
    {
        // General case
        $code = 400;

        if (method_exists($exception, 'getStatusCode')) {
            $code = $exception->getStatusCode();
        }

        /** @var ResponseFactory $responseFactory */
        $responseFactory = response();

        $headers = [];
        if ($this->isHttpException($exception)) {
            /** @var HttpException $exception */
            $headers = $exception->getHeaders();
        }

        return $responseFactory->json(
            $this->convertExceptionToArray($exception),
            $code,
            $headers,
            JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES
        );
    }

    protected function convertExceptionToArray(Exception $exception): array
    {
        return config('app.debug', false)
            ? [
                'message' => $exception->getMessage(),
                'exception' => get_class($exception),
                'file' => $exception->getFile(),
                'line' => $exception->getLine(),
                'trace' => collect($exception->getTrace())->map(
                    function ($trace) {
                        return Arr::except($trace, ['args']);
                    }
                )->all(),
            ]
            : [
                'message' => $this->isHttpException($exception) ? $exception->getMessage() : __('inside.server_error'),
            ];
    }

    /**
     * Render an unauthenicated
     */
    protected function unauthenticated(
        Request $request,
        AuthenticationException $exception
    ): JsonResponse|RedirectResponse {
        return $this->isApiCall($request) ?
            response()->json(['message' => $exception->getMessage()], 401) : redirect('/?redirect='.$request->path());
    }

    /**
     * Convert a validation exception into a JSON response.
     */
    protected function convertValidationExceptionToResponse(ValidationException $exception): SymfonyResponse
    {
        if (! is_null($exception->response)) {
            return $exception->response;
        }

        /** @var Request $request */
        $request = request();

        $messageKey = $exception->getMessage();
        if ($messageKey == 'The given data was invalid.') {
            $messageKey = 'validation.failed';
        }

        return $this->isApiCall($request) ? response()->json(
            [
                'message' => Lang::get($messageKey),
                'errors' => $exception->errors(),
            ],
            $exception->status
        ) : $this->invalid($request, $exception);
    }

    /**
     * Full invalid answer
     *
     * @param  Request  $request
     * @param  ValidationException  $exception
     * @return RedirectResponse
     */
    protected function invalid(Request $request, ValidationException $exception): RedirectResponse
    {
        return redirect($exception->redirectTo ?? $this->previousUrl($request))->withInput(
            Arr::except($request->input(), $this->dontFlash)
        )->withErrors($exception->errors(), $exception->errorBag);
    }

    /**
     *  Whoops !
     */
    protected function renderExceptionWithWhoops(Throwable $e): string
    {
        return tap(
            new Whoops(),
            function ($whoops) {
                $whoops->appendHandler($this->whoopsHandler());
                $whoops->writeToOutput(false);
                $whoops->allowQuit(false);
            }
        )->handleException($e);
    }

    protected function whoopsHandler(): \Whoops\Handler\Handler
    {
        try {
            return app(HandlerInterface::class);
        } catch (BindingResolutionException $e) {
            return (new WhoopsHandler())->forDebug();
        }
    }

    /**
     * render a simple http exception
     *
     * @param  HttpException  $e
     * @return SymfonyResponse
     */
    protected function renderHttpException(HttpException $e): SymfonyResponse
    {
        $this->registerErrorViewPaths();

        if (View::exists($view = $this->getHttpExceptionView($e))) {
            /** @var Response $response */
            $response = response(
                View::make(
                    $view,
                    [
                        'errors' => new ViewErrorBag(),
                        'exception' => $e,
                    ]
                )->render(),
                $e->getStatusCode(),
                $e->getHeaders()
            );

            return $response;
        }

        return $this->convertExceptionToResponse($e);
    }

    /**
     * Convert exception to response
     *
     * @param  Exception  $e
     * @return SymfonyResponse
     */
    protected function convertExceptionToResponse(Exception $e): SymfonyResponse
    {
        return SymfonyResponse::create(
            $this->renderExceptionContent($e),
            $this->isHttpException($e) ? $e->getStatusCode() : 500, // @phpstan-ignore-line
            $this->isHttpException($e) ? $e->getHeaders() : [] // @phpstan-ignore-line
        );
    }

    /**
     * Render exception content
     */
    protected function renderExceptionContent(Exception $e): string
    {
        try {
            return config('app.debug') && class_exists(Whoops::class) ? $this->renderExceptionWithWhoops($e)
                : $this->renderExceptionWithSymfony($e, config('app.debug'));
        } catch (Exception $e) {
            return $this->renderExceptionWithSymfony($e, config('app.debug'));
        }
    }

    /**
     * Render exception with symfony exception handler
     */
    protected function renderExceptionWithSymfony(Exception $e, bool $debug): string
    {
        $renderer = new HtmlErrorRenderer($debug);

        return $renderer->render($e)->getAsString();
    }

    /**
     * Register error view paths
     */
    protected function registerErrorViewPaths(): void
    {
        $paths = collect(config('view.paths'))->filter(
            function ($path) {
                return $path !== false;
            }
        );

        View::replaceNamespace(
            'errors',
            $paths->map(
                function ($path) {
                    return "{$path}/errors";
                }
            )->push(__DIR__.'/views')->all()
        );
    }

    /**
     * Get exception view
     */
    protected function getHttpExceptionView(HttpExceptionInterface $exception): string
    {
        return "errors::{$exception->getStatusCode()}";
    }

    /**
     * is Exception an http exception
     */
    protected function isHttpException(Exception $e): bool
    {
        return $e instanceof HttpExceptionInterface;
    }

    /**
     * render for console !
     *
     * @param  OutputInterface  $output
     * @param  Exception  $e
     */
    public function renderForConsole($output, Exception $e): void
    {
        if ($e instanceof CommandNotFoundException) {
            $message = str($e->getMessage())->explode('.')->first();

            if (! empty($alternatives = $e->getAlternatives())) {
                $message .= '. Did you mean one of these?';
                $output->write('<fg=white;bg=red;options=bold> ERROR </> ');
                $output->writeln($message);
                foreach ($alternatives as $alternative) {
                    $output->writeln('<fg=white>⇂ '.$alternative.'</>');
                }
            } else {
                $output->write('<fg=white;bg=red;options=bold> ERROR </> ');
                $output->writeln($message);
            }

            return;
        }
        (new ConsoleApplication())->renderThrowable($e, $output);
    }

    public function previousUrl(Request $request): string
    {
        $router = app(UrlGenerator::class);

        $referrer = $request->headers->get('referer');

        $url = $referrer ? $router->to($referrer) : '/';

        if ($url) {
            return $url;
        }

        return $router->to('/');
    }

    public function shouldReport(Exception $e): bool
    {
        return ! $this->shouldntReport($e);
    }
}
