<?php

declare(strict_types=1);

namespace Inside\Events;

use Closure;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Events\Dispatcher as BaseDispatcher;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use Inside\Support\Traits\ReflectsClosures;
use ReflectionException;

final class Dispatcher extends BaseDispatcher
{
    use ReflectsClosures;

    protected function queueHandler($class, $method, $arguments): void
    {
        [$listener, $job] = $this->createListenerAndJob(
            $class,
            $method,
            $arguments
        );

        $connection = $this->resolveQueue()->connection(
            $listener->connection ??
            null
        );

        $queue = method_exists($listener, 'viaQueue')
            ? $listener->viaQueue()
            : $listener->queue ?? null;

        isset($listener->delay)
            ? $connection->laterOn($queue, $listener->delay, $job)
            : $connection->pushOn($queue, $job);
    }

    /**
     * @param Closure|string|array $listener
     * @param bool $wildcard
     *
     * @return Closure
     */
    public function makeListener($listener, $wildcard = false): Closure
    {
        if (is_string($listener)) {
            return $this->createClassListener($listener, $wildcard);
        }

        if (
            is_array($listener) && isset($listener[0])
            && is_string($listener[0])
        ) {
            return $this->createClassListener($listener, $wildcard);
        }

        return function ($event, $payload) use ($listener, $wildcard) {
            /** @var Closure $listener */
            if ($wildcard) {
                return $listener($event, $payload);
            }

            return $listener(...array_values($payload));
        };
    }

    /**
     * @param  mixed  $listener
     * @param $wildcard
     * @return Closure
     */
    public function createClassListener(mixed $listener, $wildcard = false): Closure
    {
        return function ($event, $payload) use ($listener, $wildcard) {
            /** @var callable $callable */
            $callable = $this->createClassCallable($listener);

            if ($wildcard) {
                return call_user_func(
                    $callable,
                    $event,
                    $payload
                );
            }

            return $callable(...array_values($payload));
        };
    }

    /**
     * @param  string|string[]  $listener
     *
     * @return Closure|array
     * @throws BindingResolutionException
     */
    protected function createClassCallable($listener): Closure|array
    {
        [$class, $method] = is_array($listener) ? $listener
            : $this->parseClassCallable($listener);

        if (! method_exists($class, $method)) {
            $method = '__invoke';
        }

        if ($this->handlerShouldBeQueued($class)) {
            return $this->createQueuedHandlerCallable($class, $method);
        }

        $listener = $this->container->make($class);

        return $this->handlerShouldBeDispatchedAfterDatabaseTransactions($listener)
            ? $this->createCallbackForListenerRunningAfterCommits(
                $listener,
                $method
            ) : [$listener, $method];
    }

    /**
     * @param mixed $listener
     *
     * @return bool
     */
    protected function handlerShouldBeDispatchedAfterDatabaseTransactions($listener): bool
    {
        return ($listener->afterCommit ?? null)
            && $this->container->bound('db.transactions');
    }

    /**
     * @param mixed $listener
     * @param string $method
     *
     * @return Closure
     */
    protected function createCallbackForListenerRunningAfterCommits($listener, string $method): Closure
    {
        return function () use ($method, $listener) {
            $payload = func_get_args();

            $this->container->make('db.transactions')->addCallback(function () use ($listener, $method, $payload) {
                $listener->$method(...$payload);
            });
        };
    }

    /**
     * Register an event subscriber with the dispatcher.
     *
     * @param  object|string  $subscriber
     * @return void
     * @throws ReflectionException
     */
    public function subscribe($subscriber): void
    {
        $subscriber = $this->resolveSubscriber($subscriber);

        $events = $subscriber->subscribe($this);

        if (is_array($events)) {
            foreach ($events as $event => $listeners) {
                foreach (Arr::wrap($listeners) as $listener) {
                    if (is_string($listener) && method_exists($subscriber, $listener)) {
                        $this->listen($event, [get_class($subscriber), $listener]);

                        continue;
                    }

                    $this->listen($event, $listener);
                }
            }
        }
    }

    /**
     * @param  QueuedClosure|Closure|string|array  $events
     * @param  QueuedClosure|Closure|string|array|null  $listener
     * @return void
     * @throws ReflectionException
     */
    public function listen(mixed $events, mixed $listener = null)
    {
        if ($events instanceof Closure) {
            collect($this->firstClosureParameterTypes($events))
                ->each(function ($event) use ($events) {
                    $this->listen($event, $events);
                });

            return;
        } elseif ($events instanceof QueuedClosure) {
            collect($this->firstClosureParameterTypes($events->closure))
                ->each(function ($event) use ($events) {
                    $this->listen($event, $events->resolve());
                });

            return;
        } elseif ($listener instanceof QueuedClosure) {
            $listener = $listener->resolve();
        }

        foreach ((array) $events as $event) {
            if (Str::contains($event, '*')) {
                // @phpstan-ignore-next-line
                $this->setupWildcardListen($event, $listener);
            } else {
                $this->listeners[$event][] = $listener;
            }
        }
    }

    /**
     * Setup a wildcard listener callback.
     *
     * @param  string  $event
     * @param  Closure|string  $listener
     * @return void
     */
    protected function setupWildcardListen($event, $listener): void
    {
        $this->wildcards[$event][] = $listener;

        $this->wildcardsCache = [];
    }

    /**
     * Fire an event and call the listeners.
     *
     * @param  string|object  $event
     * @param  mixed  $payload
     * @param  bool  $halt
     * @return array|null
     */
    public function dispatch($event, $payload = [], $halt = false)
    {
        // When the given "event" is actually an object we will assume it is an event
        // object and use the class as the event name and this event itself as the
        // payload to the handler, which makes object based events quite simple.
        [$event, $payload] = $this->parseEventAndPayload(
            $event,
            $payload
        );

        if ($this->shouldBroadcast($payload)) {
            $this->broadcastEvent($payload[0]);
        }

        $responses = [];

        foreach ($this->getListeners($event) as $listener) {
            $response = $listener($event, $payload);

            // If a response is returned from the listener and event halting is enabled
            // we will just return this response, and not call the rest of the event
            // listeners. Otherwise we will add the response on the response list.
            if ($halt && ! is_null($response)) {
                return $response;
            }

            // If a boolean false is returned from a listener, we will stop propagating
            // the event to any further listeners down in the chain, else we keep on
            // looping through the listeners and firing every one in our sequence.
            if ($response === false) {
                break;
            }

            $responses[] = $response;
        }

        return $halt ? null : $responses;
    }

    public function getListeners($eventName): array
    {
        $listeners = array_merge(
            $this->prepareListeners($eventName),
            $this->wildcardsCache[$eventName] ?? $this->getWildcardListeners($eventName)
        );

        return class_exists($eventName, false)
            ? $this->addInterfaceListeners($eventName, $listeners)
            : $listeners;
    }

    /**
     * Prepare the listeners for a given event.
     *
     * @param  string  $eventName
     * @return Closure[]
     */
    protected function prepareListeners(string $eventName): array
    {
        $listeners = [];

        foreach ($this->listeners[$eventName] ?? [] as $listener) {
            $listeners[] = $this->makeListener($listener);
        }

        return $listeners;
    }
}
