<?php

namespace Inside\Database\Query;

use Closure;
use Illuminate\Database\Eloquent\Builder as BaseEloquentBuilder;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Database\Query\Builder as BaseQueryBuilder;
use Illuminate\Database\Query\Expression;
use Illuminate\Support\LazyCollection;
use Inside\Database\Eloquent\Builder as EloquentBuilder;
use InvalidArgumentException;

class Builder extends BaseQueryBuilder
{
    public function addSelect($column)
    {
        $columns = is_array($column) ? $column : func_get_args();

        foreach ($columns as $as => $column) {
            if (is_string($as) && $this->isQueryable($column)) {
                if (empty($this->columns)) {
                    $this->select($this->from.'.*');
                }
                $this->selectSub($column, $as);
            } else {
                $this->columns[] = $column;
            }
        }

        return $this;
    }

    /**
     * @param mixed $value
     * @return bool
     */
    protected function isQueryable($value): bool
    {
        return $value instanceof self
            || $value instanceof BaseEloquentBuilder
            || $value instanceof Relation
            || $value instanceof Closure;
    }

    /**
     * @param mixed $query
     * @return array
     */
    protected function parseSub($query): array
    {
        if (
            $query instanceof self
            || $query instanceof BaseEloquentBuilder
            || $query instanceof Relation
        ) {
            return [$query->toSql(), $query->getBindings()];
        } elseif (is_string($query)) {
            return [$query, []];
        } else {
            throw new InvalidArgumentException(
                'A subquery must be a query builder instance, a Closure, or a string.'
            );
        }
    }

    public function orderBy($column, $direction = 'asc'): self
    {
        if ($this->isQueryable($column)) {
            [$query, $bindings] = $this->createSub($column);

            $column = new Expression('('.$query.')');

            $this->addBinding($bindings, $this->unions ? 'unionOrder' : 'order');
        }

        $direction = strtolower($direction);

        if (! in_array($direction, ['asc', 'desc'], true)) {
            throw new InvalidArgumentException('Order direction must be "asc" or "desc".');
        }

        $this->{$this->unions ? 'unionOrders' : 'orders'}[] = [
            'column' => $column,
            'direction' => $direction,
        ];

        return $this;
    }
}
