<?php

namespace Inside\Database\Eloquent;

use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use Inside\Database\Query\Builder as QueryBuilder;

/**
 * Trait WithEnhancedBuilder
 */
trait WithEnhancedBuilder
{
    /**
     * WithEnhancedBuilder constructor.
     */
    public function __construct(array $attributes = [])
    {
        $this->bootIfNotBooted();

        $this->initializeTraits();

        $this->syncOriginal();

        $this->fill($attributes);
    }

    public function getTable(): string
    {
        return $this->table;
    }

    /**
     * @return mixed
     */
    public static function query()
    {
        return (new static())->newQuery();
    }

    /**
     * @param $query
     * @return Builder
     */
    public function newEloquentBuilder($query): Builder
    {
        return new Builder($query);
    }

    /**
     * @return QueryBuilder
     */
    protected function newBaseQueryBuilder(): QueryBuilder
    {
        $connection = $this->getConnection();

        return new QueryBuilder(
            $connection,
            $connection->getQueryGrammar(),
            $connection->getPostProcessor()
        );
    }

    public function mergeCasts(array $casts): void
    {
        $this->casts = array_merge($this->casts, $casts);
    }

    /**
     * @param array $attributes
     * @param false $exists
     * @return static
     */
    public function newInstance($attributes = [], $exists = false)
    {
        $model = new static((array) $attributes);

        $model->exists = $exists;

        $model->setConnection(
            $this->getConnectionName()
        );

        $model->setTable($this->getTable());

        $model->mergeCasts($this->casts);

        return $model;
    }

    /**
     * boot if not booted !
     */
    protected function bootIfNotBooted(): void
    {
        if (! isset(static::$booted[static::class])) {
            static::$booted[static::class] = true;

            $this->fireModelEvent('booting', false);

            static::booting();
            static::boot();
            static::booted();

            $this->fireModelEvent('booted', false);
        }
    }

    /**
     * booting ( to be override )
     */
    protected static function booting(): void
    {
        //
    }

    /**
     * Booting model traits
     */
    protected static function bootTraits(): void
    {
        $class = static::class;

        $booted = [];

        static::$traitInitializers[$class] = [];

        foreach (class_uses_recursive($class) as $trait) {
            $method = 'boot'.class_basename($trait);

            if (method_exists($class, $method) && ! in_array($method, $booted)) {
                forward_static_call([$class, $method]);

                $booted[] = $method;
            }

            if (method_exists($class, $method = 'initialize'.class_basename($trait))) {
                static::$traitInitializers[$class][] = $method;

                static::$traitInitializers[$class] = array_unique(
                    static::$traitInitializers[$class]
                );
            }
        }
    }

    /**
     * Initializing traits
     */
    protected function initializeTraits(): void
    {
        foreach (static::$traitInitializers[static::class] as $method) {
            $this->{$method}();
        }
    }

    /**
     * booted
     */
    protected static function booted(): void
    {
        //
    }

    public function getRawOriginal(?string $key = null, mixed $default = null): mixed
    {
        return Arr::get($this->original, $key, $default);
    }
}
