<?php

namespace Inside\Console\Commands;

use Illuminate\Contracts\Events\Dispatcher;
use Illuminate\Database\ConnectionResolverInterface;
use Illuminate\Filesystem\Filesystem;
use Illuminate\Support\HigherOrderTapProxy;
use Inside\Console\Command;
use Inside\Database\MysqlConnection;
use Inside\Database\Schema\SchemaState;
use Inside\Events\SchemaDumped;

class SchemaDumpCommand extends Command
{
    /**
     * Signature
     *
     * @var string
     */
    protected $signature = 'inside:schema:dump
                {--database= : Connection database utilisée}
                {--path= : Le chemin ou sauvegardé le dump}
                {--prune : Suppression des migrations existantes}';

    protected $description = 'Dump des schema de base de données';

    public function handle(ConnectionResolverInterface $connections, Dispatcher $dispatcher): void
    {
        /** @var string|null $database */
        $database = $this->input->getOption('database');

        $prune = (bool) $this->option('prune');

        /** @var MysqlConnection $connection */
        $connection = $connections->connection($database);

        $path = $this->path($connection);
        $this->schemaState($connection)->dump($path);

        $dispatcher->dispatch(new SchemaDumped($connection, $path));

        $this->info('Schema sauvegardé ['.$path.']');

        if ($prune) {
            (new Filesystem())->deleteDirectory(
                database_path('migrations'),
                $preserve = false
            );

            $this->info('Migrations pruned successfully.');
        }
    }

    /**
     * Get schema state
     *
     * @param MysqlConnection $connection
     * @return SchemaState
     */
    protected function schemaState(MysqlConnection $connection): SchemaState
    {
        return $connection->getSchemaState()->handleOutputUsing(
            function ($type, $buffer) {
                $this->output->write($buffer);
            }
        );
    }

    /**
     * Get path
     *
     * @param MysqlConnection $connection
     * @return HigherOrderTapProxy|mixed
     */
    protected function path(MysqlConnection $connection)
    {
        /** @var string|null $path */
        $path = $this->option('path');

        return tap(
            $path ?: database_path('schema/'.$connection->getName().'-schema.sql'),
            function ($path) {
                ensure_directory_exists(dirname($path));
            }
        );
    }
}
