<?php

namespace Inside\Console\Commands;

use Carbon\Carbon;
use Closure;
use Cron\CronExpression;
use DateTimeZone;
use Illuminate\Console\Application;
use Illuminate\Console\Scheduling\CallbackEvent;
use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Support\Collection;
use Inside\Console\Command;
use ReflectionClass;
use ReflectionFunction;
use Symfony\Component\Console\Terminal;

class ScheduleListCommand extends Command
{
    protected $signature = 'schedule:list {--timezone= : The timezone that times should be displayed in}';

    protected $description = 'Liste des tâches';

    protected static ?Closure $terminalWidthResolver = null;

    public function handle(Schedule $schedule): void
    {
        $events = collect($schedule->events());

        if ($events->isEmpty()) {
            $this->output->warning('Aucune tâche définie.');

            return;
        }

        $terminalWidth = self::getTerminalWidth();

        $expressionSpacing = $this->getCronExpressionSpacing($events);

        $timezone = new DateTimeZone($this->option('timezone') ?? config('app.timezone'));

        $events = $events->map(function ($event) use ($terminalWidth, $expressionSpacing, $timezone) {
            $expression = $this->formatCronExpression($event->expression, $expressionSpacing);

            $command = $event->command;
            $description = $event->description;

            if (! $this->output->isVerbose()) {
                $command = str_replace([Application::phpBinary(), Application::artisanBinary()], [
                    'php',
                    preg_replace("#['\"]#", '', Application::artisanBinary()),
                ], $event->command);
            }

            if ($event instanceof CallbackEvent) {
                if (class_exists($event->description)) {
                    $command = $event->description;
                    $description = '';
                } else {
                    $command = 'Closure at: '.$this->getClosureLocation($event);
                }
            }

            $command = mb_strlen($command) > 1 ? "{$command} " : '';

            $nextDueDateLabel = 'Next Due:';

            $nextDueDate = Carbon::create((CronExpression::factory($event->expression))
                ->getNextRunDate(Carbon::now()->setTimezone($event->timezone))
                ->setTimezone($timezone)
            );
            if ($nextDueDate === false) {
                return;
            }

            $nextDueDate = $this->output->isVerbose()
                ? $nextDueDate->format('Y-m-d H:i:s P')
                : $nextDueDate->diffForHumans();

            $hasMutex = $event->mutex->exists($event) ? 'Has Mutex › ' : '';

            $dots = str_repeat('.', max(
                $terminalWidth - mb_strlen($expression.$command.$nextDueDateLabel.$nextDueDate.$hasMutex) - 8, 0
            ));

            // Highlight the parameters...
            $command = preg_replace("#(php artisan [\w\-:]+) (.+)#", '$1 <fg=yellow;options=bold>$2</>', $command);

            return [sprintf(
                '  <fg=yellow>%s</>  %s<fg=cyan>%s %s%s %s</>',
                $expression,
                $command,
                $dots,
                $hasMutex,
                $nextDueDateLabel,
                $nextDueDate
            ), $this->output->isVerbose() && mb_strlen($description) > 1 ? sprintf(
                '  <fg=cyan>%s%s %s</>',
                str_repeat(' ', mb_strlen($expression) + 2),
                '⇁',
                $description
            ) : ''];
        });

        $this->lines(
            $events->flatten()->filter()->prepend('')->push('')->toArray()
        );
    }

    /**
     * @param  array<int, int>  $spacing
     */
    private function formatCronExpression(string $expression, array $spacing): string
    {
        $expressions = explode(' ', $expression);

        return collect($spacing)
            ->map(fn ($length, $index) => str_pad($expressions[$index], $length))
            ->implode(' ');
    }

    private function getClosureLocation(CallbackEvent $event): string
    {
        $callback = tap((new ReflectionClass($event))
            ->getProperty('callback'))
            ->setAccessible(true)
            ->getValue($event);

        if ($callback instanceof Closure) {
            $function = new ReflectionFunction($callback);

            return sprintf(
                '%s:%s',
                str_replace($this->laravel->basePath().DIRECTORY_SEPARATOR, '', $function->getFileName() ?: ''),
                $function->getStartLine()
            );
        }

        if (is_array($callback)) {
            return sprintf('%s::%s', $callback[0]::class, $callback[1]);
        }

        return sprintf('%s::__invoke', $callback::class);
    }

    /**
     * @return array<int, int>
     */
    private function getCronExpressionSpacing(Collection $events): array
    {
        $rows = $events->map(fn ($event) => array_map('mb_strlen', explode(' ', $event->expression)));

        return collect($rows[0] ?? [])->keys()->map(fn ($key) => $rows->max($key))->toArray();
    }

    public static function getTerminalWidth(): int
    {
        return is_null(static::$terminalWidthResolver)
            ? (new Terminal())->getWidth()
            : call_user_func(static::$terminalWidthResolver);
    }

    /**
     * Set a callback that should be used when resolving the terminal width.
     */
    public static function resolveTerminalWidthUsing(?Closure $resolver): void
    {
        static::$terminalWidthResolver = $resolver;
    }
}
