<?php

namespace Inside\Console\Commands;

use Illuminate\Support\Facades\Route;
use Inside\Console\Command;
use Inside\Facades\RouteFormatter;
use Symfony\Component\Console\Input\InputOption;

class RouteListCommand extends Command
{
    protected $name = 'inside:route:list';

    protected $description = 'Liste de toutes les routes enregistrées';

    public function handle(): void
    {
        if (empty(Route::getRoutes())) {
            $this->error('Aucune route enregistrée');

            return;
        }

        if (empty($routes = RouteFormatter::setCompact($this->option('compact'))
            ->setReverse($this->option('reverse'))
            ->setName($this->option('name'))
            ->setPath($this->option('path'))
            ->setMethod($this->option('method'))
            ->setSort($this->option('sort'))
            ->setColumns($this->option('columns'))
            ->setExceptPath($this->option('except-path'))
            ->getRoutes())) {
            $this->error('Aucune route trouvée!');

            return;
        }

        $this->displayRoutes($routes);
    }

    /**
     * @param  array  $routes
     */
    protected function displayRoutes(array $routes): void
    {
        if ($this->option('json')) {
            $this->line(collect($routes)
                ->map(function ($route) {
                    $route['middleware'] = empty($route['middleware']) ?
                        [] :
                        explode(',', $route['middleware']);

                    return $route;
                })
                ->values()
                ->toJson());

            return;
        }

        $this->table(RouteFormatter::getHeaders(), $routes);
    }

    /**
     * @return array[]
     */
    protected function getOptions(): array
    {
        return [
            [
                'columns',
                null,
                InputOption::VALUE_OPTIONAL | InputOption::VALUE_IS_ARRAY,
                'Les colonnes utilisées pour créer le tableau du listing',
            ],
            ['compact', 'c', InputOption::VALUE_NONE, 'Version compact : juste la méthode, l\'URI et l\'action'],
            ['json', null, InputOption::VALUE_NONE, 'Version json'],
            ['method', null, InputOption::VALUE_OPTIONAL, 'Filtrer par méthode'],
            ['name', null, InputOption::VALUE_OPTIONAL, 'Filtrer par nom'],
            ['path', null, InputOption::VALUE_OPTIONAL, 'Filtrer par uri'],
            [
                'exclude-path',
                null,
                InputOption::VALUE_OPTIONAL,
                'Ne pas afficher les routes qui correspondent au pattern',
            ],
            ['reverse', 'r', InputOption::VALUE_NONE, 'Tri inverse'],
            [
                'sort',
                null,
                InputOption::VALUE_OPTIONAL,
                'Colonne utilisée pour le tri ( pour garder l\'ordre de chargement des routes mettre precedence )',
                'uri',
            ],
        ];
    }
}
