<?php

namespace Inside\Console\Commands;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Inside\Console\Command;
use Symfony\Component\Yaml\Yaml;

class Pm2ConfigCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'pm2:config {--directory=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Création d\'un fichier process.yml pour pm2';

    /**
     * Where to start artisan
     */
    protected ?string $directory = null;

    /**
     * Execute the console command.
     *
     * @return void
     */
    public function handle(): void
    {
        $config = [
            'apps' => [],
        ];

        /** @var string|null $directory */
        $directory = $this->option('directory');
        $this->directory = $directory;

        $prefix = config('app.code');
        if (! $prefix) {
            $this->error('Le code d\'application n\'est pas configuré dans le fichier .env');

            return;
        }
        $prefix = env('PM2_PREFIX', $prefix);
        $prefix = Str::lower($prefix);

        // Add worker for indexation
        if (config('scout.queue') !== null) {
            $config['apps'][] = $this->addWorker(
                $prefix.'_indexation',
                config('scout.queue.queue', env('APP_CODE', 'sid2').'_indexation')
            );
        }

        // worker to handle image style generation
        $config['apps'][] = $this->addWorker(
            $prefix.'_images',
            get_images_queue_name()
        );

        // Default workers
        // Get default queue name
        if (config('queue.default', 'sync') != 'sync') {
            $queue = config(
                'queue.connections.'.config('queue.default', 'sync').'.queue',
                env('APP_CODE', 'sid2').'_default'
            );
            $workers = config('app.workers');

            if (Str::endsWith($queue, '_default')) {
                $queues = [$queue];
                $config['apps'][] = $this->addWorker(
                    $prefix.'_main_worker',
                    implode(',', $queues),
                    $workers
                );
            } else {
                $queues = [$queue];
                $config['apps'][] = $this->addWorker(
                    $prefix.'_main_worker',
                    implode(',', $queues),
                    $workers
                );
            }

            // Adding customs workers
            foreach (config('workers', []) as $worker => $options) {
                $config['apps'][] = [
                    'script' => $options['script'],
                    'name' => $prefix.'_'.$worker,
                    'cwd' => $this->directory ?? cms_base_path(),
                    'args' => $options['args'],
                    'interpreter' => $options['interpreter'],
                    'instances' => $options['instances'] ?? 1,
                    'watch' => false,
                ];
            }
        }

        File::put(cms_base_path('process.yml'), Yaml::dump($config));

        $this->info('process.yml correctement généré. Il suffit désormais de lancer pm2 start process.yml.');
    }

    protected function addWorker(string $name, string $queue, int $workers = 1): array
    {
        return [
            'script' => 'vendor/maecia/inside/artisan',
            'name' => $name,
            'cwd' => $this->directory ?? cms_base_path(),
            'args' => 'queue:work --queue='.$queue.' --tries=3 --timeout=300 --memory=512',
            'interpreter' => PHP_BINARY,
            'instances' => $workers,
            'watch' => false,
        ];
    }
}
