<?php

declare(strict_types=1);

namespace Inside\Console\Commands;

use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Inside\Console\Command;
use Inside\Content\Contracts\SchemaService;

final class MigrateWysiwygImages extends Command
{
    protected $name = 'inside:migrate:images';

    protected $signature = 'inside:migrate:images {from}';

    protected $description = 'Migration des images du wysiwyg d\'une instance à l\'instance courante';

    public function handle(): int
    {
        $oldUrl = $this->argument('from');
        if (! is_string($oldUrl) || ! filter_var($oldUrl, FILTER_VALIDATE_URL)) {
            $this->error('from is not valid');

            return Command::FAILURE;
        }
        $newUrl = config('app.url');
        if (Str::endsWith('/', $oldUrl)) {
            $oldUrl = substr($oldUrl, 0, -1);
        }
        if (Str::endsWith('/', $newUrl)) {
            $newUrl = substr($newUrl, 0, -1);
        }
        $this->info('Correction des images de ['.$oldUrl.'] vers ['.$newUrl.']');

        // Fix inside settings
        $this->write('Correction des <fg=cyan>settings</> si nécessaire');
        $this->writeResult(
            $this->runReplaceSql('inside_settings', 'value', $oldUrl, $newUrl)
        );
        $this->separatorLine();

        $schema = App::make(SchemaService::class);
        foreach ($schema->getAllFieldsListingOfType('wysiwyg') as $type => $fieldNames) {
            $this->writeln('Correction du type <fg=magenta>'.$type.'</>');
            foreach ($fieldNames as $fieldName) {
                $table = $schema->isContentType($type) ? type_to_table($type) : section_type_to_table($type);
                if (! is_null($table)) {
                    $this->write('Correction du champ <fg=cyan>'.$fieldName.'</> si nécessaire');
                    $this->writeResult(
                        $this->runReplaceSql(
                            $table,
                            $fieldName,
                            $oldUrl,
                            $newUrl
                        )
                    );
                }
            }
            $this->separatorLine();
        }

        return Command::SUCCESS;
    }

    private function runReplaceSql(string $table, string $column, string $oldUrl, string $newUrl): bool
    {
        return DB::statement(
            'UPDATE '.$table.' SET '.$column.' = REPLACE('.$column.', ?, ?);',
            [
                $oldUrl,
                $newUrl,
            ]);
    }
}
