<?php

namespace Inside\Console\Commands;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\File;
use Inside\Console\Command;
use stdClass;

class LaravelEchoServerConfigCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inside:laravel-echo-server:config {--port=6001} {--proxy-pass} {--cert-path=} {--cert-chain-path=} {--key-path=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Création d\'un fichier laravel-echo-server.json ';

    public function handle(): void
    {
        $port = (int) $this->option('port');

        /** @var string $certPath */
        $certPath = $this->option('cert-path') ?? '';

        /** @var string $certChainPath */
        $certChainPath = $this->option('cert-chain-path') ?? '';

        /** @var string $keyPath */
        $keyPath = $this->option('key-path') ?? '';

        $proxyPass = (bool) $this->option('proxy-pass');

        /** @var string $allowOrigin */
        $allowOrigin = $this->option('allow-origin') ?? ($proxyPass ? '127.0.0.1' : '*');

        File::put(
            'laravel-echo-server.json',
            $this->prepareLaravelEchoServerConfig(
                $port,
                $certPath,
                $certChainPath,
                $keyPath,
                $allowOrigin,
                $proxyPass
            )->toJson(
                JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT
            )
        );

        if ($proxyPass) {
            $proxyPassConfig = <<<CONFIG
RewriteEngine On
RewriteCond %{REQUEST_URI}  ^/socket.io            [NC]
RewriteCond %{QUERY_STRING} transport=websocket    [NC]
RewriteRule /(.*)           ws://localhost:$port/$1 [P,L]

ProxyPass        /socket.io http://localhost:$port/socket.io
ProxyPassReverse /socket.io http://localhost:$port/socket.io
CONFIG;
            $this->warn('Configuration à ajouter à la config apache:');
            $this->writeln('<fg=magenta>'.str_repeat('-', 80).'</fg=magenta>');
            $this->comment($proxyPassConfig);
            $this->writeln('<fg=magenta>'.str_repeat('-', 80).'</fg=magenta>');
        }

        $this->info(
            'laravel-echo-server.json correctement généré. Il suffit désormais de lancer laravel-echo-server start.'
        );
    }

    /**
     * @param  int  $port
     * @param  string  $certPath
     * @param  string  $certChainPath
     * @param  string  $keyPath
     * @param  string  $allowOrigin
     * @param  bool  $proxyPass
     * @return Collection
     */
    protected function prepareLaravelEchoServerConfig(
        int $port,
        string $certPath,
        string $certChainPath,
        string $keyPath,
        string $allowOrigin,
        bool $proxyPass
    ): Collection {
        $isSecured = preg_match('#^https://#', config('app.url'));

        if (! empty($certPath) && ! File::exists($certPath)) {
            $certPath = '';
        }
        if (! empty($certPath) && ! File::exists($certChainPath)) {
            $certChainPath = '';
        }
        if (! empty($certPath) && ! File::exists($keyPath)) {
            $keyPath = '';
        }

        return collect(
            [
                'authHost' => config('app.url'),
                'authEndpoint' => '/api/v1/broadcasting/auth',
                'clients' => [],
                'database' => 'redis',
                'databaseConfig' => [
                    'redis' => [
                        'keyPrefix' => config('database.redis.options.prefix'),
                    ],
                ],
                'devMode' => config('app.debug', false),
                'host' => null,
                'port' => $port,
                'protocole' => (! $proxyPass && $isSecured) ? 'https' : 'http',
                'socketio' => new stdClass(),
                'secureOptions' => 67108864, // SSL_OP_NO_TLSv1
                'sslCertPath' => $certPath,
                'sslKeyPath' => $keyPath,
                'sslCertChainPath' => $certChainPath,
                'sslPassphrase' => '',
                'subscribers' => [
                    'http' => false,
                    'redis' => true,
                ],
                'apiOriginAllow' => [
                    'allowCors' => true,
                    'allowOrigin' => $allowOrigin,
                    'allowMethods' => 'GET, POST, PUT, DELETE',
                    'allowHeaders' => 'Origin, X-Requested-With, X-Auth-Token, X-Socket-Id, Content-Type, Accept, Authorization, Access-Control-Allow-Headers, Access-Control-Allow-Methods, Access-Control-Allow-Origin, api-token',
                ],
            ]
        );
    }
}
