<?php

namespace Inside\Console\Commands;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Carbon;
use Illuminate\Support\Composer;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\Broadcast;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Queue;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Redis;
use Illuminate\Support\Facades\Request;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\View;
use Illuminate\Support\InteractsWithTime;
use Illuminate\Support\Str;
use Inside\Console\Command;
use Inside\Facades\Inside;
use Inside\Facades\Package;
use Inside\Services\AliasLoader;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Process\Process;
use Throwable;
use Tymon\JWTAuth\Console\JWTGenerateSecretCommand;
use Tymon\JWTAuth\Providers\LumenServiceProvider;

abstract class InstallationTools extends Command
{
    use InteractsWithTime;

    public const LINUX_SYSTEM_INSTALLATION = 'linux';

    public const WINDOWS_SYSTEM_INSTALLATION = 'windows';

    public const ANY_SYSTEM_INSTALLATION = 'any';

    protected const REQUIRED_STORAGE_DIRECTORIES = [
        'app',
        'app/menus',
        'app/public',
        'app/purifier',
        'app/sync',
        'app/tmp',
        'app/config',
        'framework/views',
        'framework/cache',
        'framework/cache/data',
        'logs',
        'tmp',
    ];

    protected const REQUIRED_DRUPAL_DIRECTORIES = [
        'sites/default/files/config/sync',
        'sites/default/files/public',
        'sites/default/files/private',
        'sites/default/files/translations',
        'sites/default/files/sync',
    ];

    protected const TO_BE_REMOVED = [
        '.htaccess',
        '.editorconfig',
    ];

    protected const FILE_TO_COPY = [
        '.env' => [
            'overwrite' => false,
            'path' => 'vendor/maecia/inside/.env.example',
        ],
        'front/conf/conf.env.js' => [
            'overwrite' => false,
            'path' => 'vendor/maecia/inside/core/app/Console/Commands/stubs/conf.env.js.stub',
        ],
        'sites/default/settings.php' => [
            'overwrite' => false,
            'path' => 'vendor/maecia/inside-drupal/example.settings.php',
        ],
        'sites/default/services.yml' => [
            'overwrite' => false,
            'path' => 'vendor/maecia/inside-drupal/example.services.yml',
        ],
        '.htaccess' => [
            'overwrite' => false,
            'path' => 'vendor/maecia/inside-drupal/.htaccess',
        ],
        'index.php' => [
            'overwrite' => true,
            'path' => 'vendor/maecia/inside-drupal/web/index.php',
        ],
        'autoload.php' => [
            'overwrite' => true,
            'path' => 'vendor/maecia/inside-drupal/web/autoload.php',
        ],
        'opcache_reset.php' => [
            'overwrite' => true,
            'path' => 'vendor/maecia/inside/opcache_reset.php',
        ],
        'update.php' => [
            'overwrite' => true,
            'path' => 'vendor/maecia/inside-drupal/web/update.php',
        ],
        'robots.txt' => [
            'overwrite' => true,
            'path' => 'vendor/maecia/inside/robots.example.txt',
        ],
        'robots-https.txt' => [
            'overwrite' => true,
            'path' => 'vendor/maecia/inside/robots.example.txt',
        ],
        'public/index.php' => [
            'overwrite' => true,
            'path' => 'vendor/maecia/inside-drupal/web/public/index.php',
        ],
        'public/opcache_reset.php' => [
            'overwrite' => true,
            'path' => 'vendor/maecia/inside/opcache_reset.php',
        ],
        'public/.htaccess' => [
            'overwrite' => false,
            'path' => 'vendor/maecia/inside-drupal/.htaccess',
        ],
        'public/robots.txt' => [
            'overwrite' => true,
            'path' => 'vendor/maecia/inside/robots.example.txt',
        ],
        'public/robots-https.txt' => [
            'overwrite' => true,
            'path' => 'vendor/maecia/inside/robots.example.txt',
        ],
        '.editorconfig' => [
            'overwrite' => true,
            'path' => 'vendor/maecia/inside-drupal/.editorconfig',
        ],
    ];

    protected const SYMBOLIC_LINKS = [
        'vendor/maecia/inside/artisan' => 'artisan',
        'vendor' => 'vendor/maecia/inside/vendor',
    ];

    protected const PATCHES = [
        [
            'cwd' => 'vendor/maecia-fork/lumen-framework',
            'patch' => 'vendor/maecia/inside-drupal/patches/lumen-framework.patch',
            'level' => 0,
        ],
        [
            'cwd' => '',
            'patch' => 'vendor/maecia/inside-drupal/patches/path-processor.patch',
            'level' => 0,
        ],
        [
            'cwd' => '',
            'patch' => 'vendor/maecia/inside-drupal/patches/file-transliteration.patch',
            'level' => 1,
        ],
        [
            'cwd' => 'vendor/vaites/php-apache-tika',
            'patch' => 'vendor/maecia/inside-drupal/patches/php-apache-tika.patch',
            'level' => 1,
        ],
    ];

    protected const FILES_TO_CLEANUP = [
        // path => system where the file SHOULD BE REMOVED
        'readme.md' => self::ANY_SYSTEM_INSTALLATION,
        'web.config' => self::LINUX_SYSTEM_INSTALLATION,
        '.htaccess' => self::WINDOWS_SYSTEM_INSTALLATION,
    ];

    protected Composer $composer;

    /**
     * InstallationTools constructor.
     *
     * @param Composer $composer
     */
    public function __construct(Composer $composer)
    {
        parent::__construct();
        $this->logEnabled = $this->insideIsInstalled();
        $this->composer = $composer;
    }

    protected function checkStorage(): void
    {
        $this->line('Vérification des dossiers storage');
        $done = false;

        // Check for storage path
        if (File::exists(inside_base_path('storage'))) {
            // Create new directory storage
            if (! File::exists(storage_path('framework/views'))) {
                File::makeDirectory(storage_path('framework/views'), 0770, true, true);
            }
            if (! File::exists(storage_path('framework/cache'))) {
                File::makeDirectory(storage_path('framework/cache'), 0770, true, true);
            }
            if (! File::exists(storage_path('framework/cache/data'))) {
                File::makeDirectory(storage_path('framework/cache/data'), 0770, true, true);
            }
            // We detected an old storage system, migrate!
            File::deleteDirectory(inside_base_path('storage'), false);
        }

        foreach (self::REQUIRED_STORAGE_DIRECTORIES as $path) {
            if (! $this->laravel->make('files')->exists(storage_path($path))) {
                $this->info('Création du dossier ['.storage_path($path).']');
                $this->laravel->make('files')->makeDirectory(storage_path($path), 0775, true, true);
                $done = true;
            }
        }
        if ($this->laravel->make('files')->exists(inside_base_path('storage'))) {
            $this->laravel->make('files')->deleteDirectory(inside_base_path('storage'), false);
            $this->info('Suppression du dossier ['.inside_base_path('storage').']');
            $done = true;
        }
        if (! $done) {
            $this->info('Tout est OK');
        }
        $this->line(str_repeat('-', 80));
    }

    protected function checkDrupal(): void
    {
        $this->line('Vérification des dossiers drupal');
        $done = false;

        foreach (self::REQUIRED_DRUPAL_DIRECTORIES as $path) {
            if (! $this->laravel->make('files')->exists(cms_base_path($path))) {
                $this->info('Création du dossier ['.cms_base_path($path).']');
                $this->laravel->make('files')->makeDirectory(cms_base_path($path), 0775, true, true);

                $done = true;
            }
        }
        if (! $done) {
            $this->info('Tout est OK');
        }
        $this->line(str_repeat('-', 80));
    }

    /**
     * Uniquement executer sur une installation
     */
    protected function removeUnnecessaryFiles(): void
    {
        $this->line('Suppression des fichiers non nécessaires');
        $done = false;
        foreach (self::TO_BE_REMOVED as $path) {
            if ($this->laravel->make('files')->exists(cms_base_path($path))) {
                $this->info(
                    'Suppression du fichier ['.cms_base_path($path).']'
                );
                $this->laravel->make('files')->delete(cms_base_path($path));
                $done = true;
            }
        }
        if (! $done) {
            $this->info('Tout est OK');
        }
        $this->line(str_repeat('-', 80));
    }

    protected function cleanupBasepath(): void
    {
        $this->line('Suppression des fichiers');
        $status = true;
        foreach (self::FILES_TO_CLEANUP as $path => $system) {
            if (
                ($system === self::WINDOWS_SYSTEM_INSTALLATION
                    && $this->getSystemInstallation() === self::LINUX_SYSTEM_INSTALLATION)
                || ($system === self::LINUX_SYSTEM_INSTALLATION
                    && $this->getSystemInstallation() === self::WINDOWS_SYSTEM_INSTALLATION)
            ) {
                continue;
            }
            if ($this->laravel->make('files')->exists(cms_base_path($path))) {
                $this->write(
                    'Suppression du fichier <fg=cyan>'.cms_base_path($path).'</fg=cyan>'
                );
                $result = $this->laravel->make('files')->delete(cms_base_path($path));
                $this->writeResult($result);
                if (! $result) {
                    $status = false;
                }
            }
        }
        if (! $status) {
            $this->info("Certains fichiers n'ont pu être effacés");
        }
        $this->line(str_repeat('-', 80));
    }

    protected function installNecessaryFiles(): void
    {
        $this->line('Copie des fichiers nécessaires');
        $done = false;
        foreach (self::FILE_TO_COPY as $path => $info) {
            if ($path === '.env' && env('DOCKER_CONTAINER')) {
                $this->writeln('Pas de copie du .env dans un environnement docker');
                continue;
            }

            if (! $this->laravel->make('files')->exists(cms_base_path($info['path']))) {
                $this->warning(
                    'Le fichier ['.cms_base_path($info['path']).'] est manquant.'
                );
                continue;
            }
            if (
                (! $this->laravel->make('files')->exists(cms_base_path($path)) || $info['overwrite'])
                && $this->laravel->make('files')->exists(dirname(cms_base_path($path)))
            ) {
                $this->info(
                    'Copie du fichier ['.cms_base_path($info['path']).'] => ['.cms_base_path($path).']'
                );
                $this->laravel->make('files')->copy(cms_base_path($info['path']), cms_base_path($path));
                $done = true;
            }
        }
        if (! $done) {
            $this->info('Tout est OK');
        }
        $this->line(str_repeat('-', 80));
    }

    protected function createSymLinks(): void
    {
        $this->line('Création des liens symboliques');
        foreach (self::SYMBOLIC_LINKS as $from => $to) {
            $from = cms_base_path($from);
            $to = cms_base_path($to);
            if ($this->laravel->make('files')->exists($to) && is_link($to)) {
                $this->info('Lien symbolique déjà existant ['.$to.'] => on le supprime');
                $this->laravel->make('files')->delete($to);
            }
            $this->info('Création du lien symbolique ['.$from.'] => ['.$to.']');
            if ($this->getSystemInstallation() === self::LINUX_SYSTEM_INSTALLATION || ! is_file($from)) {
                $this->laravel->make('files')->link($from, $to);
            }
        }
        $this->line(str_repeat('-', 80));
    }

    protected function getSystemInstallation(): string
    {
        return windows_os() ? self::WINDOWS_SYSTEM_INSTALLATION : self::LINUX_SYSTEM_INSTALLATION;
    }

    protected function checkAssets(): void
    {
        $this->line('Vérification des assets');
        if (! $this->laravel->make('files')->exists(cms_base_path('assets'))) {
            $this->call('asset:link');
            $this->info('Lien symbolique des assets créé ');
        } else {
            $this->info('Tout est OK');
        }
        $this->line(str_repeat('-', 80));
    }

    protected function checkHorizonAssets(): void
    {
        $this->line('Publication des assets Horizon');
        $this->call('horizon:assets');
        $this->line(str_repeat('-', 80));
    }

    protected function checkPublicLinks(): void
    {
        $this->line('Vérification des liens symboliques publics');
        $this->call('public:link');
        $this->line(str_repeat('-', 80));
    }

    protected function migrate(): void
    {
        $this->line('Mise-à-jour de la base de données');

        $this->call('migrate', ['--force' => true]);
        $this->line(str_repeat('-', 80));
    }

    protected function queueRestart(): void
    {
        $this->line('Redémarrage des workers');

        $this->call('queue:restart');
        $this->line(str_repeat('-', 80));
    }

    protected function removeDevelGenerateModule(): void
    {
        $query = DB::table('key_value')->where('collection', 'system.schema')->where('name', 'devel_generate');
        if (! $query->exists()) {
            return;
        }
        $this->write('Suppression de <fg=cyan>devel_generate</fg=cyan>');
        $this->writeResult($query->delete() > 0);
        $this->line(str_repeat('-', 80));
    }

    protected function drush(string $command): void
    {
        $this->line("Exécution de drush [$command]");

        $process = $this->getProcess(cms_base_path());

        $process->setCommandLine(
            ($this->getSystemInstallation() === self::WINDOWS_SYSTEM_INSTALLATION ? 'vendor\\bin\\drush '
                : 'vendor/bin/drush ').$command
        );

        $this->info('  drush '.$command);
        $process->run();

        try {
            if (! $process->isSuccessful()) {
                throw new ProcessFailedException($process);
            }
        } catch (ProcessFailedException $exception) {
            Log::debug($exception->getMessage());
            $this->warn('drush failed');
        }

        $this->warn($process->getErrorOutput());
        $this->info($process->getOutput());
        $this->line(str_repeat('-', 80));
    }

    protected function patchVendors(): void
    {
        $this->line('Application des patches');
        foreach (self::PATCHES as $patch) {
            $this->info('Préparation process dans ['.cms_base_path($patch['cwd']).']');
            $process = $this->getProcess(cms_base_path($patch['cwd']));
            $command = "patch -N --batch -p{$patch['level']} < ".cms_base_path($patch['patch']);
            $process->setCommandLine($command);

            $this->info('  Patch ['.$command.']');
            $process->run();

            try {
                if (! $process->isSuccessful()) {
                    throw new ProcessFailedException($process);
                }
            } catch (ProcessFailedException $exception) {
                Log::debug($exception->getMessage());

                if (str($process->getOutput())->contains('patch detected!')) {
                    $this->info('  Patch already applied');
                } else {
                    $this->warn('  Patch failed');
                    $this->warn($process->getOutput());
                }
            }
        }
        $this->line(str_repeat('-', 80));
    }

    protected function generateAppKey(): void
    {
        if (env('DOCKER_CONTAINER')) {
            $this->writeln('Pas de génération de la clé de sécurité dans un environnement docker');

            return;
        }

        $this->writeln('Génération de la clé de sécurité');
        $this->call('key:generate', ['--force' => true]);
        $this->line(str_repeat('-', 80));
    }

    protected function reloadPackages(): void
    {
        $this->line('Rechargement des packages insides');

        Package::reload();

        foreach (Package::list() as $package => $info) {
            $this->info("{$package} : {$info->getVersion()} ( {$info->getCommit()} ) ");
        }
        $this->line(str_repeat('-', 80));
    }

    protected function checkVersion(bool $reload = true): void
    {
        $this->line('Vérification de la version');
        if (! $this->laravel->make('files')->exists(cms_base_path('version.yml'))) {
            $this->info('Fichier de version manquant =>  initialisation 1.0.0');

            $this->laravel->make('files')->put(
                cms_base_path('version.yml'),
                "version:\n  major: 1\n  minor: 0\n  patch: 0"
            );
        } else {
            if ($reload) {
                $this->line('Version reload');
                $this->call('version:reload');
            } else {
                $this->info('Tout est OK');
            }
        }
        $this->line(str_repeat('-', 80));
    }

    protected function cleanCache(): void
    {
        try {
            $this->line('Nettoyage cache');
            $this->call('cache:clear');
            $this->call('cache:clear', ['store' => 'file']);
            $this->line(str_repeat('-', 80));
        } catch (\Exception $exception) {
            $this->error('Connexion à Redis impossible, le cache n\'a pas pu être nettoyé');
        }
    }

    protected function cleanViewCache(): void
    {
        $this->line('Nettoyage cache vue');
        $this->call('view:clear');
        $this->line(str_repeat('-', 80));
    }

    protected function rebuildModels(): void
    {
        try {
            $this->line('Reconstruction des classes dynamiques');
            $this->composer->dumpAutoloads();
            $this->call('models:rebuild');
            $this->line(str_repeat('-', 80));
        } catch (Throwable $exception) {
            $this->error('Connexion à Redis impossible, le cache n\'a pas pu être nettoyé');
        }
    }

    protected function dumpAutoloads(): void
    {
        $this->line('Reconstruction des classes dynamiques');
        $this->composer->dumpAutoloads();
        $this->line(str_repeat('-', 80));
    }

    protected function rebuildIndexes(): void
    {
        $this->line("Reconstruction de l'indexation de recherche");
        $this->call('index:rebuild');
        $this->line(str_repeat('-', 80));
    }

    protected function updateThemeConfig(): void
    {
        $this->line('Mise-à-jour des paramètres du thème.');
        $this->warning('Il se peut qu\'il faille lancer la commande à la main après avoir rebuild le front');
        $this->call('inside:theme:update');
        $this->line(str_repeat('-', 80));
    }

    protected function rebuildIdeHelpers(): void
    {
        if (
            $this->laravel->environment() === 'local'
            && class_exists(\Barryvdh\LaravelIdeHelper\IdeHelperServiceProvider::class)
        ) {
            $this->line('Préparation des aliases');

            $aliases = [];
            foreach (Package::list() as $package => $information) {
                $facadePath = cms_base_path('vendor/'.$information->getName().'/src/Facades');

                $finder = new Finder();
                if (File::exists($facadePath)) {
                    foreach (
                        $finder->in($facadePath)->files()->name('*.php') as $file
                    ) {
                        $className = \pathinfo($file->getFilename(), PATHINFO_FILENAME);
                        $baseClass = $className;
                        foreach ($information->getNamespaces() as $namespace => $path) {
                            $localPath = cms_base_path(
                                'vendor/'.$information->getName().'/'.$path.'Facades/'.$file->getFilename()
                            );
                            if (str_replace('/', DIRECTORY_SEPARATOR, $localPath) == $file->getRealPath()) {
                                $className = $namespace.'Facades\\'.Str::studly($className);
                            }
                        }
                        $aliases['Inside_'.$baseClass] = $className;
                    }
                }
            }

            // Add special inside facades
            $aliases['Inside'] = Inside::class;
            $aliases['InsidePackage'] = Package::class;

            $aliases = array_merge(
                $aliases,
                [
                    'App' => App::class,
                    'Arr' => Arr::class,
                    'Artisan' => Artisan::class,
                    'Auth' => Auth::class,
                    'Blade' => Blade::class,
                    'Broadcast' => Broadcast::class,
                    'Bus' => Bus::class,
                    'Cache' => Cache::class,
                    'Config' => Config::class,
                    'Cookie' => Cookie::class,
                    'Crypt' => Crypt::class,
                    'DB' => DB::class,
                    'Eloquent' => Model::class,
                    'Event' => Event::class,
                    'File' => File::class,
                    'Gate' => Gate::class,
                    'Hash' => Hash::class,
                    'Lang' => Lang::class,
                    'Log' => Log::class,
                    'Mail' => Mail::class,
                    'Notification' => Notification::class,
                    'Password' => Password::class,
                    'Queue' => Queue::class,
                    'Redirect' => Redirect::class,
                    'Redis' => Redis::class,
                    'Request' => Request::class,
                    // 'Response'     => Response::class,
                    'Route' => Route::class,
                    'Schema' => Schema::class,
                    'Session' => Session::class,
                    'Storage' => Storage::class,
                    'Str' => Str::class,
                    'URL' => URL::class,
                    'Validator' => Validator::class,
                    'View' => View::class,
                ]
            );

            config(['app.aliases' => array_merge(config('app.aliases', []), $aliases)]);
            AliasLoader::getInstance(config('app.aliases'))->register();

            $this->line("Rechargement de l'autocomplétion");
            $this->call('ide-helper:generate');
            $this->call('ide-helper:meta');
            $this->line(str_repeat('-', 80));
        }
    }

    protected function secureResources(): void
    {
        $this->line('Sécurisation des storages');
        $this->call('inside:secure:resources');
        $this->line(str_repeat('-', 80));
    }

    protected function saveLastCleanup(): void
    {
        file_put_contents(
            storage_path('framework/inside_cleanup'),
            json_encode(
                [
                    'time' => $this->currentTime(),
                ],
                JSON_PRETTY_PRINT
            )
        );
    }

    protected function needToCleanup(): bool
    {
        if (! file_exists(storage_path('framework/inside_cleanup'))) {
            $this->saveLastCleanup(); // Initial save

            return true;
        }

        /** @var string $fileContent */
        $fileContent = file_get_contents(storage_path('framework/inside_cleanup'));
        $value = json_decode($fileContent);
        if ($value === null || ! isset($value->time)) {
            unlink(storage_path('framework/inside_cleanup')); // Wrong filename

            return true;
        }

        $lastUpdate = get_date($value->time);
        if ($lastUpdate?->diffInDays() >= 7) {
            $this->saveLastCleanup(); // Update last cleanup time

            return true;
        }

        return false;
    }

    protected function cleanUp(): void
    {
        $this->write("<fg=cyan>Nettoyage d'Inside nécessaire</fg=cyan>");
        if (! $this->needToCleanup()) {
            $this->writeResult(false);

            return;
        }
        $this->writeResult(true);

        //   $this->line("Nettoyage complet d'Inside");
        //   $this->call('inside:cleanup --check'); // Atm only check
        //   $this->line('');
        $this->line('Nettoyage des notifications');
        $this->call('inside:notif:cleanup');

        $this->line('Nettoyage des réactions');
        $this->call('inside:reaction:cleanup');

        $this->line('Nettoyage des incohérences de permission');
        $this->call('inside:permission:cleanup');

        $this->line('Nettoyage des tests oubliés');
        $this->call('inside:tests:cleanup');

        $this->line(str_repeat('-', 80));
    }

    protected function setAssetsLink(): void
    {
        $this->line('Création des liens symboliques pour les assets des settings');
        $this->call('assets:link');
        $this->line(str_repeat('-', 80));
    }

    protected function setMenusLink(): void
    {
        $this->line('Création des liens symboliques pour les menus');
        $this->call('menus:link');
        $this->line(str_repeat('-', 80));
    }

    protected function insideIsInstalled(): bool
    {
        if (file_exists(storage_path('framework/inside_installed'))) {
            return true;
        }

        return false;
    }

    protected function markInsideHasInstalled(): void
    {
        $this->line("Finalisation de l'installation");
        file_put_contents(
            storage_path('framework/inside_installed'),
            json_encode(
                [
                    'installed' => $this->currentTime(),
                ],
                JSON_PRETTY_PRINT
            )
        );
        $this->line(str_repeat('-', 80));
    }

    protected function updateInstalledLockFile(): void
    {
        // Update inside_installed file
        $installedData = [];
        try {
            /** @var string $fileContent */
            $fileContent = file_get_contents(storage_path('framework/inside_installed'));
            $installedData = json_decode($fileContent, true);
            if (isset($installedData['time'])) {
                $installedData['installed'] = $installedData['time'];
            }
            $installedData['updated'] = $this->currentTime();
        } catch (Throwable $exception) {
        }
        file_put_contents(
            storage_path('framework/inside_installed'),
            json_encode($installedData, JSON_PRETTY_PRINT)
        );
    }

    protected function getProcess(string $workingPath): Process
    {
        return (new Process([], $workingPath))->setTimeout(null);
    }

    protected function checkJwtSecretAndSetIfNeeded(): void
    {
        $this->line('Vérification de la configuration de Jwt');
        $this->write('<fg=cyan>Clé privé JWT présente ?</fg=cyan>');
        if (! is_null(env('JWT_SECRET'))) {
            $this->writeResult(true);

            $this->line(str_repeat('-', 80));

            return;
        }
        $this->writeResult(false);

        if (! app()->providerIsLoaded(LumenServiceProvider::class)) {
            app()->register(LumenServiceProvider::class);
            $jwtCommand = new JWTGenerateSecretCommand();
            $jwtCommand->setAliases(['jwt:secret']);
            $this->getApplication() and $this->getApplication()->add($jwtCommand);
        }

        $this->call('jwt:secret', ['--always-no' => true, '--force' => true]);
        $this->line(str_repeat('-', 80));
    }

    protected function upgradeDrupalSettings(): void
    {
        $this->line('Mise-à-jour des settings drupal');
        $this->upgradeDiscoveryCacheDrupalSettings();
        $this->upgradeDotenvDrupalSettings();
        $this->upgradeFilePathsDrupalSettings();
        $this->addVerboseModeOnDrupalSettings();
        $this->line(str_repeat('-', 80));
    }

    protected function upgradeFilePathsDrupalSettings(): void
    {
        $this->write('Mise-à-jour du système <fg=magenta>File Paths</fg=magenta>');
        $drupalSettingsPath = cms_base_path('sites/default/settings.php');
        File::chmod($drupalSettingsPath, 0664);
        $code = File::get($drupalSettingsPath);

        if (preg_match('#\\$settings\[\'file_private_path\'\]\s*\=\s*null;#mi', $code)) {
            $this->writeResult(false);
            $this->writeln('<fg=red>Mise-à-jour déjà effective</fg=red>');

            return;
        }

        $code = preg_replace(
            [
                '#(\\$settings\[\'file_public_path\'\]\s*\=\s*)[^;]*;#mi',
                '#(\\$settings\[\'file_private_path\'\]\s*\=\s*)[^;]*;#mi',
            ],
            [
                '${1}env(\'APP_STORAGE_PATH\', \'sites/default/files/public\');',
                '${1}null;',
            ],
            $code
        );

        if (is_string($code)) {
            $this->writeResult(true);
            $this->write('<fg=cyan>Sauvegarde du fichier settings.php</fg=cyan>');
            $this->writeResult(File::put($drupalSettingsPath, $code) !== false);
        } else {
            $this->writeResult(false);
        }

        $this->line(str_repeat('-', 80));
    }

    protected function upgradeDotenvDrupalSettings(): void
    {
        $this->write('Mise-à-jour du nouveau système <fg=magenta>Dotenv</fg=magenta>');
        $drupalSettingsPath = cms_base_path('sites/default/settings.php');
        File::chmod($drupalSettingsPath, 0664);
        $code = File::get($drupalSettingsPath);
        if (! Str::contains($code, 'Inside\\Support\\LoadEnvironmentVariables')) {
            $this->writeResult(false);
            $this->writeln('<fg=red>Mise-à-jour déjà effective</fg=red>');

            return;
        }

        $oldCodes = [
            '#try\s+\{\r?\n\s+\(new\s+Dotenv\\\\Dotenv\(__DIR__\s+\.\s+\'/\.\./\.\./\'\)\)->load\(\);\r?\n\}\s+catch\s+\(Dotenv\\\\Exception\\\\InvalidPathException\s+\$e\)\s+\{\r?\n\}#mi',
            '#\(new\ \\\\Inside\\\\Support\\\\LoadEnvironmentVariables\(\r?\n\s+dirname\(realpath\(__DIR__\.\'\/\.\.\/\'\)\)\r?\n\s*\)\)->bootstrap\(\);#mi',
        ];

        $newCode = <<<CODE
(new \Laravel\Lumen\Bootstrap\LoadEnvironmentVariables(
    dirname(realpath(__DIR__.'/../'))
))->bootstrap();
\$settings['config_sync_directory'] = 'sites/default/files/config/sync';
\$settings['skip_permissions_hardening'] = TRUE;
CODE;
        $code = preg_replace($oldCodes, $newCode, $code, -1);

        if (is_string($code)) {
            $oldCode = '#\$settings\[\'install_profile\'\]\s+=\s+\'minimal\';\r?\n#mi';
            $code = preg_replace($oldCode, '', $code, -1);
        }

        $this->writeResult(true);

        if (is_string($code)) {
            $this->write('<fg=cyan>Sauvegarde du fichier settings.php</fg=cyan>');
            File::chmod($drupalSettingsPath, 0664);
            $this->writeResult(File::put($drupalSettingsPath, $code) !== false);
        }

        $this->line(str_repeat('-', 80));
    }

    protected function upgradeDiscoveryCacheDrupalSettings(): void
    {
        $this->write('Activation du cache <fg=magenta>Discovery</fg=magenta>');
        $drupalSettingsPath = cms_base_path('sites/default/settings.php');
        File::chmod($drupalSettingsPath, 0664);
        $code = File::get($drupalSettingsPath);

        // $settings['cache']['bins']['discovery']          = 'cache.backend.null';
        $oldCode = '#\$settings\[\'cache\'\]\[\'bins\'\]\[\'discovery\'\]\s+=\s+\'cache.backend.null\';#mi';
        if (preg_match($oldCode, $code) === 0) {
            $this->writeResult(false);
            $this->writeln('<fg=red>Mise-à-jour déjà effective</fg=red>');

            return;
        }

        $newCode = <<<'CODE'
$settings['cache']['bins']['discovery']          = 'cache.backend.database';
CODE;
        $count = 0;
        $code = preg_replace($oldCode, $newCode, $code, -1, $count);

        $this->writeResult(true);

        if (is_string($code)) {
            $this->write('<fg=cyan>Sauvegarde du fichier settings.php</fg=cyan>');
            File::chmod($drupalSettingsPath, 0664);
            $this->writeResult(File::put($drupalSettingsPath, $code) !== false);
        }

        $this->line(str_repeat('-', 80));
    }

    protected function addVerboseModeOnDrupalSettings(): void
    {
        $this->write("Ajout du paramètre du <fg=magenta>niveau d'erreur</fg=magenta>");
        $drupalSettingsPath = cms_base_path('sites/default/settings.php');
        File::chmod($drupalSettingsPath, 0664);
        $code = File::get($drupalSettingsPath);

        if (str_contains($code, '# Error level v2')) {
            $this->writeResult(false);
            $this->writeln('<fg=red>Mise-à-jour déjà effective</fg=red>');

            return;
        }

        $code = str_replace('# Error level', '', $code);
        $code = preg_replace('/\$config\[\'system.logging\'\]\[\'error_level\'\](.*?);/', '', $code);

        $newCode = <<<'CODE'
# Error level v2
$config['system.logging']['error_level'] = ERROR_REPORTING_HIDE;
CODE;
        $count = 0;
        $code .= "\n$newCode";

        $this->writeResult(true);

        $this->write('<fg=cyan>Sauvegarde du fichier settings.php</fg=cyan>');
        File::chmod($drupalSettingsPath, 0664);
        $this->writeResult(File::put($drupalSettingsPath, $code) !== false);

        $this->line(str_repeat('-', 80));
    }

    protected function optimize(): void
    {
        $this->write("<fg=magenta>Optimisation de l'autoload</fg=magenta>");
        $this->composer->dumpOptimized();
        $this->writeResult(true);
        $this->writeln('<fg=magenta>Mise en cache des routes</fg=magenta>');
        $this->call('inside:route:cache');
        $this->line(str_repeat('-', 80));
    }

    protected function rebuildTranslations(): void
    {
        $this->writeln('<fg=cyan>Reconstruction des traductions</fg=cyan>');
        $this->write('<fg=yellow>Synchronisation des traductions depuis le front spécifique</fg=yellow>');
        $this->call('inside:translation:front:sync');
        $this->writeResult(true);
        $this->write('<fg=yellow>Nettoyage des traductions overrides pérénisées</fg=yellow>');
        $this->call('inside:translation:cleanup');
        $this->writeResult(true);
        $this->writeln('<fg=yellow>Nettoyage des caches de traduction</fg=yellow>');
        $this->call('inside:translation:cache:clear');
        $this->writeResult(true);
        $this->line(str_repeat('-', 80));
    }

    protected function clearCacheRoute(): void
    {
        $this->line('Nettoyage du cache des routes');
        $this->call('inside:route:clear');
        $this->line(str_repeat('-', 80));
    }

    protected function insideUpdatedSuccessfully(): void
    {
        file_put_contents(
            storage_path('framework/inside_updated'),
            json_encode(
                [
                    'updated' => $this->currentTime(),
                ],
                JSON_PRETTY_PRINT
            )
        );
    }

    protected function addOneSignalFiles(): void
    {
        if (! config('onesignal.desktop_activated')) {
            return;
        }

        $this->line('Add Onesignal js files');
        $this->call('inside:onesignal:install');
        $this->line(str_repeat('-', 80));
    }
}
