<?php

declare(strict_types=1);

namespace Inside\Console\Commands;

use Doctrine\DBAL\Schema\Table;
use Illuminate\Database\ConnectionResolverInterface;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Schema;
use Inside\Application;
use Inside\Console\Command;

final class InstallEmoji extends Command
{
    protected $name = 'inside:install:emoji';

    protected $signature = 'inside:install:emoji {--force : Force update to utf8mb4}';

    protected $description = 'Install les emoji si non disponible';

    /** @var array<string, string> */
    protected array $typeMappings = [
        'bit' => 'string',
        'citext' => 'string',
        'enum' => 'string',
        'geometry' => 'string',
        'geomcollection' => 'string',
        'linestring' => 'string',
        'ltree' => 'string',
        'multilinestring' => 'string',
        'multipoint' => 'string',
        'multipolygon' => 'string',
        'point' => 'string',
        'polygon' => 'string',
        'sysname' => 'string',
    ];

    public function handle(ConnectionResolverInterface $connections): int
    {
        if (env('DB_CONNECTION') !== 'mysql') {
            $this->warn(__('Cette commande ne fonctionne que pour le driver "mysql"'));

            return self::SUCCESS;
        }

        $charset = config('database.connections.'.config('database.default').'.charset');

        $force = (bool) $this->option('force');

        if (! $force && $charset === 'utf8mb4') {
            $this->warn(__('Les emoji sont déjà disponible sur cette instance !'));

            return self::SUCCESS;
        }

        // Mise-à-jour du dot env
        $this->updateDotEnv();

        $connection = $connections->connection();

        if (! method_exists($connection, 'getDoctrineSchemaManager') || ! method_exists($connection, 'getDoctrineConnection')) {
            return self::FAILURE;
        }

        $schema = $connection->getDoctrineSchemaManager();
        foreach ($this->typeMappings as $type => $value) {
            $connection->getDoctrineConnection()->getDatabasePlatform()->registerDoctrineTypeMapping($type, $value);
        }

        // Mise-à-jour des tables
        $this->updateTables(collect($schema->listTables())->flatMap(fn (Table $table) => [$table->getName()])->toArray());

        return self::SUCCESS;
    }

    private function updateDotEnv(): void
    {
        $this->write('<fg=cyan>Mise-à-jour du .env</>');

        /** @var Application $app */
        $app = app();
        $env = File::get($app->environmentFilePath());
        if (preg_match('#^\s*DB_CHARSET\s*=\s*.*$#im', $env) > 0) {
            $env = preg_replace('#^\s*DB_CHARSET\s*=\s*.*$#im', 'DB_CHARSET=utf8mb4', $env, -1);
        } else {
            $env = preg_replace('#^\s*(DB_PASSWORD\s*=\s*.*)$#im', "$1\nDB_CHARSET=utf8mb4", $env, -1);
        }
        if (is_string($env)) {
            if (preg_match('#^\s*DB_COLLATION\s*=\s*.*$#im', $env) > 0) {
                $env = preg_replace('#^\s*DB_COLLATION\s*=\s*.*$#im', 'DB_COLLATION=utf8mb4_unicode_ci', $env, -1);
            } else {
                $env = preg_replace('#^DB_CHARSET=utf8mb4$#im', "DB_CHARSET=utf8mb4\nDB_COLLATION=utf8mb4_unicode_ci\n", $env, -1);
            }
        }
        if (is_string($env)) {
            $success = File::put($app->environmentFilePath(), $env);
            $this->writeResult($success !== false);
        }
    }

    /**
     * @param  array<int, string>  $tables
     */
    private function updateTables(array $tables): void
    {
        DB::statement('SET foreign_key_checks = 0');
        $this->writeln('<fg=magenta>Conversion des tables</>');
        foreach ($tables as $table) {
            $collation = DB::table('INFORMATION_SCHEMA.TABLES')->where('table_schema', env('DB_DATABASE'))->where('table_name', $table)->first('TABLE_COLLATION');
            if (! isset($collation->TABLE_COLLATION)
            || $collation->TABLE_COLLATION === 'utf8mb4_unicode_ci'
                || $collation->TABLE_COLLATION === 'utf8mb4_general_ci') {
                // Already set
                continue;
            }

            $this->write('<fg=cyan>'.$table.' => </>');
            $this->writeResult(DB::statement(
                "ALTER TABLE `$table` CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci"
            ));
        }
        DB::statement('SET foreign_key_checks = 1');
    }
}
