<?php

namespace Inside\Console\Commands;

use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\File;
use Inside\Authentication\Models\ApplicationClient;
use Inside\Console\Command;
use Tymon\JWTAuth\Claims\Factory;
use Tymon\JWTAuth\Facades\JWTAuth;
use Tymon\JWTAuth\Validators\PayloadValidator;

class CypressConfigCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inside:cypress:config';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Création du fichier cypress.env.json';

    public function handle(): void
    {
        try {
            File::put(
                cms_base_path('cypress.env.json'),
                $this->prepareCypressConfig()->toJson(
                    JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT
                )
            );
        } catch (Exception $e) {
            $this->writeln(
                "<fg=red>Echec de la génération du fichier <fg=cyan>cypress.env.json</fg=cyan>: <fg=magenta>{$e->getMessage()}</fg=magenta>.</fg=red>"
            );

            return;
        }

        $this->info(
            '<fg=cyan>cypress.env.json</fg=cyan> correctement généré.'
        );
    }

    /**
     * @throws Exception
     */
    protected function prepareCypressConfig(): Collection
    {
        return collect(
            [
                'APP_CODE'  => config('app.code'),
                'BASE_URL'  => config('app.url'),
                'JWT_TOKEN' => $this->getJwtToken(),
            ]
        );
    }

    /**
     * @return string|null
     * @throws Exception
     */
    protected function getJwtToken(): ?string
    {
        try {
            $client = ApplicationClient::where('name', 'cypress')->firstOrFail();
        } catch (ModelNotFoundException $exception) {
            throw new Exception(
                "L'application cypress n'est pas supporté par Inside pour le moment."
            );
        }
        JWTAuth::factory()->setTTL(null);
        app(Factory::class)->setTTL(null);
        app(PayloadValidator::class)->setRequiredClaims([
            'iss',
            'iat',
            'nbf',
            'sub',
            'jti',
        ]);

        return JWTAuth::fromSubject($client);
    }
}
