<?php

namespace Inside;

use Carbon\CarbonInterval;
use Closure;
use Exception;
use Illuminate\Contracts\Foundation\Application as ApplicationContract;
use Illuminate\Contracts\Notifications\Dispatcher as DispatcherContract;
use Illuminate\Contracts\Notifications\Factory as FactoryContract;
use Illuminate\Database\DatabaseServiceProvider;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Carbon;
use Illuminate\Support\Env;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Str;
use Inside\Concerns\ErrorHandling;
use Inside\Contracts\CachesRoutes;
use Inside\Events\InsideBooted;
use Inside\Events\LocaleUpdated;
use Inside\Facades\Package;
use Inside\I18n\Providers\TranslationServiceProvider;
use Inside\Logging\LogManager;
use Inside\Notify\ChannelManager;
use Inside\Providers\FilesystemServiceProvider;
use Inside\Providers\PackageServiceProvider;
use Inside\Providers\RouteServiceProvider;
use Inside\Routing\Router;
use Inside\Support\EnvironmentDetector;
use Inside\Support\Filesystem;
use Laravel\Lumen\Application as LumenApplication;

/**
 * Class Application
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 *
 * @property string|null $basePath
 * @property string|null $corePath
 */
class Application extends LumenApplication implements ApplicationContract, CachesRoutes
{
    use ErrorHandling;

    protected array $insideAliases = [];

    /**
     * @var callable[]
     */
    protected array $bootingCallbacks = [];

    /**
     * @var callable[]
     */
    protected array $bootedCallbacks = [];

    protected ?string $environmentPath = null;

    protected string $environmentFile = '.env';

    protected ?bool $isRunningInConsole = null;

    protected bool $hasBeenBootstrapped = false;

    /**
     * @var string[]
     */
    protected array $absoluteCachePathPrefixes = ['/', '\\'];

    /**
     * Application constructor.
     *
     * @param  string|null  $basePath
     * @throws Exception
     */
    public function __construct($basePath = null)
    {
        parent::__construct($basePath);
        $this->registerFilesystem();
        $this->registerChannelManager();
    }

    protected function registerFilesystemBindings(): void
    {
        $this->singleton(
            'filesystem',
            function () {
                return $this->loadComponent(
                    'filesystems',
                    FilesystemServiceProvider::class,
                    'filesystem'
                );
            }
        );
        $this->singleton(
            'filesystem.disk',
            function () {
                return $this->loadComponent(
                    'filesystems',
                    'Illuminate\Filesystem\FilesystemServiceProvider',
                    'filesystem.disk'
                );
            }
        );
        $this->singleton(
            'filesystem.cloud',
            function () {
                return $this->loadComponent(
                    'filesystems',
                    'Illuminate\Filesystem\FilesystemServiceProvider',
                    'filesystem.cloud'
                );
            }
        );
    }

    protected function bootstrapContainer()
    {
        static::setInstance($this);

        $this->instance('app', $this);
        $this->instance(self::class, $this);

        $this->instance('path', $this->path());

        $this->instance('env', $this->environment());

        $this->registerContainerAliases();
    }

    /**
     * Get a storage Path
     *
     * @param $path
     * @return string
     */
    public function storagePath($path = ''): string
    {
        return cms_base_path('storage').($path ? DIRECTORY_SEPARATOR.$path : $path);
    }

    /**
     * Bootstrap the router instance.
     */
    public function bootstrapRouter(): void
    {
        $this->router = new Router($this);
    }

    /**
     * Register file system bindings
     */
    protected function registerFilesystem(): void
    {
        $this->availableBindings['filesystem'] = 'registerFilesystemBindings';
        $this->availableBindings['filesystem.disk'] = 'registerFilesystemBindings';
        $this->availableBindings['Illuminate\Contracts\Filesystem\Filesystem'] = 'registerFilesystemBindings';
        $this->availableBindings['Illuminate\Contracts\Filesystem\Factory'] = 'registerFilesystemBindings';
    }

    /**
     * Register file system binding
     */
    protected function registerFilesBindings(): void
    {
        $this->singleton('files', function () {
            return new Filesystem();
        });
    }

    /**
     * is Inside down for maintenance
     *
     * @return bool
     */
    public function isDownForMaintenance(): bool
    {
        return file_exists($this->storagePath().'/framework/inside_down');
    }

    /**
     * Application has booted ?
     *
     * @return bool
     */
    public function isBooted(): bool
    {
        return $this->booted;
    }

    /**
     * @param  ServiceProvider|string  $provider
     * @param  bool  $noBoot
     * @return ServiceProvider|null
     * @throws Exception
     */
    public function register($provider, $noBoot = false): ?ServiceProvider
    {
        if (! $provider instanceof ServiceProvider) {
            $provider = new $provider($this);
        }

        if (! $provider instanceof ServiceProvider) {
            return null;
        }

        if (array_key_exists($providerName = get_class($provider), $this->loadedProviders)) {
            return null; // Already registered !
        }

        if (method_exists($provider, 'register')) {
            $provider->register();
        }

        if (property_exists($provider, 'bindings')) {
            foreach ($provider->bindings as $key => $value) {
                $this->bind($key, $value);
            }
        }

        if (property_exists($provider, 'singletons')) {
            foreach ($provider->singletons as $key => $value) {
                $this->singleton($key, $value);
            }
        }

        $this->loadedProviders[$providerName] = $provider;

        if (! $noBoot && (($provider instanceof DatabaseServiceProvider || $this->isBooted()))) {
            $this->bootProvider($provider);
        }

        return $provider;
    }

    /**
     * @throws Exception
     */
    public function boot()
    {
        if ($this->isBooted()) {
            return;
        }
        $this->fireAppCallbacks($this->bootingCallbacks);

        foreach ($this->loadedProviders as $provider) {
            if (! $provider instanceof RouteServiceProvider && ! $provider instanceof DatabaseServiceProvider) {
                $this->bootProvider($provider);
            }
        }

        $this->booted = true;

        // Register package service providers ( now that all registered provider are booted )
        // It will be directly booted if needed
        $this->register(PackageServiceProvider::class);

        // Boot main route provider
        if ($this->providerIsLoaded(RouteServiceProvider::class)) {
            $this->bootProvider($this->loadedProviders[RouteServiceProvider::class]);
        }

        $this->fireAppCallbacks($this->bootedCallbacks);

        InsideBooted::dispatch($this);
    }

    public function getMiddlewares(): array
    {
        return $this->middleware;
    }

    /**
     * Does the provider $class is loaded
     */
    public function providerIsLoaded(ServiceProvider|string $provider): bool
    {
        $name = is_string($provider) ? $provider : get_class($provider);

        return array_key_exists($name, $this->loadedProviders) && $this->loadedProviders[$name];
    }

    protected function fireAppCallbacks(array $callbacks): void
    {
        foreach ($callbacks as $callback) {
            $callback($this);
        }
    }

    /**
     * Get current application locale
     */
    public function getLocale(): string
    {
        return $this['config']->get('app.locale');
    }

    /**
     * Set current locale
     */
    public function setLocale($locale): void
    {
        $fullLocale = null;
        if (strlen($locale) > 3) {
            $fullLocale = $locale;
            [$locale, $unused] = explode('_', $fullLocale, 2);
        }

        $this['config']->set('app.locale', $locale);

        $this['translator']->setLocale($locale);

        $this['events']->dispatch(new LocaleUpdated($locale));

        $this->refreshLocales($locale, $fullLocale);
    }

    /**
     *  Refresh all tools to use $locale or current locale
     */
    public function refreshLocales(?string $locale = null, ?string $fullLocale = null): void
    {
        if (is_null($locale)) {
            $locale = $this->getLocale();
        }
        if ($fullLocale === null) {
            $fullLocale = $locale.'_'.Str::Upper($locale);
        }
        setlocale(LC_TIME, $locale, $fullLocale, $fullLocale.'.utf8');
        $currentLocale = setlocale(LC_TIME, '0');

        if ($currentLocale && ! Str::endsWith($currentLocale, '.utf8')) {
            Carbon::setUtf8(true);
        }

        Carbon::setLocale($locale);
        CarbonInterval::setLocale($locale);
    }

    /**
     * Application is running locally ?
     */
    public function isLocal(): bool
    {
        return $this['env'] === 'local';
    }

    /**
     * Application is in production ?
     */
    public function isProduction(): bool
    {
        return $this['env'] === 'production';
    }

    protected function bootProvider(ServiceProvider $provider): void
    {
        if ($provider instanceof Providers\ServiceProvider) {
            $provider->callBootingCallbacks();
        }

        if (method_exists($provider, 'boot')) {
            $this->call([$provider, 'boot']);
        }

        if ($provider instanceof Providers\ServiceProvider) {
            $provider->callBootedCallbacks();
        }
    }

    /**
     * @param  mixed  $callback
     * @throws Exception
     */
    public function booting($callback)
    {
        $this->bootingCallbacks[] = $callback;
    }

    /**
     * @param  mixed  $callback
     */
    public function booted($callback)
    {
        $this->bootedCallbacks[] = $callback;

        if ($this->isBooted()) {
            $this->fireAppCallbacks([$callback]);
        }
    }

    /**
     * Get current used environment path
     *
     * @return string
     */
    public function environmentPath(): string
    {
        return $this->environmentPath ?: cms_base_path();
    }

    /**
     * use Environment path $path
     *
     * @param  string  $path
     * @return $this
     */
    public function useEnvironmentPath(string $path): static
    {
        $this->environmentPath = $path;

        return $this;
    }

    /**
     * Load environment from $file
     *
     * @param  string  $file
     * @return $this
     */
    public function loadEnvironmentFrom($file): static
    {
        $this->environmentFile = $file;

        return $this;
    }

    /**
     * Get environment file
     *
     * @return string
     */
    public function environmentFile(): string
    {
        return $this->environmentFile ?: '.env';
    }

    /**
     * Get environment file path
     *
     * @return string
     */
    public function environmentFilePath(): string
    {
        return $this->environmentPath().DIRECTORY_SEPARATOR.$this->environmentFile();
    }

    /**
     * get Bootstrap cache
     *
     * @param  string  $path
     * @return string
     */
    public function bootstrapPath($path = ''): string
    {
        return $this->basePath.DIRECTORY_SEPARATOR.'bootstrap'.($path ? DIRECTORY_SEPARATOR.$path : $path);
    }

    protected function registerTranslationBindings()
    {
        if (! class_exists(TranslationServiceProvider::class)) {
            parent::registerTranslationBindings();

            return;
        }
        $this->singleton(
            'translator',
            function () {
                $this->configure('app');

                $this->instance('path.lang', $this->getLanguagePath());

                $this->register(TranslationServiceProvider::class);

                return $this->make('translator');
            }
        );
    }

    /**
     * @throws Exception
     */
    public function registerConfiguredProviders()
    {
        // Not supported
        throw new Exception('Not supported');
    }

    /**
     * @return string|void
     * @throws Exception
     */
    public function getCachedServicesPath()
    {
        // Not supported
        throw new Exception('Not supported');
    }

    /**
     * @return string|void
     * @throws Exception
     */
    public function getCachedPackagesPath()
    {
        // Not supported
        throw new Exception('Not supported');
    }

    /**
     * @return bool
     */
    public function routesAreCached(): bool
    {
        return $this['files']->exists($this->getCachedRoutesPath());
    }

    /**
     * get Cached routes path
     *
     * @return string
     */
    public function getCachedRoutesPath(): string
    {
        return $this->normalizeCachePath('APP_ROUTES_CACHE', 'cache/routes.php');
    }

    /**
     * normalize a cache path
     */
    protected function normalizeCachePath(string $key, string $default): mixed
    {
        if (is_null($env = Env::get($key))) {
            return $this->bootstrapPath($default);
        }

        return Str::startsWith($env, $this->absoluteCachePathPrefixes)
            ? $env
            : $this->basePath($env);
    }

    public function getProviders($provider): array
    {
        $name = is_string($provider) ? $provider : get_class($provider);

        return Arr::where($this->loadedProviders, function ($value) use ($name) {
            return $value instanceof $name;
        });
    }

    public function getProvider(ServiceProvider|string $provider): ?ServiceProvider
    {
        return array_values($this->getProviders($provider))[0] ?? null;
    }

    /**
     * Application has been bootstrapped ?
     *
     * @return bool
     */
    public function hasBeenBootstrapped(): bool
    {
        return $this->hasBeenBootstrapped;
    }

    public function terminate()
    {
    }

    /**
     * @param  string  $provider
     * @param  null  $service
     */
    public function registerDeferredProvider($provider, $service = null)
    {
        parent::registerDeferredProvider($provider);
    }

    /**
     * @param  string|null  $path
     * @return string
     */
    public function basePath($path = null): string
    {
        if (isset($this->basePath)) {
            return $this->basePath.($path ? '/'.$path : $path);
        }

        if ($this->runningInConsole()) {
            $this->basePath = getcwd().'/vendor/maecia/inside';
        } else {
            if (is_string($basePath = realpath(getcwd().'/../'))) {
                $this->basePath = $basePath;
            }
        }

        return $this->basePath($path);
    }

    /**
     * @param  string|null  $path
     * @return string
     */
    public function corePath($path = null): string
    {
        if (isset($this->corePath)) {
            return $this->corePath.($path ? '/'.$path : $path);
        }

        if ($this->runningInConsole()) {
            $this->corePath = getcwd().'/vendor/maecia/inside/core';
        } else {
            if (is_string($corePath = realpath(getcwd().'/../'))) {
                $this->corePath = $corePath;
            }
        }

        return $this->corePath($path);
    }

    /**
     * Add an inside internal alias
     *
     * @param  string  $abstract
     * @param  string  $alias
     */
    public function insideAlias(string $abstract, string $alias): void
    {
        $this->insideAliases[$abstract] = $alias;
    }

    /**
     * Get inside internal aliases
     *
     * @return array
     */
    public function getInsideAliases(): array
    {
        return $this->insideAliases;
    }

    /**
     * Get current route
     *
     * @return array
     */
    public function getCurrentRoute(): array
    {
        return $this->currentRoute;
    }

    /**
     * Register event service bindings
     */
    protected function registerEventBindings(): void
    {
        $this->singleton(
            'events',
            function () {
                $this->register('Inside\Providers\EventServiceProvider');

                return $this->make('events');
            }
        );
    }

    protected function registerChannelManager(): void
    {
        $this->singleton(ChannelManager::class, function ($app) {
            return new ChannelManager($app);
        });

        $this->alias(
            ChannelManager::class,
            DispatcherContract::class
        );

        $this->alias(
            ChannelManager::class,
            FactoryContract::class
        );
    }

    /**
     * $request route is a fallback ?
     *
     * @param  Request  $request
     * @return bool
     */
    public function routeIsFallback(Request $request): bool
    {
        $dispatched = $this->createDispatcher()->dispatch($request->getMethod(), $request->getPathInfo());

        if (count($dispatched) >= 2) {
            $found = $dispatched[0];
            $handler = $dispatched[1];
        } else {
            return false;
        }

        if ($found !== 1) {
            return false;
        }

        return array_key_exists('as', $handler) && in_array($handler['as'], ['fallback', 'maintenance', 'layout']);
    }

    /**
     * Does application is running in console ?
     *
     * @return bool
     */
    public function runningInConsole(): bool
    {
        if ($this->isRunningInConsole === null) {
            $this->isRunningInConsole = Env::get('APP_RUNNING_IN_CONSOLE') ?? (\PHP_SAPI === 'cli' || \PHP_SAPI === 'phpdbg');
        }

        return $this->isRunningInConsole;
    }

    /**
     * @param  array  $bootstrappers
     */
    public function bootstrapWith(array $bootstrappers): void
    {
        $this->hasBeenBootstrapped = true;

        foreach ($bootstrappers as $bootstrapper) {
            $this['events']->dispatch('bootstrapping: '.$bootstrapper, [$this]);

            $this->make($bootstrapper)->bootstrap($this);

            $this['events']->dispatch('bootstrapped: '.$bootstrapper, [$this]);
        }
    }

    public function detectEnvironment(Closure $callback): string
    {
        $args = $_SERVER['argv'] ?? null;

        return $this['env'] = (new EnvironmentDetector())->detect($callback, $args);
    }

    public function getCachedConfigPath(): string
    {
        return $this->normalizeCachePath('APP_CONFIG_CACHE', 'cache/config.php');
    }

    public function loadDeferredProviders()
    {
    }

    public function resolveProvider($provider): ServiceProvider
    {
        /** @phpstan-ignore-next-line */
        return new $provider($this);
    }

    protected function registerLogBindings()
    {
        $this->singleton('Psr\Log\LoggerInterface', function () {
            $this->configure('logging');

            return new LogManager($this);
        });
    }

    public function environment(...$environments): bool|string
    {
        if (count($environments) > 0) {
            $patterns = is_array($environments[0]) ? $environments[0] : $environments;

            return Str::is($patterns, parent::environment());
        }

        return parent::environment();
    }

    public function shouldSkipMiddleware(): bool
    {
        return $this->bound('middleware.disable') &&
            $this->make('middleware.disable') === true;
    }
}
