<?php

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Contracts\WysiwygImageService;
use Inside\Content\Exceptions\FieldSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Facades\ShortUrl;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Users;
use Inside\Content\Models\Field;
use Inside\Content\Models\Model as ContentModel;
use Inside\Content\Models\Section;
use Inside\Content\Services\WysiwygImage;

if (! function_exists('class_to_type')) {
    /**
     * Convert a class to a machine name type
     */
    function class_to_type(object|string $class = ''): string
    {
        return Str::snake(class_basename($class));
    }
}

if (! function_exists('type_to_class')) {
    /**
     * Convert a class to a machine name type
     */
    function type_to_class(?string $type = null): string
    {
        return 'Inside\\Content\\Models\\Contents\\'.Str::studly($type ?? '');
    }
}

if (! function_exists('section_type_to_class')) {
    /**
     * Convert a section type to its class name
     */
    function section_type_to_class(string $type = ''): string
    {
        return 'Inside\\Content\\Models\\Sections\\'.Str::studly($type);
    }
}

if (! function_exists('section_type_to_table')) {
    /**
     * Convert a section type to its table name
     * @param string $type
     * @return string
     */
    function section_type_to_table(string $type)
    {
        return 'inside_section_'.Str::snake($type);
    }
}

if (! function_exists('class_to_attribute')) {
    /**
     * Convert a class to a machine name type
     * @param string $class
     * @return string
     */
    function class_to_attribute(string $class = '')
    {
        return Str::camel(class_basename($class));
    }
}

if (! function_exists('table_to_class')) {
    /**
     * Convert a table to a class name
     * @param string $table
     * @return string|null
     */
    function table_to_class(string $table = '')
    {
        if ($table) {
            $type = str_replace('inside_content_', '', $table);

            return type_to_class($type);
        }

        return null;
    }
}

if (! function_exists('class_to_table')) {
    /**
     * Convert a class name to table
     * @param string $class
     * @return string|null
     */
    function class_to_table(string $class = '')
    {
        if ($class) {
            $table = str_replace('Inside\Content\Models\Contents\\', 'inside_content', $class);
            $table = str_replace('Inside\Content\Models\Sections\\', 'inside_section', $table);

            return Str::snake($table);
        }

        return null;
    }
}

if (! function_exists('type_to_table')) {
    /**
     * Convert a type to a table name
     * @param string $type
     * @return string|null
     */
    function type_to_table(string $type = '')
    {
        if ($type) {
            return 'inside_content_'.Str::snake($type);
        }

        return null;
    }
}

if (! function_exists('type_to_stats_table')) {
    /**
     * Convert a type to a stats table name
     * @param string $type
     * @return string|null
     */
    function type_to_stats_table(string $type = '')
    {
        if ($type) {
            return 'inside_statistics_'.Str::snake($type);
        }

        return null;
    }
}

if (! function_exists('protected_file_url')) {
    /**
     * Get a protected file url
     *
     * @param  Model  $content  Content or Section // TODO: ask Maxime if Section work if type Content or Model
     * @param  string  $fieldName  fieldName
     * @param  bool  $relative  relative url default true
     * @param  string|null  $style  optional style if available
     * @param  bool  $webp
     * @return string|null
     */
    function protected_file_url(Model $content, string $fieldName, bool $relative = true, ?string $style = null, bool $webp = false): ?string
    {
        $path = $content->{$fieldName};

        if (empty($path)) {
            return null;
        }

        try {
            $options = InsideSchema::getFieldOptions(class_to_type($content), $fieldName);
        } catch (FieldSchemaNotFoundException $e) {
            return null;
        }

        if ((empty($style) || $options['type'] !== 'image') && config('app.disable_protected_files', false) === true) {
            return ($relative ? '' : config('app.url').'/').config('app.app_storage_path').'/'.$path;
        }

        if (! in_array($options['type'], ['image', 'file'])) {
            return null;
        }

        if ($style !== null && $options['type'] == 'image') {
            try {
                $path = get_stylized_image_url($content, $fieldName, $style, $relative, $webp);

                if (config('app.disable_protected_files', false) === true) {
                    return $path;
                }
            } catch (Exception $e) {
            }
        }

        if ($path[0] != '/') {
            $path = '/'.$path;
        }

        $url = $path ? Storage::disk('local')->url(
            'files/'.($content->slug[0] ?? ($content->uuid.'/'.class_to_type($content))).$path
        ) : '';

        $return = $relative ? str_replace(Storage::disk('local')->url(''), '', $url) : $url;

        return str_replace('#', '%23', $return);
    }
}

if (! function_exists('newsletter_parse')) {
    /**
     * @param Content|null $newsletter
     * @param Users|User $user
     * @param Content $content
     * @param string $text
     * @return string
     */
    function newsletter_parse(?Content $newsletter, $user, Content $content, string $text): string
    {
        preg_match_all('/src\s*=\s*"([^"]*wysiwyg[^"]*)"/', $text, $matches);

        foreach ($matches[1] as $url) {
            $parsedUrl = parse_url($url);
            $path = ($parsedUrl && isset($parsedUrl['path'])) ? $parsedUrl['path'] : null;
            if (is_null($path)) {
                continue;
            }
            $path = str_replace('/wysiwyg/', 'wysiwyg/', $path);
            $new = newsletter_signed_url($user, $newsletter, $content, $path, '');
            $text = str_replace($url, $new, $text);
        }

        return preg_replace_callback(
             '/<img([^>]*class\s*=\s*"[^"]*inside-image[^"]*"[^>]*)>/i',
             function (array $matches): string {
                 $imgTag = $matches[0];

                 $widthValue = null;
                 $isTooWide = false;

                 if (preg_match('/\swidth\s*=\s*"(\d+)(%?)"/i', $imgTag, $widthMatch)) {
                     $unit = $widthMatch[2] ?: 'px';
                     $numericWidth = (int) $widthMatch[1];
                     $widthValue = $numericWidth.$unit;

                     if ($unit === 'px' && $numericWidth > 600) {
                         $isTooWide = true;
                         $widthValue = '100%';
                     }
                 }

                 $imgTag = preg_replace('/\swidth\s*=\s*"\d+%?"/i', '', $imgTag);

                 if (stripos($imgTag, 'style=') !== false) {
                     $imgTag = preg_replace_callback(
                         '/style\s*=\s*"(.*?)"/i',
                         function ($styleMatch) use ($widthValue) {
                             $style = $styleMatch[1];
                             $style .= '; max-width:100%; height:auto; display:block;';
                             if ($widthValue) {
                                 $style .= " width:{$widthValue};";
                             }

                             return 'style="'.trim($style).'"';
                         },
                         $imgTag // @phpstan-ignore-line
                     );
                 } else {
                     $style = 'max-width:100%; height:auto; display:block;';
                     if ($widthValue) {
                         $style .= " width:{$widthValue};";
                     }

                     $imgTag = preg_replace('/<img/i', '<img style="'.trim($style).'"', $imgTag); // @phpstan-ignore-line
                 }

                 if ($widthValue && ! str_contains($imgTag, 'width=')) {
                     $imgTag = preg_replace('/<img/', '<img width="'.$widthValue.'"', $imgTag); // @phpstan-ignore-line
                 }

                 return $imgTag;  // @phpstan-ignore-line
             },
             $text
         ) ?? '';
    }
}

if (! function_exists('newsletter_signed_url')) {
    /**
     * @throws Exception
     */
    function newsletter_signed_url(
        Users|User $user,
        ?Content $newletter,
        Content|Section $content,
        ?string $path,
        ?string $default
    ): string {
        if (is_null($path)) {
            if (is_null($default)) {
                throw new Exception('newsletter_signed_url path is null and does not have a default path');
            }

            return url($default);
        }

        if (config('app.disable_protected_files', false)) {
            $path = str_replace(config('app.app_storage_path').'/', '', $path);
        }

        $filePath = $path ? Storage::disk('local')->path($path) : null;

        if (Str::startsWith($path, 'wysiwyg/')) {
            $path = str_replace('wysiwyg/images/', '', $path);

            $service = new WysiwygImage();
            $image = $service->load($path);

            if ($image && md5_file($image->path) === $image->hash) {
                return url('newsletter_content_files/'.class_to_type($image).'/'.$image->id);
            }
        }

        if (! $filePath || ! Storage::exists($path)) {
            if (is_null($default)) {
                throw new Exception('newsletter_signed_url failed to get but does not have a default path');
            }

            return url($default);
        }

        return url('newsletter_content_files/'.class_to_type($content).'/'.$content->uuid);
    }
}

if (! function_exists('get_stylized_image_url')) {
    /**
     * Get an image using asked style, if not possible get main style one
     *
     * @param Model $content   content/section
     * @param string $fieldName field name
     * @param string $style     Style name
     * @param bool $relative  Using relative url ( default to false )
     * @param bool $webp
     * @return string|null
     */
    function get_stylized_image_url(Model $content, string $fieldName, string $style, bool $relative = false, bool $webp = false)
    {
        if (! InsideSchema::hasField(class_to_type($content), $fieldName)) {
            return null;
        }

        $options = InsideSchema::getFieldOptions(class_to_type($content), $fieldName);

        if ($options['type'] != 'image') {
            throw new InvalidArgumentException('get_stylized_image_url is only used on image field');
        }
        /** @var string|null $original */
        $original = $content->{$fieldName};
        if ($original == null) {
            return null;
        }

        // if style doesn't exist it might be the main webp we are looking for
        if ($style && ($style === 'main-webp' || isset($options['image_styles']) && ! empty($options['image_styles'])
            && in_array(
                $style,
                $options['image_styles']
            ))
        ) {
            if (! $webp && $style !== 'main-webp') {
                $stylizedImage = sprintf('styles/%s/%s', $style, $original);
            } else {
                $info = pathinfo($original);
                $stylizedImage = sprintf('styles/%s/%s/%s-%s.webp', ($style !== 'main-webp' ? $style : 'main'), $info['dirname'], $info['filename'], $info['extension'] ?? '');
            }
            if (Storage::disk('local')->exists($stylizedImage)) {
                if (config('app.disable_protected_files', false) === true) {
                    return ($relative ? '' : config('app.url').'/').config('app.app_storage_path').'/'.$stylizedImage;
                }

                $url = Storage::disk('local')->url($stylizedImage);
                /** @var string $relativeUrl */
                $relativeUrl = str_replace(Storage::disk('local')->url(''), '', $url);

                return $relative ? $relativeUrl : $url;
            }
        }

        // DISABLE_PROTECTED_FILES : true => full disable, 'unprotected' => only disable scope, false => not disable
        if (config('app.disable_protected_files', false) === true) {
            return config('app.app_storage_path').'/'.$original;
        }

        return $original;
    }
}

if (! function_exists('pretty_print_query_log')) {
    /**
     * @return void
     */
    function pretty_print_query_log(): void
    {
        $queries = [];
        foreach (DB::getQueryLog() as $query) {
            /** @var string $replaced */
            $replaced = str_replace('?', '\'%s\'', str_replace('%', '%%', $query['query']));
            $queries[] = vsprintf($replaced, $query['bindings']);
        }
        dump($queries);
    }
}

if (! function_exists('get_reference_class')) {
    function get_reference_class(string $type, string $field): ?string
    {
        // if full classname is given then we use it, else we assume it's a content
        if (Str::startsWith($type, "Inside\Content\Models")) {
            $contentType = $type;
        } else {
            $contentType = type_to_class($type);
        }
        $model = ContentModel::where('class', $contentType)->pluck('id')->first();
        if (is_null($model)) {
            return null;
        }
        /** @var Field|null $field */
        $field = Field::select(['type', 'options'])->where('model_id', $model)->where('name', $field)->first();

        if (is_null($field)) {
            return null;
        }
        $attributes = $field->getAttributes();
        if (! $attributes || $attributes['type'] !== 'reference') {
            return null;
        }
        $options = json_decode($attributes['options']);
        if (! $options || ! $options->target) {
            return null;
        }

        return type_to_class($options->target[0]);
    }
}

if (! function_exists('inside_shortener_content')) {
    function inside_shortener_content(string $content): string
    {
        $urlRegex = '_(?:(?:https?|ftp)://)(?:\S+(?::\S*)?@)?(?:(?!10(?:\.\d{1,3}){3})(?!127(?:\.\d{1,3}){3})(?!169\.254(?:\.\d{1,3}){2})(?!192\.168(?:\.\d{1,3}){2})(?!172\.(?:1[6-9]|2\d|3[0-1])(?:\.\d{1,3}){2})(?:[1-9]\d?|1\d\d|2[01]\d|22[0-3])(?:\.(?:1?\d{1,2}|2[0-4]\d|25[0-5])){2}(?:\.(?:[1-9]\d?|1\d\d|2[0-4]\d|25[0-4]))|(?:(?:[a-z\x{00a1}-\x{ffff}0-9]+-?)*[a-z\x{00a1}-\x{ffff}0-9]+)(?:\.(?:[a-z\x{00a1}-\x{ffff}0-9]+-?)*[a-z\x{00a1}-\x{ffff}0-9]+)*(?:\.(?:[a-z\x{00a1}-\x{ffff}]{2,})))(?::\d{2,5})?(?:/[^\s]*)?_iuS';

        if (preg_match_all($urlRegex, $content, $matches) > 0) {
            foreach ($matches[0] as $url) {
                $shortUrl = ShortUrl::destination($url)->make();
                $content = str_replace($url, $shortUrl->default_short_url, $content);
            }
        }

        return $content;
    }
}

if (! function_exists('inside_user_fullname')) {
    /**
     * get user full name from a Users Model
     *
     * @param  Users  $user
     * @return string
     */
    function inside_user_fullname(Users $user): string
    {
        return trim(Str::ucfirst($user->firstname ?? '').' '.Str::upper($user->lastname ?? ''));
    }
}

if (! function_exists('inside_user_fullname_with_email')) {
    /**
     * get user full name with email from a Users Model
     *
     * @param  Users  $user
     * @return string
     */
    function inside_user_fullname_with_email(Users $user): string
    {
        return inside_user_fullname($user).' <'.$user->email.'>';
    }
}

if (! function_exists('content_signed_url')) {
    /**
     * @throws Exception
     */
    function content_signed_url(
        User $user,
        Content|Section $content,
        ?string $path,
        ?string $default
    ): string {
        if (is_null($path)) {
            if (is_null($default)) {
                throw new Exception('content_signed_url path is null and does not have a default path');
            }

            return url($default);
        }
        if (config('app.disable_protected_files', false)) {
            $path = str_replace(config('app.app_storage_path').'/', '', $path);
        }
        $params = [
            'userUuid' => $user->uuid,
            'contentType' => class_to_type($content),
            'contentUuid' => $content->uuid,
            'path' => $path ? Storage::disk('local')->path($path) : null,
        ];

        if (Str::startsWith($path, 'wysiwyg/')) {
            $path = str_replace('wysiwyg/images/', '', $path);

            /** @var WysiwygImageService $service */
            $service = app(WysiwygImageService::class);
            $image = $service->load($path);

            if (! is_null($image) && md5_file($image->path) === $image->hash) {
                $params['path'] = $image->path;
                $path = str_replace(storage_path('app/'), '', $image->path);
            }
        }

        if (! $params['path'] || ! Storage::exists($path)) {
            // Default is not protected but who care ...
            if (is_null($default)) {
                throw new Exception('content_signed_url failed to get but does not have a default path');
            }

            return url($default);
        } else {
            // relativise path
            $params['path'] = str_replace(Storage::disk('local')->path(''), '', $path);
        }

        if (config('app.disable_protected_files', false)) {
            return config('app.url').'/'.config('app.app_storage_path').'/'.$params['path'];
        }

        /** @var string $encoded */
        $encoded = json_encode($params);
        $encrypted = encrypt($encoded);
        $signature = hash_hmac(
            'sha256',
            $encrypted,
            config('app.key')
        );

        return url('content_files/'.$signature.'/'.$encrypted);
    }
}

if (! function_exists('get_authorized_external_api_content_types')) {
    function get_authorized_external_api_content_types(): string
    {
        $types = Schema::getContentTypes();
        if (! empty(config('external_api.custom_types'))) {
            $types = [...$types, ...array_keys(config('external_api.custom_types'))];
        }

        return implode('|', $types);
    }
}

if (! function_exists('get_content_sections')) {
    function get_content_sections(Model $model): array
    {
        $sourceSections = $model->getSectionContentAttribute();
        $content = [];
        foreach ($sourceSections as $sourceSection) {
            $section = [
                'bundle' => str_replace('inside_section_', '', $sourceSection->getTable()),
            ];
            $section = array_merge($section, $sourceSection->toArray());
            $sourceSection = call_user_func(get_class($sourceSection).'::find', $sourceSection->uuid);

            if (isset($sourceSection->sectionContent)) {
                $section['content'] = [];
                $subSections = $sourceSection->sectionContent;

                foreach ($subSections as $subSectionContent) {
                    $subSection = [
                        'bundle' => str_replace('inside_section_', '', $subSectionContent->getTable()),
                    ];
                    $subSection = array_merge($subSection, $subSectionContent->toArray());
                    $section['content'][] = $subSection;
                    $subSectionContent = null;
                }
                $subSections = null;
            }
            $sourceSection = null;
            $content[] = $section;
        }

        return $content;
    }
}
