<?php

namespace Inside\Content\Transformers;

use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Contracts\RevisionService;
use Inside\Content\Contracts\Transformer;
use Inside\Content\Exceptions\FieldSchemaNotFoundException;
use Inside\Content\Exceptions\SchemaNotFoundException;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Facades\Package;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\Role as RoleService;
use Inside\Permission\Models\Role;
use Inside\Reaction\Models\Reaction;
use Inside\Slug\Models\Slug;
use Inside\Statistics\Facades\Stats;
use Inside\Statistics\Models\Statistic;
use Inside\Workflow\Facades\Proposal;
use Symfony\Component\HttpFoundation\File\MimeType\MimeTypeGuesser;
use Throwable;

class ContentTransformer implements Transformer
{
    protected array $transformed = [];

    protected array $filters = [];

    protected array $permissions = [];

    /**
     * Set request filters
     * @param  array  $filters
     * @return void
     */
    public function setFilters(array $filters): void
    {
        $this->filters = $filters;
    }

    /**
     * Add edit/delete permissions to the transformed data
     */
    protected function addPermissions(Content $model, ?array &$data): void
    {
        /** @var ?User $user */
        $user = Auth::user();

        if ($user instanceof User && ! is_null($model->uuid)) {
            $isSectionType = Schema::isSectionType(class_to_type($model));
            $data['permission'] = [
                'update' => $isSectionType || Permission::allowed('update', class_to_type($model), $model->uuid),
                'delete' => $isSectionType || Permission::allowed('delete', class_to_type($model), $model->uuid),
            ];
        }
    }

    public function transform(
        ?Model $model = null,
        array $fields = [],
        array &$data = [],
        bool $anonymized = false
    ): array {
        if (! $model || ! is_object($model)) {
            return $data;
        }

        // Special reaction case
        if ($model instanceof Reaction) {
            $modelType = 'reactions';
        } elseif ($model instanceof Statistic) {
            $modelType = 'statistics';
        } else {
            $modelType = class_to_type($model);

            if (! Schema::hasModel($modelType)) {
                warn_front('[transform] No model type ['.$modelType.']');

                return $data;
            }
        }

        try {
            $schema = Schema::getSchemaInformation($modelType);
        } catch (SchemaNotFoundException) {
            warn_front('[transform] No schema for type ['.$modelType.']');

            return $data;
        }

        if (isset($model->content_type) && empty($fields)) {
            $data['content_type'] = $model->content_type;

            if ($model->slug) {
                $data['slug'] = $model->slug;
            }
        }

        if (empty($fields)) {
            $fields = array_keys($schema['fields'] ?? ['uuid']);
            $fields = array_diff(
                $fields,
                [
                    'slug',
                    'count',
                    'total_count',
                    'total',
                    'parent_tree',
                    'content_type',
                    'tree',
                    'has_content',
                    'pid',
                    'authors',
                    'birthday_sent',
                    'seniority_anniversary_sent',
                    'can_read',
                    'revisionable',
                ]
            );

            // If no fields is asked, we want to get admin stuffs as well ( really useful in search result for example )
            if (! in_array('admin', $fields, true)) {
                array_push($fields, 'admin');
            }
        }

        foreach ($fields as $key => $field) {
            if ($field === null) {
                warn_front('[transform] empty field');
                continue;
            }

            $alias = $attribute = is_array($field) ? key($field) : $field;
            if (is_int($attribute) && is_string($key)) {
                $alias = $attribute = $key;
                $field = [$attribute => $field];
            }

            if (is_array($field) && isset($field[$attribute]['filters'])) {
                if (isset($field[$attribute]['filters']['alias'])) {
                    $alias = $field[$attribute]['filters']['alias'];
                }
            }

            // Check for sluggable
            if ($attribute == 'slug') {
                $sluggable = (bool) ($schema['options']['aliasable'] ?? false);
                if (! $sluggable) {
                    // warn_front('[transform] model type '.$modelType.' is not sluggagle, impossible to get slug attribute');
                    continue;
                }
            }
            $type = isset($schema['fields'][$attribute]) ? $schema['fields'][$attribute]['type'] : '';
            if (in_array($type, ['reference', 'comment'])) {
                $this->transformPivot($model, $field, $data[$alias]);
                continue;
            }

            // Apply reverse strategy
            if (Str::contains($attribute, ':')) {
                [$reversedAttribute, $reversed] = explode(':', $attribute, 2);
                if ($reversed == 'reverse') {
                    $type =
                        isset($schema['fields'][$reversedAttribute]) ? $schema['fields'][$reversedAttribute]['type']
                            : '';
                    $attribute = $reversedAttribute;
                }
            }

            if (! in_array($attribute, [
                'count',
                'total_count',
                'total',
                'section',
                'parent_tree',
                'tree',
                'has_content',
                'has_statistics',
                'admin',
                'workflow',
                'can_read',
                'revisionable',
            ])
            ) {
                // Normal field
                $data[$alias] = $this->changeValue($type, $model, $attribute, $anonymized);
            }

            // retrieve params
            if (is_array($field)) {
                $params = $field[$attribute];
            } else {
                // Front didn't set params
                $params = [];
            }

            if (is_string($params)) {
                $params = [$params];
            }

            // Special has section field
            if ($attribute == 'has_section') {
                // This is a special field with special fields/filters ( same way as pivots )
                $sectionFields = Arr::pull($params, 'fields', []);

                if (empty($sectionFields)) {
                    $sectionFields = ['content'];
                }
                $data['has_section'] = [];
                foreach ($sectionFields as $sectionFieldName) {
                    if (isset($schema['fields'][$sectionFieldName])
                        && $schema['fields'][$sectionFieldName]['type'] == 'section'
                    ) {
                        $data['has_section'][$sectionFieldName] =
                            $model->{'section_'.$sectionFieldName.'_count'} > 0;
                    }
                }
            } elseif ($attribute == 'has_statistics') {
                $data['has_statistics'] = Stats::hasStatistics(class_to_type($model));
            } elseif ($attribute === 'can_read') {
                $roles = Role::all();
                $data['can_read'] = [];

                foreach ($roles as $role) {
                    if (RoleService::can('read', $model, $role->id)) {
                        $data['can_read'][] = RoleService::getHumanName($role->name);
                    }
                }
            } elseif ($attribute === 'revisionable' && $model instanceof Content) {
                /** @var RevisionService $revisionService */
                $revisionService = App::make(RevisionService::class);
                $data['revisionable'] = $revisionService->isEnabled($model) && $model->revisions?->count() > 1;
            } elseif ($attribute == 'admin') {
                if ($model->isTranslatable) {
                    $data['admin']['translations'] = collect($model->getTranslationIds())->map(fn ($translation) => [
                        'uuid' => $translation['uuid'],
                        'langcode' => $translation['langcode'],
                        'content_type' => $model->content_type,
                    ])->toArray();
                }

                // Only contents is subject to permission
                if ($model instanceof Content) {
                    $this->addPermissions($model, $data['admin']);
                }
            } elseif ($attribute == 'translations') {
                $data['translations'] = $model->getTranslations(['uuid_host', 'uuid', 'title', 'langcode', 'slug']);
            }

            if ($type == 'section') {
                // This is a special field with special fields/filters ( same way as pivots )
                $sectionFields = Arr::pull($params, 'fields', []);

                $sections = $model->{Str::snake('section_'.$attribute)};

                $data[$alias] = null;

                if ($sections->count()) {
                    $data[$alias] = [];
                }

                foreach ($sections as $section) {
                    if (Package::has('inside-quiz') && $section->section_type === 'question') {
                        continue;
                    }

                    $transform = self::transform($section, $sectionFields);
                    if ($transform) {
                        $data[$alias][] = array_merge(
                            ['section_type' => $section->section_type],
                            $transform
                        );
                    }
                }
            }

            if ($attribute == 'tree' && method_exists($model, 'getChildrenIfExist')) {
                $children = $model->getChildrenIfExist($this->filters);

                if (! isset($data['children']) || ! is_array($data['children'])) {
                    $data['children'] = [];
                }

                foreach ($children as $child) {
                    $childrenData = [];
                    self::transform($child, $fields, $childrenData);
                    $data['children'][] = $childrenData;
                }
            }

            // Special parent tree ( used for breadcrumbs )
            if ($attribute == 'parent_tree' && $model instanceof Content) {
                $parentTreeFields = Arr::pull($params, 'fields', []);
                $parentTreeFilters = Arr::pull($params, 'filters', []);

                // get Parents as an array of parent by content_type ordered from leaf to tree &
                // filtered by filters['content_type']
                $parents = $model->getParentsIfExist($parentTreeFilters);

                // Transform each result
                foreach ($parents as $type) {
                    foreach ($type as $parent) {
                        $parentData = [];
                        self::transform(
                            $parent,
                            array_diff($parentTreeFields, ['parent_tree']),
                            $parentData
                        );
                        $data['parent_tree'][$parent->content_type][] = $parentData;
                    }
                }
            }

            $notificationsSent = [
                'birthday_sent' => 'birthday',
                'seniority_anniversary_sent' => 'seniorityAnniversary',
            ];

            foreach ($notificationsSent as $notificationAction => $notificationType) {
                if ($attribute == $notificationAction) {
                    $notificationTypeId = DB::table('inside_notifications_types')
                        ->where('action', $notificationType)
                        ->pluck('id')
                        ->first();
                    $currentYear = Carbon::now()->year;

                    if ($notificationTypeId) {
                        /** @var ?User $authUser */
                        $authUser = Auth::user();
                        if (! $authUser instanceof User) {
                            return $data;
                        }
                        $data[$notificationAction] = DB::table('inside_notifications')->where(
                            [
                                ['notification_type_id', '=', $notificationTypeId],
                                ['data', 'like', '%"from":"'.$authUser->uuid.'"%'],
                                ['user_uuid', '=', $model->uuid],
                                ['created_at', 'like', $currentYear.'-%'],
                            ]
                        )->exists();
                    }
                }
            }

            if ($attribute === 'workflow' && Package::has('inside-workflow')) {
                $data['workflow'] = $model->workflow;
                if (! empty($data['workflow'])) {
                    $data['workflow']['histories'] = Proposal::getHistories($data['workflow']['id']);
                }
            }
        }

        if (
            $model->content_type == 'advanced_form_submissions' &&
            data_get($data, 'advanced_forms.confidentiality', false) &&
            ! empty($data['authors'])
        ) {
            $data['authors'] = [];
        }

        return $data;
    }

    /**
     * @param  Model|null  $model
     * @param  mixed  $pivot
     * @param  mixed  $data
     * @return void
     */
    protected function transformPivot(Model $model = null, $pivot = [], &$data = []): void
    {
        if (! $model || ! is_object($model)) {
            return;
        }

        $attribute = $pivot;

        if ($model instanceof Reaction) {
            $modelType = 'reactions';
        } elseif ($model instanceof Statistic) {
            $modelType = 'statistics';
        } else {
            $modelType = class_to_type($model);
        }

        $fields = [];
        $filters = [];

        if (is_array($pivot)) {
            $attribute = key($pivot);
            if (is_string($attribute)) {
                $fields = $pivot[$attribute];

                $filters = Arr::pull(
                    $fields,
                    'filters',
                    [
                        'limit' => 0,
                        'offset' => null,
                        'sort' => null,
                        'reverse' => false,
                    ]
                );

                $fields = Arr::pull($fields, 'fields', $fields);
            } else {
                $filters = [
                    'limit' => 0,
                    'offset' => null,
                    'sort' => null,
                    'reverse' => false,
                ];
                $fields = $pivot;
            }
        }

        try {
            $schema = Schema::getSchemaInformation($modelType);
        } catch (SchemaNotFoundException $exception) {
            warn_front('[transformPivot] '.$exception->getMessage());

            return;
        }
        if (! is_array($schema)) {
            return;
        }

        $cardinality = $schema['fields'][$attribute]['options']['cardinality'] ?? -1;

        if ($cardinality == 1) {
            $data = null;
        } else {
            $data = ['data' => []];
        }

        // Retro on reverse relation
        // If attribute contains a dot it is necessarily a reversed attribute
        /** @var string $attribute */
        $reversed = Str::contains($attribute, '.');

        if (Str::contains($attribute, ':')) {
            [$attribute, $reversed] = explode(':', $attribute, 2);
            $reversed = ($reversed == 'reverse');
        }

        $relation = $attribute;
        if (isset($filters['reverse'])
            && $filters['reverse']
            && array_key_exists('fields', $schema)
            && is_array($schema['fields'])
        ) {
            $reversed = true;
            // If self reference, we must prefix by type
            if ($schema['fields'][$attribute]['target'][0] == $modelType) {
                if ($relation === $modelType.'.'.$modelType) {
                    $relation = $modelType;
                }

                $relation = $modelType.'_'.$relation;
            }
        }

        $target = $relation;
        $relation = Str::camel($reversed ? 'reverse_'.str_replace('.', '_', $relation) : $relation);

        if (collect($fields)->search('unique_views') || collect($fields)->search('total_views')) {
            $query = $model->{$relation}();

            if (in_array('unique_views', $fields)) {
                Stats::addUniqueViewsToQuery($target, $query);
            }

            if (in_array('total_views', $fields)) {
                Stats::addTotalViewsToQuery($target, $query);
            }

            $pivots = $query->get();
        } else {
            $pivots = $model->{$relation};
        }

        if (is_null($pivots)) {
            $data = null;

            return;
        }

        if (is_string($pivots)) {
            $targetTypes = $schema['fields'][$attribute]['options']['target'] ?? [];
            if (! array_key_exists('fields', $schema)
                || ! is_array($schema['fields'])
                || $schema['fields'][$attribute]['type'] != 'reference'
                || $cardinality != 1
                || ! in_array('users', $targetTypes)) {
                return;
            }

            $replacers = config('contents.anonymous_users.fields', []);
            $data = [
                'firstname' => trans($replacers['firstname']),
                'lastname' => trans($replacers['lastname']),
            ];
        }

        if ($model->contentType === 'users' && ! empty($this->filters['langcode'])) {
            $pivots->transform(fn ($pivot) => $pivot->getTranslationIfExist($this->filters['langcode']));
        }

        if (isset($filters['status']) || isset($filters['status:eq'])) {
            $status = $filters['status'] ?? $filters['status:eq'];
            $pivots = $pivots->where('status', $status);
        }

        if (get_class($pivots) !== Collection::class) {
            $data = [];
            $protectedFields = [
                'slug',
                'count',
                'total_count',
                'total',
                'parent_tree',
                'content_type',
                'tree',
                'has_content',
                'pid',
                'admin',
                'authors',
                'reactions',
                'can_read',
                'revisionable',
                'statistics',
            ];

            if (empty($fields) && in_array($attribute, ['users', 'authors'])) {
                try {
                    $pivotSchema = Schema::getSchemaInformation(class_to_type($pivots));
                    if (! is_array($pivotSchema)
                        || ! array_key_exists('fields', $pivotSchema)
                        || ! is_array($pivotSchema['fields'])) {
                        throw new Exception(__('Invalide schema for [:type]', ['type' => class_to_type($pivots)]));
                    }
                    $fields = array_diff(
                        array_keys($pivotSchema['fields']),
                        $protectedFields
                    );
                } catch (Exception $exception) {
                    warn_front('[transformPivot] '.$exception->getMessage());
                    Log::error('[ContentTransformer::transformPivot] failed => '.$exception->getMessage());
                }
            }

            // if content_types is defined in configs, we anonymize users.
            $this->transform(
                $pivots,
                $fields,
                $data,
                in_array($modelType, config('contents.anonymous_users.content_types', []))
            );

            return;
        }

        $protectedAttributes = ['reactions', 'statistics'];

        if (! in_array($attribute, $protectedAttributes)) {
            if (in_array('count', $fields)) {
                $data['count'] = $pivots->count();
            }

            if (in_array('total_count', $fields)) {
                $data['total_count'] = $this->countPivotChildren($attribute, $pivots->where('pid', null)->toArray());
            }

            if (in_array('total', $fields)) {
                $query = call_user_func(
                    'Inside\Content\Models\Contents\\'.Str::studly($attribute).'::withoutGlobalScopes'
                );

                if (isset($filters['langcode'])) {
                    $query->where('langcode', $filters['langcode']);
                }

                $data['total'] = $query->count();
            }
        }

        $targetTypes = $schema['fields'][$attribute]['target'] ?? [];
        if (! is_array($targetTypes)) {
            $targetTypes = [$targetTypes];
        }
        try {
            $isTranslatable = ! in_array('users', $targetTypes) && $model->isTranslatable;
        } catch (Exception $e) {
            $isTranslatable = false;
        }

        if (! isset($filters['sort']) && isset($this->filters['sort'])) {
            // Use main content filters if available
            $filters['sort'] = $this->filters['sort'];
        }
        $genericFilters = Arr::except($filters, ['sort', 'langcode', 'offset', 'reverse', 'limit', 'alias']);

        // Filter on it
        $pivots = $pivots->when(
            isset($filters['offset']) && ! empty($filters['offset']),
            function ($pivots) use ($filters) {
                return $pivots->offset($filters['offset']);
            }
        )->when(
            isset($filters['langcode']),
            function ($pivots) use ($filters) {
                return $pivots->where('langcode', $filters['langcode']);
            }
        )->when(
            $isTranslatable && ! isset($filters['langcode']) && isset($this->filters['langcode']),
            function ($pivots) {
                return $pivots->where('langcode', $this->filters['langcode']);
            }
        )->when(
            isset($filters['sort']),
            function ($pivots) use ($filters) {
                $sort = $filters['sort'];

                if (is_array($sort)) {
                    foreach (array_reverse($sort) as $toSort) {
                        [$column, $dir] = [$toSort, 'asc'];

                        if (false !== strpos($toSort, ':')) {
                            [$column, $dir] = explode(':', $toSort, 2);
                        }
                        if ($dir == 'asc') {
                            $pivots = $pivots->sortBy($column);
                        } else {
                            $pivots = $pivots->sortByDesc($column);
                        }
                    }

                    return $pivots;
                }

                [$column, $dir] = [$sort, 'asc'];

                if (Str::contains($sort, ':')) {
                    // Get column to sort & direction
                    [$column, $dir] = explode(':', $sort, 2);
                }

                if ($column == 'random') {
                    return $pivots->shuffle();
                }

                // Don't sort on nonexistent column
                if ($pivots->isEmpty()) {
                    return $pivots; // Nothing to sort
                }
                if (! Schema::hasSchemaInformation(class_to_type($pivots->first()))
                    || ! array_key_exists(
                        $column,
                        Schema::getSchemaInformation(class_to_type($pivots->first()))['fields'] ?? []
                    )
                ) {
                    return $pivots;
                }

                if ($dir == 'asc') {
                    return $pivots->sortBy($column);
                }

                return $pivots->sortByDesc($column);
            }
        )->when(
            ! empty($genericFilters),
            function ($pivots) use ($genericFilters, $targetTypes) {
                // Manage filters
                return $this->applyFiltersOnPivot(
                    Arr::first($targetTypes),
                    $pivots,
                    $genericFilters
                );
            }
        )->when(
            isset($filters['limit']) && $filters['limit'] > 0,
            function ($pivots) use ($filters) {
                return $pivots->take($filters['limit']);
            }
        );

        if (empty($fields) && $pivots->count()) {
            $pivot = $pivots->first();
            $pivotType = class_to_type($pivot);

            try {
                $schema = Schema::getSchemaInformation($pivotType);
            } catch (SchemaNotFoundException) {
                $schema = [];
            }
            // All except reference (avoid infinite loop)
            $fields = array_keys(
                array_filter(
                    $schema['fields'] ?? [],
                    function ($pivotOptions) {
                        return ! in_array($pivotOptions['type'], ['reference', 'comment', 'reactions', 'statistics']);
                    },
                    ARRAY_FILTER_USE_BOTH
                )
            );
            $fields = array_diff(
                $fields,
                ['slug', 'count', 'content_type', 'parent_tree', 'tree', 'has_content', 'pid', 'admin', 'authors']
            );
        }

        foreach ($pivots as $pivot) {
            // TODO: find a better way to filter archived contents in pivots
            if (isset($filters['archived']) && Package::has('inside-archive')) {
                $archived =
                    DB::table('inside_archives')->where('type', get_class($pivot))->where('uuid', $pivot->uuid)->where(
                        'date',
                        '<=',
                        date('Y-m-d H:i:s')
                    )->exists();

                if ((bool) $filters['archived'] !== $archived) {
                    continue;
                }
            }

            // TODO : rework reaction
            if (in_array($attribute, $protectedAttributes) && ! is_array($pivot)) {
                $tmpData = [];
                $this->transform($pivot, $fields, $tmpData);
                $data['data'][$pivot->type][] = $tmpData;
                continue;
            }

            if ($reversed) {
                $reversedCardinality = $filters['reversed_cardinality'] ?? -1;
                if ($reversedCardinality == 1) {
                    if (is_null($data)) {
                        $data = [];
                    }
                    $this->transform($pivot, $fields, $data);
                } else {
                    $tmpData = [];
                    $this->transform($pivot, $fields, $tmpData);
                    $data['data'][] = $tmpData;
                }
            } else {
                if ($cardinality == 1) {
                    if (is_null($data)) {
                        $data = [];
                    }
                    $this->transform($pivot, $fields, $data);
                } else {
                    $tmpData = [];
                    $this->transform($pivot, $fields, $tmpData);
                    $data['data'][] = $tmpData;
                }
            }
        }

        if (in_array('count', $fields) && $cardinality != 1) {
            if (in_array($attribute, $protectedAttributes)) { // used for stats and reactions
                foreach ($data['data'] as $type => $reaction) {
                    $data['count'][$type] = is_array($data['data'][$type]) ? count($data['data'][$type]) : 0;
                }
            } else { // used for normal references
                $data['count'] = count($data['data']);
            }
        }
    }

    /**
     * Transform collection of contents.
     *
     * @param  Collection  $collection
     * @param  array  $fields
     *
     * @return array
     */
    public function transformCollection(\Illuminate\Support\Collection $collection, array $fields = []): array
    {
        $this->transformed['data'] = [];

        if (in_array('count', $fields)) {
            $this->transformed['count'] = count($collection);
        }

        foreach ($collection as $model) {
            $this->transformed['data'][] = self::transform($model, $fields);
        }

        if (in_array('total_count', $fields)) {
            // Get a global count with children etc... if exists
            if (in_array('tree', $fields)) {
                // Add children to the counter
                $this->transformed['total_count'] = $this->countWithChildren($this->transformed['data']);
            } else {
                $this->transformed['total_count'] = count($collection);
            }
        }

        return $this->transformed;
    }

    /**
     * Apply value change according to it's type
     *
     * @param  string|null  $type
     * @param  Model  $model  Section or content
     * @param  string  $fieldName
     * @param  bool  $anonymized
     *
     * @return mixed
     */
    protected function changeValue(?string $type, Model $model, string $fieldName, bool $anonymized = false): mixed
    {
        $value = $model->{$fieldName};
        if ($value === null) {
            return null;
        }

        if ($anonymized && class_to_type($model) === 'users' && ! $model->status) {
            $replacers = config('contents.anonymous_users.fields', []);
            if (isset($replacers[$fieldName])) {
                return trans($replacers[$fieldName]);
            }
        }

        switch ($type) {
            case 'file':
                if (empty($value)) {
                    return null;
                }

                $realpath = Storage::disk('local')->path($value);
                $value = [
                    'mimetype' => '',
                    'path' => protected_file_url($model, $fieldName),
                    'size' => '',
                    'basename' => '',
                ];

                if (is_string($realpath) && ! empty($realpath) && File::exists($realpath)) {
                    $value['basename'] = basename($realpath);
                    $guesser = MimeTypeGuesser::getInstance();
                    $value['mimetype'] = $guesser->guess($realpath);
                    $value['size'] = filesize($realpath);

                    $width = $height = null;
                    try {
                        $imageSize = getimagesize($realpath);

                        if ($imageSize) {
                            [$width, $height] = $imageSize;
                        }
                    } catch (Exception) {
                        Log::warning('[ContentTransformer::changeValue] failed to get image size');
                    }

                    if ($width && $height) {
                        $value['width'] = $width;
                        $value['height'] = $height;
                    }
                }
                break;
            case 'image':
                if ($model->content_type === 'users' && $fieldName === 'image') {
                    $value = [
                        'main' => 'avatar/users/'.$model->uuid,
                        'avatar' => 'avatar/users/'.$model->uuid,
                    ];

                    break;
                }

                if (empty($value)) {
                    return null;
                }

                $original = $value;
                $value = protected_file_url($model, $fieldName);
                $realpath = Storage::disk('local')->path($original);
                // we add by default a webp file for the main image too so the front can use an optimized image
                // we add a little trick because it's not a real existing style, just the main image as webp
                /** @var array $value */
                $value = [
                    'main' => $value,
                    'webp' => [
                        'main' => protected_file_url($model, $fieldName, true, 'main-webp', true),
                    ],
                ];

                // Add styled image
                $options = Schema::getFieldOptions(class_to_type($model), $fieldName);
                if (isset($options['image_styles']) && ! empty($options['image_styles'])) {
                    foreach ($options['image_styles'] as $style) {
                        $value[Str::slug($style)] = protected_file_url($model, $fieldName, true, $style);
                        $value['webp'][Str::slug($style)] = protected_file_url(
                            $model,
                            $fieldName,
                            true,
                            $style,
                            true
                        );
                    }
                }

                break;
            case 'link':
                $value = $this->transformLink($value);
                break;
            case 'timestamp':
                if (is_string($value) && \DateTime::createFromFormat('Y-m-d G:i:s', $value) !== false) {
                    $value = strtotime($value);
                }
                break;
            case 'text':
            case 'textarea':
                // No secure here, front uses v-text
                break;
            case 'comment':
            case 'wysiwyg':
                // Purify!

                if (($type === 'textarea' && class_to_type($model) === 'forms' && $fieldName === 'inputs') || class_to_type($model) === 'advanced_table') {
                    // We don't want to purify forms inputs field, it's a JSON string / We don't want to purify the content of advanced table
                    break;
                }

                $textType = $type;
                if ($type === 'wysiwyg' && class_to_type($model) === 'comments') {
                    $textType = 'comment';
                }
                $value = iClean($value, (string) $textType);
                break;
        }

        return $value;
    }

    /**
     * @param  string  $value
     * @return array
     * @deprecated
     */
    protected function transformLink(string $value): array
    {
        $url = $value;
        $info = parse_url($url);
        $external = ($info && isset($info['host']) && isset($info['scheme']) && $info['scheme'].'://'.$info['host'] !== env('APP_URL'));
        $fragment = '';

        if ($info && isset($info['fragment'])) {
            if (($fragment = trim($info['fragment'])) != '') {
                $fragment = '#'.$fragment;
            }
        }

        $value = [
            'url' => ! $external && $info && isset($info['path']) ? ($info['path'].$fragment) : $url,
            'uuid' => '',
            'type' => '',
            'external' => $external,
        ];
        $slug = Slug::where('slug', '=', $url)->first();

        // TODO check if this is still used by the front
        if ($slug) {
            $value['type'] = $slug->type;
            $value['uuid'] = $slug->uuid;
        } else {
            if ($url == 'home') {
                // Special case for home
                $value['url'] = '';
            }

            if ($info && isset($info['scheme']) && in_array($info['scheme'], ['uuid', 'mailto'])) {
                switch ($info['scheme']) {
                    case 'uuid':
                        $url = explode(':', $value['url']);
                        $value['url'] = $url[1];
                        $value['type'] = $url[0];
                        $value['uuid'] = $url[1];
                        break;
                    case 'mailto':
                        $value['url'] = $url;
                        $value['type'] = 'mailto';
                        break;
                }
            }
        }

        return $value;
    }

    protected function countWithChildren(array $items): int
    {
        $result = 0;

        foreach ($items as $item) {
            if (isset($item['children'])) {
                $result += ($this->countWithChildren($item['children']) + 1);
                continue;
            }

            $result++;
        }

        return $result;
    }

    protected function countPivotChildren(mixed $model, array $pivots): int
    {
        $count = 0;
        foreach ($pivots as $pivot) {
            // Get children
            $query = call_user_func('Inside\Content\Models\Contents\\'.Str::studly($model).'::query');
            $children = $query->where('pid', $pivot->uuid)->get();
            $count += ($this->countPivotChildren($model, $children) + 1);
        }

        return $count;
    }

    /**
     * Apply $filters on $pivot of $type
     *
     * @param  string  $type
     * @param  Collection  $pivot
     * @param  array  $filters
     * @return Collection
     */
    protected function applyFiltersOnPivot(string $type, Collection $pivot, array $filters = []): Collection
    {
        if (! Schema::isModel($type)) {
            return $pivot;
        }
        if (empty($filters)) {
            return $pivot;
        }
        foreach ($filters as $filter => $value) {
            $operator = null;
            if (Str::contains($filter, ':')) {
                [$filter, $operator] = explode(':', $filter, 2);
                $operator = ContentHelper::getOperatorFromString($operator);
            }

            if (in_array($filter, ['archived'])) {
                continue;
            }

            if ($operator === null) {
                $operator = '=';
            }

            if ($filter === '$or') {
                if (! is_array($value) || empty($value)) {
                    continue;
                }
                $groups = [];
                foreach ($value as $k => $g) {
                    $groups[] = is_numeric($k) ? $g : [$k => $g];
                }

                $pivot = $pivot->filter(function ($item) use ($groups, $type) {
                    foreach ($groups as $group) {
                        $subset = $this->applyFiltersOnPivot(
                            $type,
                            new \Illuminate\Database\Eloquent\Collection([$item]),
                            $group
                        );
                        if ($subset->isNotEmpty()) {
                            return true;
                        }
                    }

                    return false;
                });

                continue;
            }
            try {
                $fieldOption = Schema::getFieldOptions($type, $filter);
                $fieldType = $fieldOption['type'];
            } catch (FieldSchemaNotFoundException $exception) {
                // /!\ published_at | created_at | updated_at are not registered in table inside_fields but still exist /!\
                if (! in_array($filter, ['published_at', 'created_at', 'updated_at'])) {
                    Log::error('[ContentTransformer::applyFiltersOnPivot] '.$exception->getMessage());
                    continue;
                }

                $fieldType = 'timestamp';
            }
            if ($fieldType == 'reference' && $pivot->isNotEmpty()) {
                // Filter pivot on reference
                return $pivot->filter(function ($item) use ($filter, $value) {
                    $referencedContent = $item->{$filter};
                    if (! $referencedContent) {
                        return false;
                    }
                    if ($referencedContent instanceof \Illuminate\Support\Collection) {
                        foreach ($referencedContent as $content) {
                            if ($this->checkPivotFilter($content, $value)) {
                                return true;
                            }
                        }

                        return false;
                    }

                    return $this->checkPivotFilter($referencedContent, $value);
                });
            }
            if ($fieldType == 'timestamp') {
                // Special filter on date ( front always filter on ISO date because
                // database query filter requires it, but on collection, date are always cast
                // to timestamp, so we need to convert it as well
                // Manage special value now() for date comparison
                if (Str::lower($value) == 'now()') {
                    $value = now()->timestamp;
                } else {
                    try {
                        $value = get_date($value)?->timestamp;
                    } catch (Throwable|Exception $exception) {
                        continue;
                    }
                }
            }
            try {
                $pivot = $pivot->where($filter, $operator, $value);
            } catch (Exception $exception) {
                Log::error('[ContentTransformer::applyFiltersOnPivot] failed to filter on pivot -> '.
                    $exception->getMessage());
            }
        }

        return $pivot;
    }

    /**
     * @param  Content  $content
     * @param  array  $filters
     * @return bool
     */
    protected function checkPivotFilter(Content $content, array $filters): bool
    {
        foreach ($filters as $filter => $wantedValue) {
            // Front operator format is unknown here
            $filter = Str::before($filter, ':');
            if (! isset($content->{$filter})) {
                return false;
            }

            if ($content->{$filter} instanceof \Illuminate\Support\Collection) {
                if (! $content->{$filter}->contains($wantedValue)) {
                    return false;
                }
            } else {
                if ($content->{$filter} !== $wantedValue) {
                    return false;
                }
            }
        }

        return true;
    }
}
