<?php

namespace Inside\Content\Support;

use Illuminate\Contracts\Redis\LimiterTimeoutException;
use Illuminate\Support\Facades\Redis;
use Illuminate\Support\Str;
use Inside\Content\Exceptions\ShortUrlException;
use Inside\Content\Models\ShortUrl;

class ShortUrlManager
{
    protected ?string $destinationUrl;

    protected ?string $urlKey;

    protected bool $forwardQueryParams = false;

    protected string $prefix = 's';

    public function __construct(
        protected UniqueKeyGenerator $uniqueKeyGenerator
    ) {
    }

    /**
     * @throws ShortUrlException
     */
    public function destination(?string $url): self
    {
        if (is_null($url)) {
            throw new ShortUrlException('url is empty');
        }
        $url = str_replace(config('app.url'), '', $url);
        if (Str::startsWith($url, 'http://') || Str::startsWith($url, 'https://')) {
            throw new ShortUrlException('Invalid url ('.$url.')');
        }

        if (Str::startsWith($url, '/')) {
            $url = Str::substr($url, 1);
        }

        $this->destinationUrl = $url;

        return $this;
    }

    /**
     * @throws LimiterTimeoutException
     */
    public function make(): ShortUrl
    {
        return Redis::funnel('short_url_unique_key_generator_lock')
            ->limit(8)
            ->block(10)
            ->then(function () {
                if (is_null($this->destinationUrl)) {
                    throw new ShortUrlException('missing destination url');
                }

                if (! $this->urlKey) {
                    $this->urlKey = $this->uniqueKeyGenerator->generate();
                }

                $this->checkUrlExists();
                $url = $this->storeShortUrl();
                $this->reset();

                return $url;
            });
    }

    protected function storeShortUrl(): ShortUrl
    {
        return ShortUrl::create([
            'destination_url' => $this->destinationUrl,
            'default_short_url' => config('app.url').'/'.$this->getPrefix().'/'.$this->urlKey,
            'url_key' => $this->urlKey,
            'forward_query_params' => $this->forwardQueryParams,
            'created_at' => now(),
            'updated_at' => now(),
        ]);
    }

    /**
     * @return $this
     */
    public function reset(): self
    {
        $this->urlKey = null;
        $this->prefix = 's';
        $this->forwardQueryParams = false;

        return $this;
    }

    public function getDestinationUrl(): ?string
    {
        return $this->destinationUrl;
    }

    public function getUniqueKeyGenerator(): UniqueKeyGenerator
    {
        return $this->uniqueKeyGenerator;
    }

    public function setUniqueKeyGenerator(UniqueKeyGenerator $uniqueKeyGenerator): self
    {
        $this->uniqueKeyGenerator = $uniqueKeyGenerator;

        return $this;
    }

    public function getUrlKey(): ?string
    {
        return $this->urlKey;
    }

    public function setUrlKey(string $urlKey): self
    {
        $this->urlKey = urlencode($urlKey);

        return $this;
    }

    public function isForwardQueryParams(): bool
    {
        return $this->forwardQueryParams;
    }

    public function setForwardQueryParams(bool $forwardQueryParams): self
    {
        $this->forwardQueryParams = $forwardQueryParams;

        return $this;
    }

    public function getPrefix(): string
    {
        return $this->prefix;
    }

    public function setPrefix(string $prefix): self
    {
        $this->prefix = $prefix;

        return $this;
    }

    /**
     * @throws ShortUrlException
     */
    protected function checkUrlExists(): void
    {
        if (! $this->urlKey) {
            return;
        }
        $url = ShortUrl::findByKey($this->urlKey);
        if (! is_null($url)) { // already exists
            throw new ShortUrlException('Key already exists');
        }
    }
}
