<?php

declare(strict_types=1);

namespace Inside\Content\Services\WelcomeToTheJungle;

use GuzzleHttp\Client;
use Illuminate\Support\Facades\Log;
use Inside\Content\Contracts\WelcomeToTheJungleApiClient as WelcomeToTheJungleApiClientContract;
use Inside\IZIS\Contracts\JobOffersApiClient;
use Symfony\Component\HttpFoundation\Response;

final class WelcomeToTheJungleApiClient implements WelcomeToTheJungleApiClientContract
{
    private const API_URL = 'https://www.welcomekit.co/api/v1/embed';

    private Client $client;

    public function __construct()
    {
        $this->client = new Client();
    }

    public function getJobOffers(string $organizationReference, ?string $langcode = null): array
    {
        $langcode ??= config('app.locale');

        $response = $this->client->get(self::API_URL, [
            'query' => ['organization_reference' => $organizationReference],
        ]);

        if ($response->getStatusCode() !== Response::HTTP_OK) {
            Log::error('[WelcomeToTheJungleApiClient::getJobsOffers] => invalid response', [
                'organization_reference' => $organizationReference,
                'statusCode' => $response->getStatusCode(),
                'body' => $response->getBody()->getContents(),
            ]);

            return [];
        }

        $contents = json_decode($response->getBody()->getContents(), true);

        return collect($contents['jobs'] ?? [])
            ->map(fn (array $rawJobOffer) => $this->transformRawJobOffer($rawJobOffer, $organizationReference, $langcode))
            ->all();
    }

    public function transformRawJobOffer(array $rawJobOffer, string $organizationReference, string $langcode): array
    {
        $websiteReference = collect($rawJobOffer['cms_sites_references'] ?? [])->first();
        $website = collect($rawJobOffer['websites_urls'] ?? [])
            ->first(fn (array $website) => $websiteReference === null || $website['website_reference'] === $websiteReference);

        return [
            'external_reference' => trim($rawJobOffer['reference']),
            'title' => trim($rawJobOffer['name']),
            'location_title' => trim($rawJobOffer['office']['name']),
            'contract_title' => trim($rawJobOffer['contract_type'][$langcode]),
            'external_link' => $website['url'] ?? null,
            'description' => trim($rawJobOffer['description']),
            'profile' => trim($rawJobOffer['profile']),
            'recruitment_process' => trim($rawJobOffer['recruitment_process']),
            'organization_reference' => $organizationReference,
        ];
    }
}
