<?php

declare(strict_types=1);

namespace Inside\Content\Services\Serializer;

use Illuminate\Support\Str;
use Inside\Content\Contracts\SchemaService as SchemaServiceContract;
use Inside\Content\Contracts\Serializer;
use Inside\Content\Models\Content;
use Inside\Content\Models\Section;

final class SerializerService implements Serializer
{
    public function __construct(
        protected SchemaServiceContract $schemaService
    ) {
    }

    public function serialize(Content $content): string
    {
        $this->loadAllRelations($content);

        return serialize($content);
    }

    public function unserialize(string $data): ?Content
    {
        $content = unserialize($data);
        if (! $content instanceof Content) {
            return null;
        }

        return $content;
    }

    protected function loadAllRelations(Content|Section $content): void
    {
        $type = class_to_type($content);
        $content->load($this->schemaService->getRelations($type));
        if ($content instanceof Content) {
            $content->load('images');
        }
        if ($this->schemaService->hasFieldOfType($type, 'section')) {
            foreach ($this->schemaService->getFieldListingOfType($type, 'section') as $fieldName) {
                $fieldInformation = $this->schemaService->getFieldOptions($type, $fieldName);
                $content->load(array_map(
                    fn ($section) => Str::camel($section.'_section_item'),
                    $fieldInformation['target'] ?? []
                ));
                // Load relations for each section
                /** @var array<int, Section> $sections */
                $sections = $content->{Str::snake('section_'.$fieldName)};
                foreach ($sections as $section) {
                    $this->loadAllRelations($section);
                }
            }
        }
    }
}
