<?php

declare(strict_types=1);

namespace Inside\Content\Services\Schema;

use BadMethodCallException;
use Closure;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Database\QueryException;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Content\Contracts\DrupalSchema as DrupalSchemaContract;
use Inside\Content\Contracts\SchemaService as SchemaServiceContract;
use Inside\Content\Exceptions\FieldSchemaNotFoundException;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Exceptions\SchemaNotFoundException;
use Inside\Content\Models\Field;
use Inside\Content\Models\Model;
use Inside\Facades\Package;
use Inside\Reaction\Models\Reaction;
use Inside\Statistics\Models\Statistic;
use Inside\Support\Collection;
use stdClass;

/**
 * An easy service to access schema information
 * @method array getSearchableContentTypes() get all searchable content type
 * @method array getGlobalSearchableContentTypes() get all global searchable content type
 * @method array getPermissibleContentTypes() get all permissible content type
 * @method array getAliasableContentTypes() get all aliasable content type
 * @method array getCategorizableContentTypes() get all categorizable content type
 */
final class SchemaService implements SchemaServiceContract
{
    /**
     * Stored service informations
     */
    protected ?array $models = null;

    /**
     * Store fields for faster access
     */
    protected array $fields = [];

    /**
     * Store schema ( used by front )
     */
    protected ?array $schema = null;

    public const CONTENT_MODEL_TYPE = 1;

    public const SECTION_MODEL_TYPE = 2;

    /**
     * @var string[]
     */
    protected array $fieldAliases = [
        // author field is bugged  so fix there => saved as field name ( that should be called author_uuid )
        'author' => 'authors',
    ];

    /**
     * @var string[]
     */
    protected array $systemFields = [
        'uuid',
        'title',
        'status',
        'pid',
        'update_author',
        'langcode',
        'author',
        'created_at',
        'updated_at',
        'published_at',
    ];

    public function models(): ?array
    {
        if (! $this->models) {
            $this->models = [];

            // Make a load of fields and models options
            try {
                $fields =
                    DB::table('inside_models as m')
                        ->leftJoin('inside_fields as f', 'm.id', '=', 'f.model_id')
                        ->get(
                            [
                                'm.id as model_id',
                                'm.class',
                                'm.options as model_options',
                                'f.id as field_id',
                                'f.name as field_name',
                                'f.type',
                                'f.options as field_options',
                                'f.displayed',
                            ]
                        );
            } catch (QueryException $e) {
                Log::error('[SchemaService::models] failed to load fields & models => {'.$e->getMessage().'}');

                return $this->models;
            }

            foreach ($fields as $field) {
                /** @var Model $field */
                $modelClass = $field->class;
                $modelName = class_to_type($modelClass);
                if (! isset($this->models[$modelName])) {
                    $this->models[$modelName] = [
                        'id' => $field->model_id,
                        'class' => $field->class,
                        'type' => Str::startsWith($field->class, 'Inside\\Content\\Models\\Contents\\')
                            ? self::CONTENT_MODEL_TYPE : self::SECTION_MODEL_TYPE,
                        'fields' => [],
                        'options' => json_decode($field->model_options, true) ?? [],
                    ];
                }

                if (empty($field->field_name)) {
                    continue;
                }

                $this->models[$modelName]['fields'][$field->field_name] = $this->setFieldOptions($modelName, $field);
            }
        }

        return $this->models;
    }

    public function schema(): array
    {
        if (is_null($this->schema)) {
            //TODO: rework this
            $this->schema = app(DrupalSchemaContract::class)();
        }

        return $this->schema;
    }

    public function getRelations(string $type): array
    {
        return array_map(fn ($relation) => Str::camel($relation), $this->getFieldListingOfType($type, ['reference']));
    }

    public function getRelationsWithOption(string $type): array
    {
        return collect($this->getFieldListingOfType($type, 'reference'))->mapWithKeys(fn (string $field) => [
            Str::camel($field) => $this->getFieldOptions($type, $field),
        ])->toArray();
    }

    public function getFieldNamesThatReferenceType(string $needleType): array
    {
        $fieldNames = [];
        foreach ($this->getContentTypes() as $type) {
            foreach ($this->getFieldListingOfType($type, 'reference') as $fieldName) {
                $fieldOptions = $this->getFieldOptions($type, $fieldName);
                if (in_array($needleType, $fieldOptions['target'] ?? [])) {
                    if (! isset($fieldNames[$type])) {
                        $fieldNames[$type] = [];
                    }
                    $fieldNames[$type][] = $fieldName;
                }
            }
        }

        return $fieldNames;
    }

    public function hasSchemaInformation(string $type): bool
    {
        $protected = ['reactions', 'statistics'];

        /** @var array $schema */
        $schema = $this->schema();
        $models = $this->models();
        if (! $models) {
            return false;
        }

        return in_array($type, $protected)
            || ($this->isModel($type)
                && isset(
                    $schema[$models[$type]['class']]
                ));
    }

    public function getSchemaInformation(string $type): ?array
    {
        if (! $this->hasSchemaInformation($type)) {
            throw SchemaNotFoundException::named($type);
        }

        /** @var array $schema */
        $schema = $this->schema();

        if ($type == 'reactions') {
            // TODO: reactions should not be managed this way : refactor
            return $schema[Reaction::class];
        } elseif ($type == 'statistics') {
            // TODO: statistics should not be managed this way : refactor
            return $schema[Statistic::class];
        }

        $models = $this->models();
        if (! $models) {
            return null;
        }

        return $schema[$models[$type]['class']];
    }

    public function getTypes(?Closure $callback = null): array
    {
        $models = $this->models();
        if (! $models) {
            return [];
        }

        return array_values(
            Arr::sort(
                array_keys(
                    array_filter(
                        $models,
                        function ($model) use ($callback) {
                            if ($callback !== null && ! $callback($model)) {
                                return false;
                            }

                            return true;
                        }
                    )
                )
            )
        );
    }

    public function getContentTypes(Closure $callback = null): array
    {
        $models = $this->models();
        if (! $models) {
            return [];
        }

        return array_values(
            Arr::sort(
                array_keys(
                    array_filter(
                        $models,
                        function ($model) use ($callback) {
                            if ($callback !== null && ! $callback($model)) {
                                return false;
                            }

                            return $model['type'] === self::CONTENT_MODEL_TYPE;
                        }
                    )
                )
            )
        );
    }

    public function getSectionTypes(Closure $callback = null): array
    {
        $models = $this->models();
        if (! $models) {
            return [];
        }

        return array_values(
            Arr::sort(
                array_keys(
                    array_filter(
                        $models,
                        function ($model) use ($callback) {
                            if ($callback !== null && ! $callback($model)) {
                                return false;
                            }

                            return $model['type'] === self::SECTION_MODEL_TYPE;
                        }
                    )
                )
            )
        );
    }

    public function isModel(string $type): bool
    {
        return isset($this->models()[$type]);
    }

    public function isContentType(string $type): bool
    {
        $models = $this->models();
        if (! $models) {
            return false;
        }

        return isset($models[$type]) && $models[$type]['type'] === self::CONTENT_MODEL_TYPE;
    }

    public function isSectionType(string $type): bool
    {
        $models = $this->models();
        if (! $models) {
            return false;
        }

        return isset($models[$type]) && $models[$type]['type'] === self::SECTION_MODEL_TYPE;
    }

    public function getModelInformation(string $modelName): stdClass
    {
        if (! $this->hasModel($modelName)) {
            throw ModelSchemaNotFoundException::named($modelName);
        }

        if (! $this->models) {
            throw new ModelNotFoundException();
        }

        return (object) $this->models[$modelName];
    }

    public function getModelOptions(string $modelName): array
    {
        if (! $this->hasModel($modelName)) {
            throw ModelSchemaNotFoundException::named($modelName);
        }

        if (! $this->models) {
            throw new ModelNotFoundException();
        }

        return $this->models[$modelName]['options'];
    }

    public function setModelOption(string $modelName, string $optionName, mixed $value): void
    {
        if (! $this->hasModel($modelName)) {
            throw ModelSchemaNotFoundException::named($modelName);
        }

        $this->models[$modelName]['options'][$optionName] = $value;

        Model::where('class', type_to_class($modelName))->update(
            ['options' => json_encode($this->models[$modelName]['options'])]
        );
    }

    public function updateFieldOption(string $modelName, string $fieldName, string $optionName, mixed $value): void
    {
        if (! $this->hasField($modelName, $fieldName)) {
            throw FieldSchemaNotFoundException::named($modelName, $fieldName);
        }

        $fieldOptions = $this->getFieldOptions($modelName, $fieldName);

        if (! isset($fieldOptions[$optionName])) {
            Log::error('[SchemaService::updateFieldOption] failed to update field option / option not found => {'.$optionName.' => '.$value.' }');

            return;
        }

        $fieldOptions[$optionName] = $value;
        $fieldName = $this->getOriginalFieldName($fieldName);
        /** @phpstan-ignore-next-line */
        Field::where(['model_id' => $this->models()[$modelName]['id'], 'name' => $fieldName, 'type' => $fieldOptions['type']])->update(
            ['options' => json_encode($fieldOptions)]
        );
    }

    public function hasModel(string $modelName): bool
    {
        return isset($this->models()[$modelName]);
    }

    public function hasSectionType(string $modelName): bool
    {
        return in_array($modelName, $this->getSectionTypes());
    }

    public function hasContentType(string $modelName): bool
    {
        return in_array($modelName, $this->getContentTypes());
    }

    public function getModelsWithField(string $fieldName): array
    {
        $models = $this->models();
        if (! $models) {
            return [];
        }

        return array_values(
            Arr::sort(
                array_keys(
                    array_filter(
                        $models,
                        function ($model) use ($fieldName) {
                            return $this->hasField($model, $fieldName);
                        },
                        ARRAY_FILTER_USE_KEY
                    )
                )
            )
        );
    }

    public function getSortedDisplayedFieldListing(string $modelName): array
    {
        if (! $this->hasModel($modelName)) {
            throw ModelSchemaNotFoundException::named($modelName);
        }
        $models = $this->getModels($this->models);

        return array_values(
            array_map(
                function ($fieldName) {
                    return $this->getAliasFieldName((string) $fieldName);
                },
                array_keys(
                    Arr::sort(
                        array_filter(
                            $models[$modelName]['fields'],
                            function ($field) {
                                return (bool) $field['displayed'];
                            }
                        ),
                        fn ($field) => $field['options']['weight'] ?? null
                    )
                )
            )
        );
    }

    public function getFieldListing(string $modelName, Closure $callback = null): array
    {
        if (! $this->hasModel($modelName)) {
            throw ModelSchemaNotFoundException::named($modelName);
        }
        $models = $this->getModels($this->models);
        if ($callback === null) {
            if (! isset($this->fields[$modelName])) {
                $this->fields[$modelName] = array_values(
                    Arr::sort(
                        array_map(
                            function ($fieldName) {
                                return $this->getAliasFieldName((string) $fieldName);
                            },
                            array_keys($models[$modelName]['fields'])
                        )
                    )
                );
            }

            return $this->fields[$modelName];
        }

        return array_values(
            Arr::sort(
                array_map(
                    function ($fieldName) {
                        return $this->getAliasFieldName((string) $fieldName);
                    },
                    array_keys(
                        array_filter(
                            $models[$modelName]['fields'],
                            function ($field, $fieldName) use ($callback) {
                                if ($callback !== null && ! $callback($field, $fieldName)) {
                                    return false;
                                }

                                return true;
                            },
                            ARRAY_FILTER_USE_BOTH
                        )
                    )
                )
            )
        );
    }

    public function getFieldListingOfType(string $modelName, string|array $types): array
    {
        if (! $this->hasModel($modelName)) {
            throw ModelSchemaNotFoundException::named($modelName);
        }
        if (! is_array($types)) {
            $types = [$types];
        }

        $models = $this->getModels($this->models);

        return array_values(
            Arr::sort(
                array_map(
                    function ($fieldName) {
                        return $this->getAliasFieldName((string) $fieldName);
                    },
                    array_keys(
                        array_filter(
                            $models[$modelName]['fields'],
                            function ($field) use ($types) {
                                return in_array($field['type'], $types);
                            }
                        )
                    )
                )
            )
        );
    }

    public function getAllFieldsListingOfType(string $type): array
    {
        $models = $this->models();
        if (! $models) {
            return [];
        }
        $result = [];
        foreach (array_keys($models) as $modelName) {
            $fields = array_map(
                function ($fieldName) {
                    return $this->getAliasFieldName((string) $fieldName);
                },
                array_keys(
                    array_filter(
                        $models[$modelName]['fields'],
                        function ($field) use ($type) {
                            return $field['type'] == $type;
                        }
                    )
                )
            );
            if (! empty($fields)) {
                $result[$modelName] = $fields;
            }
        }

        return $result;
    }

    public function getAllFieldsListingOfWidget(string $widget): array
    {
        $result = [];
        foreach ($this->getContentTypes() as $type) {
            foreach ($this->getFieldListingOfType($type, 'text') as $fieldName) {
                $fieldOptions = $this->getFieldOptions($type, $fieldName);

                if (($fieldOptions['widget'] == $widget)) {
                    $result[(string) $type] = [$type, $fieldName];
                }
            }
        }

        return $result;
    }

    public function getDisplayedFieldListing(string $modelName): array
    {
        if (! $this->hasModel($modelName)) {
            throw ModelSchemaNotFoundException::named($modelName);
        }

        $models = $this->getModels($this->models);

        return array_map(
            function ($fieldName) {
                return $this->getAliasFieldName((string) $fieldName);
            },
            array_keys(
                array_filter(
                    $models[$modelName]['fields'],
                    function ($field) {
                        return (bool) $field['displayed'];
                    }
                )
            )
        );
    }

    public function getFieldOptions(string $modelName, string $fieldName): array
    {
        if (! $this->hasField($modelName, $fieldName)) {
            throw FieldSchemaNotFoundException::named($modelName, $fieldName);
        }
        $fieldName = $this->getOriginalFieldName($fieldName);
        $models = $this->getModels($this->models);

        $models[$modelName]['fields'][$fieldName]['options']['widget_type'] =
            $models[$modelName]['fields'][$fieldName]['options']['widget'];
        $models[$modelName]['fields'][$fieldName]['options']['type'] =
            $models[$modelName]['fields'][$fieldName]['type'];
        $models[$modelName]['fields'][$fieldName]['options']['displayed'] =
            $models[$modelName]['fields'][$fieldName]['displayed'];

        return $models[$modelName]['fields'][$fieldName]['options'];
    }

    public function hasField(string $modelName, string $fieldName): bool
    {
        return in_array($fieldName, $this->getFieldListing($modelName));
    }

    public function hasFieldOfType(string $modelName, mixed $types): bool
    {
        return ! empty($this->getFieldListingOfType($modelName, $types));
    }

    public function hasFields(string $modelName, array $fieldNames): bool
    {
        $modelFields = $this->getFieldListing($modelName);

        foreach ($fieldNames as $fieldName) {
            $fieldName = $this->getOriginalFieldName($fieldName);
            if (! in_array($fieldName, $modelFields)) {
                return false;
            }
        }

        return true;
    }

    public function refresh(): void
    {
        $this->models = null;
        $this->schema = null;
        $this->fields = [];
    }

    public function isSystemField(string $fieldName): bool
    {
        return in_array($fieldName, $this->systemFields);
    }

    public function getOriginalFieldName(string $fieldName): string
    {
        $originalFieldNames = array_flip($this->fieldAliases);
        if (isset($originalFieldNames[$fieldName])) {
            return $originalFieldNames[$fieldName];
        }

        return $fieldName;
    }

    /**
     * Magic call to allow
     * getSearchableContentTypes
     * getPermissibleContentTypes
     * ...
     */
    public function __call(string $name, mixed $arguments): array
    {
        $matches = null;
        if (preg_match('/^get(.*)(ContentTypes|SectionTypes)$/', $name, $matches)) {
            $option = Str::snake($matches[1]);
            switch ($matches[2]) {
                case 'ContentTypes':
                    return $this->getContentTypes(
                        function ($model) use ($option) {
                            return isset($model['options'][$option]) && $model['options'][$option];
                        }
                    );
                case 'SectionTypes':
                    return $this->getSectionTypes(
                        function ($model) use ($option) {
                            return isset($model['options'][$option]) && $model['options'][$option];
                        }
                    );
            }
        }
        throw new BadMethodCallException("Method [$name] does not exist.");
    }

    public function formatModelToLegacyFront(string $type, array $modelOptions): array
    {
        $modelOptions['name'] = $type;
        $modelOptions['class'] =
            $modelOptions['type'] === self::CONTENT_MODEL_TYPE ? type_to_class($type) : section_type_to_class($type);
        //  $modelOptions['model'] = $modelOptions['class'];

        return $modelOptions;
    }

    /**
     * Reformat field option for legacy uses by front
     */
    public function formatFieldToLegacyFront(string $fieldName, array $fieldOptions): array
    {
        $fieldOptions['name'] = $fieldName;

        return $fieldOptions;
    }

    protected function getAliasFieldName(string $fieldName): string
    {
        if (isset($this->fieldAliases[$fieldName])) {
            return $this->fieldAliases[$fieldName];
        }

        return $fieldName;
    }

    private function getModels(?array $models): array
    {
        if (! $models) {
            throw new ModelNotFoundException();
        }

        return $models;
    }

    private function setFieldOptions(string $modelName, mixed $field): array
    {
        $option = json_decode($field->field_options, true) ?? [];
        if (
            is_array($this->models) &&
            $field->field_name == 'langcode' &&
            data_get($this->models, $modelName.'.options.translatable', false) == true &&
            data_get($option, 'translatable', false) == true &&
            ! empty($option['allowed_values'])
        ) {
            $option['allowed_values'] = collect($option['allowed_values'])
                ->filter(fn ($value, $key) => in_array($key, list_languages()))
                ->mapWithKeys(fn ($value, $key) => [
                    $key => collect($value)->filter(fn ($v, $k) => in_array($k, list_languages()))->toArray(),
                ])
                ->toArray();
        }

        return [
            'id' => $field->field_id,
            'class' => $field->class,
            'type' => $field->type,
            'displayed' => $field->displayed === 1,
            'options' => $option,
        ];
    }
}
