<?php

declare(strict_types=1);

namespace Inside\Content\Services\Schema;

use Exception;
use FilesystemIterator;
use Illuminate\Support\Composer;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;
use Inside\Content\Contracts\RevisionService;
use Inside\Content\Exceptions\DynamicModelNotWritable;
use Inside\Content\Exceptions\SchemaNotFoundException;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Model;
use Inside\Facades\Package;
use Inside\Support\Filesystem;

final class ClassesService
{
    protected bool $rebuildAutoLoadNeeded = true;

    public function __construct(
        protected SchemaService $schema,
        protected Filesystem $filesystem,
        protected Composer $composer
    ) {
    }

    /**
     * @throws DynamicModelNotWritable
     */
    public function rebuild(bool $force = false): void
    {
        if ($force) {
            InsideSchema::refresh();
        }

        // Remove old dynamic classes
        $this->cleanDirectory(cms_base_path('vendor/maecia/inside/content/src/Models/Contents'));
        $this->cleanDirectory(cms_base_path('vendor/maecia/inside/content/src/Models/Sections'));

        // Rebuild content classes
        foreach (InsideSchema::getContentTypes() as $type) {
            $this->generateDynamicClass($type, 'content');
        }
        // Rebuild section classes
        foreach (InsideSchema::getSectionTypes() as $type) {
            $this->generateDynamicClass($type, 'section');
        }

        InsideSchema::refresh();
        $this->rebuildAutoload();
    }

    public function rebuildAutoload(): void
    {
        if (! $this->isRebuildAutoLoadNeeded()) {
            return;
        }
        // Rebuild autoload
        $this->composer->dumpAutoloads();
        $this->composer->dumpOptimized();
    }

    /**
     * Generate dynamic class for type $type on $domain ( section or content )
     * @throws DynamicModelNotWritable
     */
    protected function generateDynamicClass(string $type, string $domain): void
    {
        $class = type_to_class($type);
        $className = str_replace(
            'Inside\\Content\\Models\\Contents\\',
            '',
            $class
        );

        if ($domain === 'section') {
            $class = section_type_to_class($type);
            $className = str_replace(
                'Inside\\Content\\Models\\Sections\\',
                '',
                $class
            );
        }
        $tableName = class_to_table($class);
        if (! $tableName) {
            return;
        }

        $data = file_get_contents((string) realpath(__DIR__.'/../../../skeleton/'.$domain.'Model.skeleton'));

        $this->addFields($data, $class);
        /** @var string $data */
        $data = (string) str_replace('__CLASSNAME__', $className, $data);
        $data = (string) str_replace('__TABLENAME__', $tableName, $data);

        // Add traits
        $this->manageTraits($data, $class, $domain);

        $directory = 'vendor/maecia/inside/content/src/Models/'
            .Str::studly(Str::plural($domain));
        if (! is_writable((string) realpath(cms_base_path($directory)))) {
            throw new DynamicModelNotWritable('Dynamic class folder '
                .cms_base_path('vendor/maecia/inside/content/src/Models/'
                    .Str::plural($domain)).' is not writtable');
        }

        $classPath = str_replace(
            '/',
            DIRECTORY_SEPARATOR,
            cms_base_path($directory.'/'.$className.'.php')
        );

        file_put_contents($classPath, $data);
        chmod($classPath, 0775);
    }

    /**
     * @param string $code
     * @param string $class
     * @param string $domain
     * @return void
     * @throws Exception
     */
    protected function manageTraits(
        string &$code,
        string $class,
        string $domain
    ) {
        $traits = [];

        // Manage Searchable Models
        $this->manageSearchableTrait($traits, $class);

        if ($domain == 'content') {
            // Manage slug
            $traits['SlugTrait'] = 'Inside\Slug\Concerns\SlugTrait';

            // Manage HasWorkflow Models
            $this->manageHasWorkflowTrait($traits, $class);

            // Manage Statistics trait
            $this->manageStatisticsTrait($traits, $class);

            // Manage Revisionable
            $this->manageRevisionableTrait($traits, $class);
        }

        // Manage Archive trait
        $this->manageArchiveTrait($traits, $class);

        // Manage publishable trait
        $this->managePublishableTrait($traits, $class);

        $code = str_replace(
            '__TRAIT_NAMESPACE__',
            'use '.implode(";\nuse ", array_values($traits)).";\n",
            $code
        );
        $code = str_replace(
            '__CLASS_TRAIT__',
            'use '.implode(', ', array_keys($traits)).";\n",
            $code
        );
    }

    /**
     * Add fields to dynamic class
     *
     * @param string $data
     * @param  string  $className
     */
    protected function addFields(string &$data, string $className): void
    {
        $modelType = class_to_type($className);

        /** @var string $pivotField */
        $pivotField = file_get_contents((string) realpath(__DIR__.'/../../../skeleton/pivotField.skeleton'));
        /** @var string $reversePivotField */
        $reversePivotField = file_get_contents((string) realpath(__DIR__.'/../../../skeleton/reversePivotField.skeleton'));
        /** @var string $reverseCommentField */
        $reverseCommentField = file_get_contents((string) realpath(__DIR__.'/../../../skeleton/reverseCommentField.skeleton'));
        /** @var string $polymorphicField */
        $polymorphicField = file_get_contents((string) realpath(__DIR__.'/../../../skeleton/polymorphicField.skeleton'));
        /** @var string $sectionField */
        $sectionField = file_get_contents((string) realpath(__DIR__.'/../../../skeleton/sectionField.skeleton'));
        $pivots = [];
        $polymorphics = [];
        $sections = [];
        $casts = [];

        try {
            $schema = InsideSchema::getSchemaInformation($modelType);
        } catch (SchemaNotFoundException) {
            $schema = [
                'fields' => [],
            ];
        }

        $protectedFields = ['authors', 'reactions', 'statistics'];

        if (! is_array($schema) || ! array_key_exists('fields', $schema) || ! is_array($schema['fields'])) {
            return;
        }

        // Manage normal fields
        foreach ($schema['fields'] as $fieldName => $fieldInfos) {
            if (! is_array($fieldInfos) || ! isset($fieldInfos['type'])) {
                continue;
            }
            if (in_array($fieldName, $protectedFields)) {
                continue;
            }
            if ($fieldInfos['type'] == 'reference' || $fieldInfos['type'] == 'comment') {
                $pivotData = $pivotField;
                $isReverse = isset($fieldInfos['reverse']) && $fieldInfos['reverse'] === true;
                if ($isReverse) {
                    $pivotData = $reversePivotField;
                }
                $target = $realTarget = $fieldInfos['name'];

                if (class_to_type($className) == 'comments') {
                    // Special comment class
                    $pivotData = $reverseCommentField;
                    $pivotData = str_replace(
                        '__FIELDNAME__',
                        Str::camel($fieldInfos['name']),
                        $pivotData
                    );
                    $pivotData = str_replace(
                        '__TARGETCONTENT__',
                        Str::studly($target),
                        $pivotData
                    );
                } else {
                    if (isset($fieldInfos['options']['target']) && ! is_array($fieldInfos['options']['target'])) {
                        $fieldInfos['options']['target'] = [$fieldInfos['options']['target']];
                    }

                    if (isset($fieldInfos['options']['target'][0]) && ! empty($fieldInfos['options']['target'][0])
                        && $fieldInfos['options']['target'][0] !== $target
                    ) {
                        $target = $fieldInfos['options']['target'][0];
                    }
                    $targetType = $targetFieldName = $fieldInfos['name'];
                    if ($isReverse) {
                        // In reverse target on options is the fieldname on target type
                        $target = $fieldInfos['target'][0];
                        if (strpos($targetType, '.') > 0) {
                            [$targetType, $targetFieldName] = explode('.', $targetFieldName, 2);
                            $targetFieldName = $targetType.'_'.$targetFieldName;
                        }
                    }
                    $realTarget = $target;
                    $pivotData = str_replace(
                        '__FIELDNAME__',
                        Str::camel($fieldInfos['name']),
                        $pivotData
                    );
                    $pivotData = str_replace(
                        '__TARGETCONTENT__',
                        Str::studly($target),
                        $pivotData
                    );
                    $target = $fieldInfos['name'];
                    if (strpos($target, '.') > 0) {
                        [$target, $trash] = explode('.', $fieldInfos['name'], 2);
                    }
                    $pivotData = str_replace('__TARGETFIELD__', Str::snake($target), $pivotData);
                    $pivotData = str_replace('__SFIELDNAME__', Str::studly($targetFieldName), $pivotData);
                    $pivotData = str_replace(
                        '__ORIGINALFIELD__',
                        $fieldInfos['field_name'] ?? Str::snake($targetFieldName),
                        $pivotData
                    );
                    $pivotData = str_replace(
                        '__ORIGINALFIELDS__',
                        implode('\', \'', $fieldInfos['options']['target'] ?? []),
                        $pivotData
                    );
                }
                if (InsideSchema::hasSchemaInformation($realTarget)) {
                    $pivots[] = $pivotData;
                }
            } elseif ($fieldInfos['type'] == 'section') {
                $targets = $fieldInfos['options']['target'] ?? [];
                $polymorphicList = [];

                foreach ($targets as $target) {
                    $polymorphicData = $polymorphicField;
                    $polymorphicData = str_replace(
                        '__POLYMORPHIC_NAME__',
                        'sectionable',
                        $polymorphicData
                    );
                    $polymorphicData = str_replace(
                        '__POLYMORPHIC_DOMAIN__',
                        'Sections',
                        $polymorphicData
                    );
                    $polymorphicData = str_replace(
                        '__POLYMORPHIC_MODEL__',
                        Str::studly($target),
                        $polymorphicData
                    );
                    $polymorphicData = str_replace(
                        '__POLYMORPHIC_FIELD__',
                        Str::camel($target).'SectionItem',
                        $polymorphicData
                    );
                    $polymorphics[] = $polymorphicData;
                    $polymorphicList[] = '\''.Str::camel($target).'SectionItem'
                        .'\'';
                }

                $sectionData = $sectionField;
                $sectionData = str_replace(
                    '__POLYMORPHIC_LIST__',
                    implode(',', $polymorphicList),
                    $sectionData
                );
                $sectionData = str_replace(
                    '__POLYMORPHIC_NAME__',
                    Str::studly($fieldName),
                    $sectionData
                );
                $sectionData = str_replace(
                    '__FIELDNAME__',
                    $fieldName,
                    $sectionData
                );
                $sections[] = $sectionData;
            }

            if (in_array($fieldInfos['type'], ['timestamp', 'datetime'])) {
                $casts[] = '\''.$fieldName.'\' => \'timestamp\'';
            } elseif (in_array($fieldInfos['type'], ['boolean', 'checkbox'])) {
                $casts[] = '\''.$fieldName.'\' => \'boolean\'';
            }
        }

        $data = str_replace('__PIVOTS__', implode("\n", $pivots), $data);
        $data = str_replace(
            '__POLYMORPHICS__',
            implode("\n", array_unique($polymorphics)),
            $data
        );
        $data = str_replace('__SECTIONS__', implode("\n", $sections), $data);
        $data = str_replace('__CASTS__', trim(implode(",\n        ", $casts)), $data);
    }

    /**
     *  Add correct searchable trait
     *
     * @param  array  $traits
     * @param  string  $class
     *
     * @throws Exception
     */
    protected function manageSearchableTrait(
        array &$traits,
        string $class
    ): void {
        $trait = search_namespace('Models\Traits\Searchable');
        if ($trait[0] === '\\') {
            $trait = substr($trait, 1);
        }
        $model = Model::where('class', $class)->first();

        if ($model) {
            $traits['Searchable'] = $trait;
        }
    }

    /**
     * @param array $traits
     * @param  string  $class
     */
    protected function manageHasWorkflowTrait(array &$traits, string $class): void
    {
        if (Package::has('inside-workflow')) {
            $traits['HasWorkflow'] = 'Inside\Workflow\Models\Traits\HasWorkflow';
        }
    }

    /**
     * @param  array  $traits
     * @param  string  $class
     */
    protected function manageStatisticsTrait(array &$traits, string $class): void
    {
        $traits['HasStatistics'] = 'Inside\Statistics\Models\Traits\HasStatistics';
    }

    protected function manageRevisionableTrait(array &$traits, string $class): void
    {
        if ((App::make(RevisionService::class))->isEnabled($class)) {
            $traits['Revisionable'] = 'Inside\Content\Models\Traits\Revisionable';
        }
    }

    /**
     * @param  array  $traits
     * @param  string  $class
     */
    protected function manageArchiveTrait(array &$traits, string $class): void
    {
        if (Package::has('inside-archive')) {
            $traits['HasArchive'] = 'Inside\Archive\Models\Traits\HasArchive';
        }
    }

    /**
     * @param  array  $traits
     * @param  string  $class
     * @return void
     */
    protected function managePublishableTrait(array &$traits, string $class)
    {
        $table = class_to_table($class);
        if ($table && Schema::hasColumn($table, 'published_at')) {
            $traits['Publishable'] = 'Inside\Content\Models\Traits\Publishable';
        }
    }

    /**
     * Clean directory $directory from php files
     *
     * @param  string  $directory
     *
     * @return bool
     */
    protected function cleanDirectory(string $directory): bool
    {
        if (! $this->filesystem->isDirectory($directory)) {
            return false;
        }

        $items = new FilesystemIterator($directory);

        foreach ($items as $item) {
            if (! is_string($item) && $item->getExtension() === 'php') {
                $this->filesystem->delete($item->getPathname());
            }
        }

        return true;
    }

    /**
     * is rebuild autoload needed
     *
     * @return bool
     */
    public function isRebuildAutoLoadNeeded(): bool
    {
        return $this->rebuildAutoLoadNeeded;
    }

    /**
     * Disable rebuild autoload on class reload
     * @return void
     */
    public function disableRebuildAutoLoad()
    {
        $this->rebuildAutoLoadNeeded = false;
    }

    /**
     * Enable rebuild autoload on class reload
     * @return void
     */
    public function enableRebuildAutoLoad()
    {
        $this->rebuildAutoLoadNeeded = true;
    }

    /**
     * @param mixed $callback
     *
     * @return mixed
     */
    public function withoutRebuildAutoLoad($callback)
    {
        $this->disableRebuildAutoLoad();

        try {
            return $callback();
        } finally {
            $this->enableRebuildAutoLoad();
        }
    }
}
