<?php

declare(strict_types=1);

namespace Inside\Content\Services\Revision;

use Carbon\Carbon;
use Closure;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Contracts\Container\Container as ContainerContract;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\UploadedFile;
use Illuminate\Pipeline\Pipeline;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Contracts\RevisionService as RevisionServiceContract;
use Inside\Content\Contracts\SchemaService as SchemaServiceContract;
use Inside\Content\Contracts\Serializer;
use Inside\Content\Contracts\WysiwygImageService;
use Inside\Content\Events\ContentRestored;
use Inside\Content\Events\ContentSavedWithImages;
use Inside\Content\Events\NewRevision;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Models\Content;
use Inside\Content\Models\Revision;
use Inside\Content\Models\Section;
use Inside\Content\Models\WysiwygImage;
use Inside\Content\Services\Revision\Widget\Widget;
use Inside\Host\Bridge\BridgeContent;
use Throwable;

final class RevisionService implements RevisionServiceContract
{
    protected bool $enable = true;

    public const TRANSFORMER_TO_FRONT_VIEW = 'toFrontView';

    public const TRANSFORMER_TO_BRIDGE_CONTENT_DATA = 'toBridgeContentData';

    /** @var array<int, string> */
    protected array $revisionMiddleware = [];

    /** @var array<int, string> */
    protected array $restoreMiddleware = [];

    public function __construct(
        protected SchemaServiceContract $schemaService,
        protected ContainerContract $container,
        protected WysiwygImageService $wysiwygImageService,
        protected Serializer $serializerService
    ) {
    }

    public function isEnabled(Content|string $content): bool
    {
        if (! $this->enable) {
            return false;
        }

        if (is_string($content)) {
            return $this->isEnabledForContentType(class_to_type($content));
        }

        return (new Pipeline($this->container))->send($content)
            ->through($this->revisionMiddleware)
            ->then(function (Content $content) {
                return $this->isEnabledForContentType(class_to_type($content));
            });
    }

    public function withoutWorkflow(Closure $callback): mixed
    {
        $this->disableRevision();

        try {
            return $callback();
        } finally {
            $this->enableRevision();
        }
    }

    public function enableRevision(): void
    {
        $this->enable = true;
    }

    public function disableRevision(): void
    {
        $this->enable = false;
    }

    public function canBeRestored(Content $content, int $version): bool
    {
        return (new Pipeline($this->container))->send($content)
            ->through($this->restoreMiddleware)
            ->then(function (Content $content) use ($version) {
                return $version < $content->current_revision->version;
            });
    }

    public function transformRevisionToFrontView(Revision $revision): array
    {
        return $this->transformRevisionWithTransformer($revision, self::TRANSFORMER_TO_FRONT_VIEW);
    }

    public function transformRevisionToBridgeContentData(Revision $revision): array
    {
        return $this->transformRevisionWithTransformer($revision, self::TRANSFORMER_TO_BRIDGE_CONTENT_DATA);
    }

    /**
     * @throws Exception
     */
    public function transformContentFieldsWithTransformer(
        Content|Section $content,
        Revision $revision,
        string $transformer,
        ?Content $root = null
    ): array {
        $type = class_to_type($content);
        $data = [];
        $fieldNames = $this->schemaService->getSortedDisplayedFieldListing($type);

        foreach ($fieldNames as $fieldName) {
            if (in_array($fieldName, ['password', 'authors', 'langcode', 'status', 'created_at'])) {
                continue;
            }

            $fieldOptions = $this->schemaService->getFieldOptions($type, $fieldName);
            try {
                $widget = $this->makeWidget($content, $revision, $fieldName, $fieldOptions, $root);
                $data[$fieldName] = match ($transformer) {
                    self::TRANSFORMER_TO_BRIDGE_CONTENT_DATA => $widget->toBridgeContentData(),
                    self::TRANSFORMER_TO_FRONT_VIEW => $widget->toFrontView(),
                    default => throw new Exception('Unsupported transformer'),
                };
            } catch (BindingResolutionException $exception) {
                Log::warning(__('[RevisionService] Widget not supported :type (:message)', [
                    'type' => $fieldOptions['widget'],
                    'message' => $exception->getMessage(),
                ]
                ));
            }
        }

        return $data;
    }

    public function findRevisionImage(string $type, string $id, int $version, string $imagePath): ?WysiwygImage
    {
        $revision = $this->findRevision($type, $id, $version);

        if (is_null($revision)) {
            return null;
        }
        $content = $this->serializerService->unserialize($revision->data);
        if (is_null($content)) {
            return null;
        }
        $image = $content->images->where('relative_url', 'wysiwyg/images/'.$imagePath)->first();
        if (is_null($image)) {
            return null;
        }

        $storage = Storage::disk('local');
        $path = 'revisions/'.$this->getRevisionFilePath($content, $revision, $image->relative_url);
        if (! $storage->exists($path)) {
            return null;
        }

        return tap($image, fn ($image) => $image->revision_path = $storage->path($path));
    }

    public function findRevisionFile(string $type, string $id, int $version, string $filePath): ?string
    {
        $revision = $this->findRevision($type, $id, $version);
        if (is_null($revision)) {
            return null;
        }
        $content = $this->serializerService->unserialize($revision->data);
        if (is_null($content)) {
            return null;
        }

        $storage = Storage::disk('local');
        $path = 'revisions/'.$this->findRevisionFilePath($content, $revision, $filePath);
        if (! $storage->exists($path)) {
            return null;
        }

        return $storage->path($path);
    }

    protected function findRevisionFilePath(Content|Section $content, Revision $revision, string $path, string $prefix = ''): ?string
    {
        $type = class_to_type($content);
        foreach ($this->schemaService->getFieldListingOfType($type, ['file', 'image']) as $fieldName) {
            $path = urldecode($path);
            if ($content->{$fieldName} === urldecode($path)) {
                return $prefix.$this->getRevisionFilePath($content, $revision, null, empty($prefix)).'/'.$path;
            }
        }

        if ($this->schemaService->hasFieldOfType($type, 'section')) {
            foreach ($this->schemaService->getFieldListingOfType($type, 'section') as $fieldName) {
                /** @var array<int, Section> $sections */
                $sections = $content->{Str::snake('section_'.$fieldName)};
                foreach ($sections as $section) {
                    $found = $this->findRevisionFilePath(
                        $section,
                        $revision,
                        $path,
                        $prefix.$this->getRevisionFilePath($content, $revision, null, empty($prefix)).'/'
                    );
                    if (! is_null($found)) {
                        return $found;
                    }
                }
            }
        }

        return null;
    }

    public function createNewRevision(Content $content): void
    {
        if (! $this->isEnabled($content)) {
            return;
        }

        try {
            $serialization = $this->serializerService->serialize($content);

            if (
                ! is_null($currentRevision = $content->current_revision) &&
                $currentRevision->data === $serialization &&
                $currentRevision->created_at->diffInSeconds() < 3
            ) {
                Log::warning('[RevisionService::createNewRevision] detected false revision');

                return;
            }

            $revision = DB::transaction(function () use ($content, $serialization) {
                return Revision::lockForUpdate()->create([
                    'contentable_type' => get_class($content),
                    'contentable_id' => $content->uuid,
                    'user_id' => $content->modificator?->uuid,
                    'locale' => $content->langcode,
                    'data' => $serialization,
                    'version' => $content->revisions->isEmpty() ? 1 : ($content->revisions->first()->version + 1),
                ]);
            });
        } catch (Throwable $exception) {
            Log::error(__(
                '[RevisionService::createNewRevision] DB::transaction failed with message :message',
                [
                    'message' => $exception->getMessage(),
                ]
            ));

            return;
        }

        $this->storeRevisionFiles($content, $revision);

        NewRevision::dispatch($revision);
    }

    public function restoreFromRevision(Content $content, Revision $revision): bool
    {
        if (! $this->canBeRestored($content, $revision->version)) {
            return false;
        }

        /** @var BridgeContent $bridge */
        $bridge = App::make(BridgeContent::class);

        $data = $this->transformRevisionToBridgeContentData($revision);
        $type = class_to_type($content);
        $data = ContentHelper::addMissings($type, $data, false);
        $data = ContentHelper::castAttributes($type, $data);

        $data['published_at'] = get_date($content->published_at)?->format('Y-m-d H:i:s');
        $data['status'] = $content->status;
        $data['author'] = $content->author;
        unset($data['created_at'], $data['updated_at']);
        $data['uuid'] = $content->uuid;

        if ($this->schemaService->hasField($type, 'last_update')) {
            $data['last_update'] = time();
        }

        try {
            $this->switchToRevisionFilePath($content, $data, $revision);
            ContentSavedWithImages::dispatch($content);
            Log::debug(__(
                '[RevisionService::restoreFromRevision] restoring content :type<:uuid>',
                [
                    'type' => class_to_type($content),
                    'id' => $content->uuid ?? '<null>',
                ]), $data);
            $status = ! is_null($bridge->contentUpdate($type, $data));

            /** @var ?User $user */
            $user = Auth::user();

            ContentRestored::dispatch($revision, $user);
        } catch (Exception $exception) {
            Log::error(__(
                '[RevisionService::restoreFromRevision] failed to update content :type (:id) => :message',
                [
                    'type' => class_to_type($content),
                    'id' => $content->uuid ?? '<null>',
                    'message' => $exception->getMessage(),
                ]));
            $status = false;
        }

        return $status;
    }

    public function cleanupRevisionBeforeDeletion(Revision $revision): void
    {
        $content = $revision->content;

        /** @phpstan-ignore-next-line */
        if (is_null($content)) {
            $content = new $revision->contentable_type();
            $content->uuid = $revision->contentable_id;
        }

        Log::debug(__(
            'Deleting files for revision :id on content :type<:uuid> from path :path',
            [
                'id' => $revision->id,
                'type' => $revision->contentable_type,
                'uuid' => $revision->contentable_id,
                'path' => $this->getRevisionFilePath($content, $revision),
            ]
        ));

        Storage::disk('local')->deleteDirectory('revisions/'.$this->getRevisionFilePath(
            content: $content,
            revision: $revision,
            withRevision: false
        ));
    }

    public function revisionMiddleware(string|array $middleware): self
    {
        if (! is_array($middleware)) {
            $middleware = [$middleware];
        }

        $this->revisionMiddleware = array_unique(array_merge($this->revisionMiddleware, $middleware));

        return $this;
    }

    public function restoreMiddleware(string|array $middleware): self
    {
        if (! is_array($middleware)) {
            $middleware = [$middleware];
        }

        $this->restoreMiddleware = array_unique(array_merge($this->restoreMiddleware, $middleware));

        return $this;
    }

    /** @return array<int, string> */
    public function getSupportedContentTypes(): array
    {
        $types = config('contents.revisionable.types', []);
        if (is_string($types) && ! empty($types)) {
            $types = explode(',', $types);
        }
        if (! is_array($types)) {
            $types = [];
        }

        return array_values(array_intersect($types, $this->schemaService->getContentTypes()));
    }

    protected function transformRevisionWithTransformer(Revision $revision, string $transformer): array
    {
        if (empty($revision->data)) {
            return [];
        }
        $content = $this->serializerService->unserialize($revision->data);
        if (is_null($content)) {
            return [];
        }

        try {
            return $this->transformContentFieldsWithTransformer($content, $revision, $transformer);
        } catch (Exception $exception) {
            Log::error(__(
                '[RevisionService::transformRevisionWithTransformer] failed to transform fields => :message',
                [
                    'message' => $exception->getMessage(),
                ]
            ));
        }

        return [];
    }

    protected function findRevision(string $type, string $id, int $version): ?Revision
    {
        return Revision::where('contentable_type', type_to_class($type))
            ->where('contentable_id', $id)
            ->where('version', $version)
            ->first();
    }

    /**
     * @throws BindingResolutionException
     */
    protected function makeWidget(
        Content|Section $content,
        Revision $revision,
        string $fieldName,
        array $options,
        ?Content $root
    ): Widget {
        return $this->container->make(
            'Inside\\Content\\Services\\Revision\\Widget\\'.Str::studly($options['widget'].'_widget'),
            [
                'content' => $content,
                'revision' => $revision,
                'fieldName' => $fieldName,
                'options' => $options,
                'root' => $root,
            ]
        );
    }

    protected function isEnabledForContentType(string $contentClass): bool
    {
        return config('contents.revisionable.enabled', false)
            && in_array($contentClass, array_intersect(
                $this->getSupportedContentTypes(),
                $this->schemaService->getContentTypes())
            );
    }

    protected function storeRevisionFiles(Content|Section $content, Revision $revision, string $prefix = ''): void
    {
        $type = class_to_type($content);
        $storage = Storage::disk('local');
        // Direct files
        foreach ($this->schemaService->getFieldListingOfType($type, ['file', 'image']) as $fieldName) {
            $filePath = $content->{$fieldName};
            if (! is_null($filePath) && $storage->exists($filePath)) {
                $destinationPath = 'revisions/'.$prefix.$this->getRevisionFilePath($content, $revision, $filePath, empty($prefix));
                Log::debug(__(
                    'Copying file :file to :destination',
                    [
                        'file' => $filePath,
                        'destination' => $destinationPath,
                    ]
                ));
                if ($storage->exists($destinationPath)) {
                    Log::debug(__(
                        'Destination :destination already exists, deleting it first',
                        [
                            'file' => $filePath,
                            'destination' => $destinationPath,
                        ]
                    ));
                    $storage->delete($destinationPath);
                }

                $storage->copy($filePath, $destinationPath);
            }
        }
        if ($content instanceof Content) {
            $this->storeWysiwygImages($content, $revision, $content->images);
        }
        // From sections
        if ($this->schemaService->hasFieldOfType($type, 'section')) {
            foreach ($this->schemaService->getFieldListingOfType($type, 'section') as $fieldName) {
                /** @var array<int, Section> $sections */
                $sections = $content->{Str::snake('section_'.$fieldName)};
                foreach ($sections as $section) {
                    $this->storeRevisionFiles(
                        $section,
                        $revision,
                        $prefix.$this->getRevisionFilePath($content, $revision, null, empty($prefix)).'/'
                    );
                }
            }
        }
    }

    /**
     * @param  Content  $content
     * @param  Revision  $revision
     * @param  Collection<WysiwygImage>  $images
     * @return void
     */
    protected function storeWysiwygImages(Content $content, Revision $revision, Collection $images): void
    {
        $storage = Storage::disk('local');
        foreach ($images as $image) {
            if (! empty($image->path) && File::exists($image->path)) {
                Log::debug(__(
                    'Copying Wysiwyg image :file to :destination',
                    [
                        'file' => $image->path,
                        'destination' => $storage->path(
                            'revisions/'.$this->getRevisionFilePath($content, $revision, $image->relative_url)
                        ),
                    ]
                ));
                $destinationPath = $storage->path(
                    'revisions/'.$this->getRevisionFilePath($content, $revision, $image->relative_url)
                );
                if (! File::exists(dirname($destinationPath))) {
                    File::makeDirectory(
                        dirname($destinationPath),
                        config('filesystems.disks.wysiwyg_images.permissions.dir.public', 0755),
                        true
                    );
                }
                if (false === File::copy(
                    $image->path,
                    $destinationPath
                )) {
                    Log::error(__(
                        'Failed to copy Wysiwyg image',
                        [
                        ]
                    ));
                    continue;
                }
                File::chmod(
                    $destinationPath,
                    config('filesystems.disks.wysiwyg_images.permissions.file.public', 0755)
                );
            }
        }
    }

    protected function switchToRevisionFilePath(
        Content|Section $content,
        array $data,
        Revision $revision,
        string $prefix = ''
    ): void {
        $type = class_to_type($content);
        // Direct files
        foreach ($this->schemaService->getFieldListingOfType($type, ['file', 'image']) as $fieldName) {
            $filePath = $data[$fieldName] ?? null;
            if (! is_null($filePath)) {
                $storedFilePath = 'revisions/'.$prefix.$this->getRevisionFilePath(
                    $content,
                    $revision,
                    $filePath,
                    empty($prefix)
                );
                Log::debug(__(
                    '[switchToRevisionFilePath] trying restoring :stored to :path for :type<:uuid>',
                    [
                        'stored' => $storedFilePath,
                        'path' => $filePath,
                        'type' => class_to_type($content),
                        'uuid' => $content->uuid ?? '<null>',
                    ]
                ));
                if (Storage::disk('local')->exists($storedFilePath)
                    && ! Storage::disk('local')->exists($filePath)) {
                    Log::debug(__(
                        '[switchToRevisionFilePath] restoring :stored to :path for :type<:uuid>',
                        [
                            'stored' => $storedFilePath,
                            'path' => $filePath,
                            'type' => class_to_type($content),
                            'uuid' => $content->uuid ?? '<null>',
                        ]
                    ));
                    Storage::disk('local')->copy($storedFilePath, $filePath);
                }
            }
        }
        // From sections
        if ($this->schemaService->hasFieldOfType($type, 'section')) {
            foreach ($this->schemaService->getFieldListingOfType($type, 'section') as $fieldName) {
                $sections = $content->{Str::snake('section_'.$fieldName)};
                foreach ($sections as $section) {
                    $sectionData = Arr::first(array_filter(
                        $data[$fieldName],
                        fn ($value) => $value['bundle'] === class_to_type($section) && $value['pgID'] === $section->uuid
                    ));
                    if (is_null($sectionData)) {
                        continue;
                    }
                    $this->switchToRevisionFilePath(
                        $section,
                        $sectionData,
                        $revision,
                        $prefix.$this->getRevisionFilePath($content, $revision, null, empty($prefix)).'/'
                    );
                }
            }
        }
        if (! $content instanceof Content) {
            return;
        }
        $storage = Storage::disk('local');
        $urls = [];
        foreach ($this->extractExistingImagesFromContent($revision) as $image) {
            $databaseImage = WysiwygImage::find($image->id);
            $disk = Storage::disk($image->disk);
            if (
                is_null($databaseImage)
                || (
                    $content->images->contains($databaseImage)
                    && $databaseImage->path !== $image->path
                )
                || ! $disk->exists($image->path)
            ) {
                $sourcePath = 'revisions/'.$this->getRevisionFilePath($content, $revision, $image->relative_url);
                if (! $storage->exists($sourcePath)) {
                    Log::error(__(
                        '[RevisionService::switchToRevisionFilePath] Failed to get file :path',
                        [
                            'path' => $sourcePath,
                        ]
                    ));
                    continue;
                }
                $file = new UploadedFile(
                    $storage->path($sourcePath),
                    $image->filename,
                    $storage->mimeType($sourcePath),
                );
                $urls[] = $this->wysiwygImageService->upload($file);
            }
        }
        $this->wysiwygImageService->store($content, $urls);
    }

    protected function getRevisionFilePath(
        Content|Section $content,
        Revision $revision,
        ?string $path = null,
        bool $withRevision = true
    ): string {
        return class_to_type($content).'/'.$content->uuid.($withRevision ? '/'.$revision->version : '').(! is_null($path) ? '/'.$path : '');
    }

    /**
     * @param  Revision  $revision
     * @return array<WysiwygImage>
     */
    protected function extractExistingImagesFromContent(Revision $revision): array
    {
        if (empty($revision->data)) {
            return [];
        }
        $content = $this->serializerService->unserialize($revision->data);
        if (is_null($content)) {
            return [];
        }

        return $content->images->all() ?? [];
    }
}
