<?php

namespace Inside\Content\Services\Queries\Traits;

use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Content\Facades\Schema;
use Inside\Support\Str;
use InvalidArgumentException;

trait DateFilterQuery
{
    public function filterByDateOffsetQuery(array $value, Builder $query): bool
    {
        if (! isset($value['offset']) || ! isset($value['date']) || ! isset($value['field'])) {
            return false;
        }
        // Example: "date_offset":{"offset":5,"date":"11-05","field":"birthday"}
        $scope = (int) floor($value['offset'] / 2);
        $givenDate = explode('-', $value['date']); //given Date
        $countDate = count($givenDate);

        if ($countDate == 3) { //YYYY-MM-DD format given
            $mainDate = new Carbon($givenDate[0].'-'.$givenDate[1].'-'.$givenDate[2]);
        } elseif ($countDate == 2) { // MM-DD format given
            $mainDate = new Carbon('2000-'.$givenDate[0].'-'.$givenDate[1]);
        } else {
            throw new InvalidArgumentException('wrong date offset');
        }

        $myDate = $mainDate->copy();

        $whereList = [];
        $whereList[] = [
            $value['field'],
            'like',
            ($countDate == 3 ? $myDate->format('Y') : '%').'-'
            .$myDate->format('m').'-'.$myDate->format('d').'%',
        ];
        // TODO:cleanup
        for ($i = 1; $i <= $scope; $i++) {
            $prevDay = $mainDate->copy()->subDays($i);
            $whereList[] = [
                $value['field'],
                'like',
                ($countDate == 3 ? $myDate->format('Y') : '%').'-'
                .$prevDay->format('m').'-'.$prevDay->format('d').'%',
            ];

            $addDay = $mainDate->copy()->addDays($i);
            $whereList[] = [
                $value['field'],
                'like',
                ($countDate == 3 ? $myDate->format('Y') : '%').'-'
                .$addDay->format('m').'-'.$addDay->format('d').'%',
            ];
        }

        $query->where(function ($subQuery) use ($whereList) {
            foreach ($whereList as $where) {
                $subQuery->orWhere($where[0], $where[1], $where[2]);
            }
        });
        $query->orderBy($value['field']);

        return true;
    }

    public function filterByDateRangeQuery(
        string $type,
        array $options,
        Builder $query
    ): bool {
        $fromDate = get_date($options['from'], '!Y-m-d') ?? get_date($options['from'], 'Y-m-d H:i:s') ?? false;

        if (! $fromDate instanceof Carbon) {
            Log::debug('[DateFilterQuery::filterByDateRangeQuery] failed to build from Date ['.($options['from'] ?? 'missing').']');

            return false;
        }

        $toDate = get_date($options['to'], '!Y-m-d') ?? get_date($options['to'], 'Y-m-d H:i:s') ?? false;

        if (! $toDate instanceof Carbon) {
            Log::debug('[DateFilterQuery::filterByDateRangeQuery] failed to build to Date ['.($options['to'] ?? 'missing').']');

            return false;
        }

        // Published_at is a special field
        if (! isset($options['field_name'])
            || (
                ! in_array($options['field_name'], ['created_at', 'updated_at', 'published_at', 'sent_at'])
                && ! Schema::hasField($type, $options['field_name'])
            )
        ) {
            Log::debug(
                '[DateFilterQuery::filterByDateRangeQuery] Field name is wrong ['.
                ($options['field_name'] ?? 'missing').']'
            );

            return false;
        }

        $sortDirection = Str::lower($options['sort_direction'] ?? 'asc');

        if (! in_array($sortDirection, ['asc', 'desc'])) {
            $sortDirection = 'asc';
        }

        if (array_key_exists('sort', $this->filters) && $this->filters['sort'] === 'random') {
            $sortDirection = 'random';
        }

        $fieldName = $options['field_name'];
        $fieldType = 'timestamp';
        if (! in_array($fieldName, ['created_at', 'updated_at', 'published_at'])) {
            $fieldOptions = Schema::getFieldOptions($type, $fieldName);
            $fieldType = $fieldOptions['type'] ?? null;
        }
        if ($fieldType != 'timestamp') {
            Log::debug(
                '[DateFilterQuery::filterByDateRangeQuery] Field name is ['
                .$options['field_name']
                .'] is not a timestamp field!'
            );

            return false;
        }

        $anyYears = (bool) $options['any_years'] ?? false;

        if (! $anyYears) {
            $query->whereBetween($fieldName, [$fromDate, $toDate]);

            if ($sortDirection !== 'random') {
                $query->orderBy($fieldName, $sortDirection);
            }
        } else {
            $userTimezone = config('app.default_user_timezone');

            $tz = new \DateTimeZone($userTimezone);
            $offsetSeconds = $tz->getOffset(Carbon::now($userTimezone));
            $offsetHours = $offsetSeconds / 3600;
            $offsetSign = ($offsetHours >= 0) ? '+' : '-';
            $absOffsetHours = abs(floor($offsetHours));
            $absOffsetMinutes = abs(($offsetHours - floor($offsetHours)) * 60);
            $mysqlOffset = sprintf('%s%02d:%02d', $offsetSign, $absOffsetHours, $absOffsetMinutes);

            $convertedField = "CONVERT_TZ($fieldName, 'UTC', '$mysqlOffset')";

            $query->whereRaw("DATE_FORMAT($convertedField, '%m%d') BETWEEN DATE_FORMAT(?, '%m%d') AND DATE_FORMAT(?, '%m%d')", [$fromDate, $toDate]);
            $query->addSelect(DB::raw("MONTH($convertedField) as {$type}_{$fieldName}_month"));
            $query->addSelect(DB::raw("DAY($convertedField) as {$type}_{$fieldName}_day"));

            if ($sortDirection !== 'random') {
                $query->orderBy("{$type}_{$fieldName}_month", $sortDirection)
                    ->orderBy("{$type}_{$fieldName}_day", $sortDirection);
            }
        }

        return true;
    }
}
