<?php

declare(strict_types=1);

namespace Inside\Content\Services\Queries;

use Exception;
use Illuminate\Container\Container;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Query\Builder as QueryBuilder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Pagination\Paginator;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Inside\Authentication\Facades\Authentication;
use Inside\Authentication\Models\User;
use Inside\Content\Contracts\QueryHelper;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Comments;
use Inside\Content\Models\Section;
use Inside\Content\Validation\Rules\LinkRule;
use Inside\Content\Validation\Rules\SectionRule;
use Inside\Database\Eloquent\Builder;
use Inside\Host\Bridge\BridgeContent;
use Inside\Support\HasMimeTypes;
use Inside\Support\Str;
use Inside\Validation\Validator;
use Laravel\Lumen\Routing\Pipeline;

#[\AllowDynamicProperties]
final class ContentQueryHelper implements QueryHelper
{
    use HasMimeTypes;

    /**
     * Some mime may have multiple mimes
     *
     * @var array<string,array<int, string>>
     */
    protected array $multiMimes = [
        'csv' => ['txt'],
        'mov' => ['qt'],
    ];

    /**
     * Some mime type may have multiple mime types
     *
     * @var array<string,array<int, string>>
     */
    protected array $multiMimeTypes = [
        'application/x-gzip' => ['application/zip'],
        'application/x-rar-compressed' => ['application/x-rar'],
    ];

    public function __construct(
        protected array $middleware = []
    ) {
    }

    /**
     * Extract fields input
     */
    public function extractFieldsInputFromRequest(Request $request): array
    {
        $fields = [];

        if ($request->isJson() && $request->json()->has('fields')) {
            $fields = json_decode($request->json()->get('fields', ''), true) ?? [];
        } else {
            if ($request->has('fields')) {
                $fields = $request->get('fields', []) ?? [];
                if (is_string($fields)) {
                    $fields = json_decode($fields, true) ?? [];
                }
            }
        }

        return $fields;
    }

    /**
     * Extract filters input
     */
    public function extractFiltersInputFromRequest(Request $request): array
    {
        $filters = [];
        $queryFilter = [];

        if ($request->isJson() && $request->json()->has('filters')) {
            $filters = json_decode($request->json()->get('filters'), true) ?? [];
        }

        if ($request->has('filters')) {
            $queryFilter = $request->get('filters', []);
            if (is_string($queryFilter)) {
                $queryFilter = json_decode($queryFilter, true) ?? [];
            }
        }

        return array_merge($filters, $queryFilter);
    }

    /**
     * Get result from query using pagination or not depending on filters
     *
     * @param  Builder  $query
     * @param  array  $filters
     * @param  bool  $complex
     * @param  int|null  $total
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator|\Illuminate\Contracts\Pagination\Paginator|\Illuminate\Database\Eloquent\Collection|Builder[]
     */
    public function getResultFromQuery(QueryBuilder|Builder $query, array $filters, bool $complex = false, ?int $total = null)
    {
        $limit = null;
        $withPagination = false;

        if (isset($filters['limit'])) {
            $limit = (int) $filters['limit'];
        }
        if (isset($filters['paginate'])) {
            $withPagination = (bool) $filters['paginate'];
        }

        if (! $withPagination) {
            if ($limit !== null) {
                $query->limit($limit);
            }

            if (isset($this->filters['offset'])) {
                $query->offset($this->filters['offset']);
            }

            return match ($query::class) {
                QueryBuilder::class => json_decode(json_encode($query->get()), true),
                default => $query->get(),
            };
        }

        if ($limit !== null) {
            return $complex ? $this->getComplexePagination($query, $limit, $total) : $query->paginate($limit);
        } else {
            return match ($query::class) {
                QueryBuilder::class => json_decode(json_encode($query->get()), true),
                default => $query->get(),
            };
        }
    }

    /**
     * Paginate a collection
     *
     * @param  Collection  $collection
     * @param  int  $limit
     * @param  int|null  $page
     * @param  int|null  $total
     * @return LengthAwarePaginator
     */
    public function getPaginationFromCollection(
        Collection $collection,
        int $limit,
        ?int $page = null,
        ?int $total = null
    ): LengthAwarePaginator {
        $page = $page ?: LengthAwarePaginator::resolveCurrentPage('page');

        return new LengthAwarePaginator(
            $collection->forPage($page, $limit)->values(),
            $total ?: $collection->count(),
            $limit,
            $page,
            [
                'path' => LengthAwarePaginator::resolveCurrentPath(),
                'pageName' => 'page',
            ]
        );
    }

    /**
     * Build a complexe pagination
     *
     * @param  Builder  $query
     * @param  int  $limit
     * @param  int|null  $total
     * @return LengthAwarePaginator|mixed
     */
    public function getComplexePagination(QueryBuilder|Builder $query, int $limit, ?int $total = null)
    {
        $currentPage = Paginator::resolveCurrentPage('page');
        $perPage = $limit;

        $options = [
            'path' => Paginator::resolveCurrentPath(),
            'pageName' => 'page',
        ];

        $items = ($total = ($total === null ? $this->getComplexQuerySize($query) : $total)) ? $query->forPage(
            $currentPage,
            $perPage
        )->get() : ($query instanceof QueryBuilder
            ? json_decode(json_encode($query), true)
            : $query->getModel()->newCollection());

        return Container::getInstance()->makeWith(
            LengthAwarePaginator::class,
            compact(
                'items',
                'total',
                'perPage',
                'currentPage',
                'options'
            )
        );
    }

    /**
     * @param  Builder  $query
     * @return int
     */
    public function getComplexQuerySize(QueryBuilder|Builder $query): int
    {
        $query = clone $query;
        $countQuery = "select count(*) as aggregate from ({$query->toSql()}) c";

        return Arr::first(DB::select($countQuery, $query->getBindings()))->aggregate;
    }

    /**
     * @param  array  $properties
     * @param mixed $result
     * @return mixed
     */
    public function addPropertiesToQueryResult(array $properties, $result)
    {
        $result = $result->toArray();
        foreach ($properties as $key => $value) {
            $result[$key] = $value;
        }

        return $result;
    }

    /**
     * Get a boolean value from a string
     * Returns true when value is "1", "true", "on", and "yes". Otherwise, returns false.
     *
     * @param  string  $value
     * @return bool
     */
    public function requestBoolean(string $value): bool
    {
        return filter_var($value, FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Apply order using front sort setting to query $query
     * @param QueryBuilder|Builder $query
     * @param array $filters
     * @param string $defaultOrder
     * @param string $defaultDirection
     * @param array|null $allowed
     * @return void
     */
    public function applySortToQuery(
        QueryBuilder|Builder $query,
        array $filters,
        string $defaultOrder,
        string $defaultDirection = 'asc',
        ?array $allowed = null
    ): void {
        $sorts = $filters['sort'] ?? $defaultOrder;

        if ($sorts == 'random') {
            $query->inRandomOrder();
        } else {
            if (! is_array($sorts)) {
                $sorts = [$sorts];
            }
            if (empty($sorts)) {
                $query->orderBy($defaultOrder, $defaultDirection);

                return;
            }

            foreach (array_reverse($sorts) as $sort) {
                $direction = 'asc';
                $order = $sort;
                if (strpos($sort, ':') !== false) {
                    [$order, $direction] = explode(':', $sort, 2);
                    if (! in_array(Str::lower($direction), ['asc', 'desc'])) {
                        $direction = 'asc';
                    }
                }
                if ($allowed !== null && ! in_array($order, $allowed)) {
                    continue;
                }
                $query->orderBy($order, $direction);
            }
        }
    }

    /**
     * @param  array  $options
     * @return mixed|null
     */
    protected function getDefaultValue(array $options)
    {
        switch ($options['type']) {
            case 'reference':
                if (! is_array($options['target']) || empty($options['target'])) {
                    return null;
                }
                $target = Arr::first($options['target']);
                $langcode = App::getLocale();
                if ($options['cardinality'] == 1) {
                    if (($default = call_user_func(type_to_class($target).'::query')
                            ->where('uuid_host', $options['default'])
                            ->where('langcode', $langcode)->first()) !== null
                    ) {
                        return $default->uuid;
                    }
                } else {
                    foreach ($options['default'] as &$option) {
                        $reference = call_user_func(type_to_class($target).'::query')->where(
                            'uuid_host',
                            $option
                        )->where('langcode', $langcode)->first();

                        if ($reference) {
                            $option = $reference->uuid;
                        }
                    }
                }
                // no break
            default:
                return $options['default'];
        }
    }

    /**
     * @param mixed $middleware
     * @return array
     */
    protected function gatherMiddlewareClassNames($middleware): array
    {
        $middleware = is_string($middleware) ? explode('|', $middleware) : (array) $middleware;

        return array_map(function ($name) {
            [$name, $parameters] = array_pad(explode(':', $name, 2), 2, null);

            return $name.($parameters ? ':'.$parameters : '');
        }, $middleware);
    }

    /**
     * @param  string  $type
     * @param  array  $data
     * @param  User  $author
     * @param  bool  $skipMiddleware
     * @return Model|mixed
     * @throws Exception
     */
    public function store(string $type, array $data, User $author, bool $skipMiddleware = false)
    {
        // apply middlewares
        $middleware = $this->gatherMiddlewareClassNames($this->middleware);
        if (count($middleware) > 0 && ! $skipMiddleware) {
            $token = Authentication::logAs($author, 'cli');
            /** @var Request $request */
            $request = app('request');

            foreach ([
                'REQUEST_URI' => config('app.url').'/api/v1/contents/'.$type,
                'REQUEST_METHOD' => 'POST',
                'CONTENT_TYPE' => 'application/json',
                'HTTP_ACCEPT' => 'application/json, text/plain',
                'HTTP_REFERER' => config('app.url').'/add/'.$type,
                'HTTP_API_TOKEN' => $token->plainTextToken,
                'HTTP_X_REQUESTED_WITH' => 'XMLHttpRequest',
            ] as $key => $value) {
                $request->server->set($key, $value);
            }

            $request->headers->set('x-requested-with', 'XMLHttpRequest');
            $request->headers->set('api-token', $token->plainTextToken);

            $request->replace($data);

            /** @var JsonResponse $jsonResponse */
            $jsonResponse = app(Pipeline::class)
                ->send($request)
                ->through($middleware)
                ->then(function (Request $data) use ($type, $author) {
                    return response()->json($this->storeContent($type, (array) $data->input(), $author));
                });

            return $jsonResponse->getData();
        }

        return $this->storeContent($type, $data, $author);
    }

    /**
     * @param  string  $type
     * @param  array  $data
     * @param  User  $author
     * @return Model
     * @throws Exception
     */
    protected function storeContent(string $type, array $data, User $author): Model
    {
        // L'auteur est tjrs unique et correspond à une colonne de la table
        // mais le front envoi un tableau ...
        if (isset($data['author']) && is_array($data['author'])) {
            $data['author'] = Arr::first($data['author']);
        }

        $data = $this->addMissings($type, $data);
        $data = $this->castAttributes($type, $data);
        if ($type == 'comments') {  // Special comments case
            // Cleanup front stuff
            unset($data['uid']);
            unset($data['authors']);
            $rules = $this->makeCommentRules($data['bundle']);
        } else {
            $rules = $this->makeRules($type, null, $author->hasAnyRole('super_administrator'));
        }
        $data = $this->validateData($data, $rules, [], $this->getAttributeNames($type));

        $bridge = new BridgeContent();
        $contentUuid = $bridge->contentInsert($type, $data);

        if (is_null($contentUuid)) {
            abort(
                500,
                (string) json_encode(
                    [
                        'error' => 'bridge did not correctly work ',
                        'type' => $type,
                    ]
                )
            );
        }
        $query = call_user_func(type_to_class($type).'::withoutGlobalScopes');

        return $query->findOrFail($contentUuid);
    }

    /**
     * @param  array  $data
     * @param  array  $rules
     * @param  array  $messages
     * @param  array  $customAttributes
     * @return array
     */
    public function validateData(array $data, array $rules, array $messages = [], array $customAttributes = []): array
    {
        /** @var Validator $factory */
        $factory = app('validator');
        $factory->resolver(
            function ($translator, $data, $rules, $messages, $customAttributes) {
                return new Validator($translator, $data, $rules, $messages, $customAttributes);
            }
        );

        return $factory->make(
            $data,
            $rules,
            $messages,
            $customAttributes
        )->validate();
    }

    /**
     * Add missing and required data if possible
     *
     * @param  string  $type
     * @param  array  $data
     * @param  bool  $creation
     * @return array
     */
    public function addMissings(string $type, array $data, bool $creation = true): array
    {
        $me = Auth::user();

        if (! isset($data['bundle'])) {
            $data['bundle'] = ($type == 'users') ? 'user' : $type;
        }

        if (! isset($data['type'])) {
            $data['type'] = ($type == 'users') ? 'user' : 'node';
        }

        if ($creation && ! isset($data['langcode'])) {
            $data['langcode'] = App::getLocale();
        }

        // Fix author
        if (isset($data['authors'])) {
            $data['author'] = is_array($data['authors']) ? Arr::first($data['authors']) : $data['authors'];
            unset($data['authors']);
        }

        if ($creation) {
            foreach (Schema::getFieldListing($type) as $fieldName) {
                if (in_array($fieldName, ['created_at', 'updated_at', 'published_at', 'status', 'langcode'])) {
                    continue;
                }
                $fieldOptions = Schema::getFieldOptions($type, $fieldName);
                if ($fieldOptions['required'] && ! array_key_exists($fieldName, $data)) {
                    $data[$fieldName] = $this->getDefaultValue($fieldOptions);
                }
            }
            $data['created_at'] = now()->format('Y-m-d H:i:s');
            if ($type !== 'users' && ! isset($data['published_at'])) {
                $data['published_at'] = $data['created_at'];
            }
        }

        isset($data['status']) || $data['status'] = true;

        if ($type == 'users') {
            return $data;
        }

        if (! $me) {
            $data['status'] = false;
        } else {
            if (empty($data['author'])) {
                $data['author'] = $me->uuid;
            }
            if (empty($data['update_author'])) {
                $data['update_author'] = $me->uuid;
            }
        }
        $data['updated_at'] = $creation ? $data['created_at'] : now()->format('Y-m-d H:i:s');

        return $data;
    }

    /**
     * Correctly cast field data ( front could give an int to save a bool, it will dirty data
     * while it should not )
     *
     * @param  string  $type
     * @param  array  $data
     * @return array
     */
    public function castAttributes(string $type, array $data): array
    {
        $fieldNames = Schema::getFieldListing($type);
        foreach ($fieldNames as $fieldName) {
            if (isset($data[$fieldName])) {
                $options = Schema::getFieldOptions($type, $fieldName);

                switch ($options['type']) {
                    case 'checkbox':
                    case 'boolean':
                        $data[$fieldName] = in_array($data[$fieldName], [true, '1', 'true', 1, 'on'], true);
                        break;
                    case 'integer':
                        $data[$fieldName] = (int) $data[$fieldName];
                        break;
                    case 'double':
                    case 'float':
                        $data[$fieldName] = (float) $data[$fieldName];
                        break;
                    case 'text':
                        $data[$fieldName] = match ($options['widget']) {
                            'phone' => preg_replace('/[^\d+]/', '', (string) $data[$fieldName]),
                            'inside_link_field_widget' => preg_replace(
                                '#^'.trim(config('app.url'), '/').'#',
                                '',
                                (string) $data[$fieldName]
                            ),
                            default => (string) $data[$fieldName],
                        };
                        break;
                    case 'wysiwyg':
                    case 'textarea':
                        $data[$fieldName] = (string) $data[$fieldName];
                        break;
                    case 'section':
                        foreach ($data[$fieldName] as $key => $sectionData) {
                            // Manage special weird front pgID
                            if (isset($sectionData['pgID']) && ! Str::isUuid($sectionData['pgID'])) {
                                $sectionData['pgID'] = null;
                            }
                            $data[$fieldName][$key] = $this->castAttributes($sectionData['bundle'], $sectionData);
                        }
                        break;
                    case 'reference':
                        // Fix front not sending correct data on references
                        if ($options['cardinality'] == 1 && is_array($data[$fieldName])) {
                            $data[$fieldName] = Arr::first($data[$fieldName]);
                        } elseif ($options['cardinality'] == -1 && Str::isUuid($data[$fieldName])) {
                            $data[$fieldName] = [$data[$fieldName]];
                        }
                        break;
                }
            }
        }

        return $data;
    }

    /**
     * Get attribute name so that we can get a nice validation error
     *
     * @param  string  $type
     * @param  string|null  $locale
     * @return array
     */
    public function getAttributeNames(string $type, ?string $locale = null): array
    {
        $attributes = [];
        if ($locale === null) {
            $locale = App::getLocale();
        }
        $fieldNames = Schema::getFieldListing($type);
        foreach ($fieldNames as $fieldName) {
            $options = Schema::getFieldOptions($type, $fieldName);
            $attributes[$fieldName] = $options['title'][$locale] ?? Arr::first($options['title']);
        }

        return $attributes;
    }

    /**
     * Get rules from options for $fieldName
     *
     * @param  string  $type
     * @param  string  $fieldName
     * @param  array  $options
     * @param  array  $rules
     * @param  string|null  $uuid
     * @param  string|null  $prefix
     * @param  bool  $force
     * @return array|null
     */
    protected function getRulesFromOptions(
        string $type,
        string $fieldName,
        array $options,
        array &$rules,
        ?string $uuid = null,
        ?string $prefix = null,
        bool $force = false
    ): ?array {
        $fieldRules = [];
        if ($uuid !== null && isset($options['required']) && (bool) $options['required']) {
            $fieldRules[] = 'sometimes';
        }
        if (isset($options['required']) && (bool) $options['required']) {
            if ($prefix !== null) {
                $fieldRules[] = 'required_if:'.$prefix.'bundle,'.$type;
            } else {
                $fieldRules[] = 'required';
            }
        } else {
            $fieldRules[] = 'nullable';
        }
        if ($type === 'users' && $fieldName === 'password' && $uuid !== null) {
            return null;
        }
        if (! $force && $type === 'users' && ! in_array($fieldName, ['image'])
            && (! isset($options['editable'])
                || ! $options['editable'])
        ) {
            return null;
        }
        if ($fieldName === 'authors') {
            return null;
        }
        switch ($options['type']) {
            case 'select':
                // Special case langcode
                if ($fieldName == 'langcode') {
                    $fieldRules[] = Rule::in(list_languages());
                }
                break;
            case 'list_float':
            case 'list_integer':
            case 'list_string':
                if (isset($options['allowed_values']) && ! empty($options['allowed_values'])) {
                    $fieldRules[] = Rule::in(array_keys(Arr::first($options['allowed_values'])));
                } else {
                    $fieldRules[] = Rule::in([]);
                }
                break;
            case 'email':
                if (config('app.secured_email_validation', true)) {
                    $fieldRules[] = 'emailpp:rfc,dns';
                } else {
                    $fieldRules[] = 'email';
                }
                break;
            case 'integer':
                $fieldRules[] = 'integer';
                break;
            case 'uri':
                if ($options['widget_type'] === 'inside_link_field_widget') {
                    $fieldRules[] = new LinkRule();
                }
                break;
            case 'text':
                switch ($options['widget_type']) {
                    case 'inside_link_field_widget':
                        $fieldRules[] = new LinkRule();
                        break;
                    case 'phone':
                        $fieldRules[] = 'phone';
                        break;
                }
                if (isset($options['max_length']) && is_int($options['max_length']) && $options['max_length'] > 0) {
                    $fieldRules[] = 'max:'.(int) $options['max_length'];
                } else {
                    $fieldRules[] = 'max:255';
                }
                // no break
            case 'wysiwyg':
            case 'textarea':
                $fieldRules[] = 'string';
                break;
            case 'link':
                break;
            case 'section':
                // $uuid
                $rules[$fieldName.'.*.bundle'] =
                    Rule::in(array_intersect($options['target'], Schema::getSectionTypes()));
                $fieldRules[] = new SectionRule();
                break;
            case 'boolean':
            case 'checkbox':
                $fieldRules[] = 'boolean';
                break;
            case 'datetime':
            case 'timestamp':
                $fieldRules[] = 'date_format:Y-m-d H:i:s';
                break;
            case 'reference':
                if ($options['cardinality'] == -1) {
                    $fieldRules[] = 'array';
                    $fieldRules[] = function ($attribute, $value, $fail) use ($options) {
                        foreach ($value as $uuid) {
                            if (! Str::isUuid($uuid)) {
                                $fail($attribute.' contains an invalid uuid');

                                return;
                            }
                            if (! isset($options['target']) || count($options['target']) != 1) {
                                $fail($attribute.'\'s target is not correclty set');

                                return;
                            }
                            $target = Arr::first($options['target']);
                            $result = call_user_func(type_to_class($target).'::find', $uuid);
                            if ($result === null) {
                                $fail($attribute.' contains an invalid uuid');

                                return;
                            }
                        }
                    };
                } elseif ($options['cardinality'] == 1) {
                    $fieldRules[] = 'uuid';
                    $fieldRules[] = function ($attribute, $value, $fail) use ($options) {
                        if (! isset($options['target']) || count($options['target']) != 1) {
                            $fail($attribute.'\'s target is not correclty set');

                            return;
                        }
                        $target = Arr::first($options['target']);
                        $result = call_user_func(type_to_class($target).'::find', $value);
                        if ($result === null) {
                            $fail($attribute.' contains an invalid uuid');

                            return;
                        }
                    };
                }
                break;
            case 'image':
                $fieldRules[] = 'image';
                $dimensions = '';
                foreach (['max_width', 'max_height', 'min_width', 'min_height'] as $key) {
                    if (isset($options[$key]) && is_int($options[$key])) {
                        if (! empty($dimensions)) {
                            $dimensions .= ',';
                        }
                        $dimensions .= $key.'='.$options[$key];
                    }
                }
                if (! empty($dimensions)) {
                    $fieldRules[] = 'dimensions:'.$dimensions;
                }
                // no break
            case 'file':
                $fieldRules[] = 'mimetypes:'.implode(
                    ',',
                    $this->getMimeTypesFromMimes($options['mimetypes'] ?? 'image/png,image/gif,image/jpeg')
                );
                $fieldRules[] = 'mimes:'.implode(
                    ',',
                    $this->getMimesFromExtensions($options['extensions'] ?? 'png gif jpg jpeg')
                );
                if (isset($options['max_filesize'])) {
                    $fieldRules[] = 'maxfilesize:'.$options['max_filesize'];
                }
                $fieldRules[] = function ($attribute, $value, $fail) {
                    if (is_string($value) && Storage::disk('local')->exists($value)) {
                        [$primaryFolder, $after] = explode('/', $value, 2);

                        // Check that file is from one of our upload dirs
                        return in_array($primaryFolder, ['chunks', 'fakes']); //TODO: make this configurable
                    }

                    return true; // We aldready know that file doesn't exist from previous rules
                };
                break;
        }

        return $fieldRules;
    }

    /**
     * Get a mime array from drupal extension config
     *
     * @param  string  $extensions
     * @return array
     */
    protected function getMimesFromExtensions(string $extensions): array
    {
        $mimes = explode(' ', $extensions);
        $result = $mimes;
        foreach ($mimes as $mime) {
            // We got a multi mime, add other possible extension for $mime
            if (isset($this->multiMimes[$mime])) {
                $result = array_merge($result, $this->multiMimes[$mime]);
            }
        }

        return array_unique($result);
    }

    /**
     * Get a mime types array from drupal mime types config
     *
     * @param  string  $mimes
     * @return array
     */
    protected function getMimeTypesFromMimes(string $mimes): array
    {
        $mimeTypes = explode(',', $mimes);
        $result = $mimeTypes;

        foreach ($mimeTypes as $type) {
            if (isset($this->multiMimeTypes[$type])) {
                $result = array_merge($result, $this->multiMimeTypes[$type]);
            }
            $extensions = $this->getExtensions($type);
            foreach ($extensions as $extension) {
                $result = array_merge($result, $this->getMimeTypes($extension));
            }
        }

        return array_unique($result);
    }

    /**
     * Make validation comments rules
     *
     * @param  string  $type
     * @return array
     */
    public function makeCommentRules(string $type): array
    {
        // Special comment rules
        $rules = [];

        foreach (['body', 'file', 'langcode', 'status'] as $fieldName) {
            $options = Schema::getFieldOptions('comments', $fieldName);
            $fieldRules = [];

            if (($fieldRules = $this->getRulesFromOptions('comments', $fieldName, $options, $rules)) !== null) {
                $rules[$fieldName] = $fieldRules;
            }
        }
        $commentables = Schema::getContentTypes(
            function ($options) {
                return array_key_exists('comments', $options['fields']);
            }
        );
        $rules['bundle'] = [Rule::in($commentables)];
        $rules[$type] = [
            'array',
            function ($attribute, $value, $fail) use ($type) {
                foreach ($value as $uuid) {
                    if (! Str::isUuid($uuid)) {
                        $fail($attribute.' contains an invalid uuid');

                        return;
                    }
                    $result = call_user_func(type_to_class($type).'::find', $uuid);
                    if ($result === null) {
                        $fail($attribute.' contains an invalid uuid');

                        return;
                    }
                }
            },
        ];
        $rules['author'] = ['uuid'];
        $rules['update_author'] = ['uuid'];
        $rules['pid'] = [
            'nullable',
            function ($attribute, $uuid, $fail) {
                if (! Str::isUuid($uuid)) {
                    $fail($attribute.' contains an invalid uuid');

                    return;
                }
                $result = Comments::find($uuid);
                if ($result === null) {
                    $fail($attribute.' contains an invalid uuid');

                    return;
                }
            },
        ];

        return $rules;
    }

    /**
     * Make validation rules for content type $type
     *
     * @param  string  $type
     * @param  string  $uuid
     * @param  bool  $force
     * @return array
     */
    public function makeRules(string $type, ?string $uuid = null, bool $force = false): array
    {
        $rules = [];

        if ($type === 'users') {
            // Users are really special and need some attention!
            if ($uuid === null) {
                $uniqueUsernameRule = Rule::unique('inside_users', 'name');
                $uniqueEmailRule = Rule::unique('inside_users', 'email');
            } else {
                $uniqueUsernameRule = Rule::unique('inside_users', 'name')->ignore($uuid, 'uuid');
                $uniqueEmailRule = Rule::unique('inside_users', 'email')->ignore($uuid, 'uuid');
            }
            $rules['name'] = ['required', 'bail', 'min:3', 'max:255', $uniqueUsernameRule];
            if (config('app.secured_email_validation', true)) {
                $rules['email'] = ['required', 'emailpp:rfc,dns', $uniqueEmailRule];
            } else {
                $rules['email'] = ['required', 'email', $uniqueEmailRule];
            }
            if ($uuid !== null) {
                $rules['name'] = array_prepend($rules['name'], 'sometimes');
                $rules['email'] = array_prepend($rules['email'], 'sometimes');
            }
        }

        // Add system rules
        $rules['bundle'] = [Rule::in(array_merge(['user'], Schema::getContentTypes()))];
        $rules['type'] = [Rule::in(['node', 'paragraph', 'user'])];
        if ($uuid !== null) {
            $rules['bundle'][] = 'required';
            $rules['type'][] = 'required';
            $rules['uuid'] = 'uuid';
        }
        $rules['author'] = ['uuid'];
        $rules['update_author'] = ['uuid'];
        $rules['langcode'] = ['locale'];
        $rules['published_at'] = 'date_format:Y-m-d H:i:s';

        foreach (Schema::getFieldListing($type) as $fieldName) {
            if ($type === 'users' && in_array($fieldName, ['name', 'email'])) {
                continue;
            }
            $options = Schema::getFieldOptions($type, $fieldName);

            if (($fieldRules = $this->getRulesFromOptions($type, $fieldName, $options, $rules, $uuid, null, $force))
                !== null
            ) {
                $rules[$fieldName] = $fieldRules;
            }
        }

        return $rules;
    }

    /**
     * @param mixed $data
     * @param string $locale
     * @param bool $fallback  if true, if translation does not exists, keep current content
     * @return mixed
     */
    public function getContentsInLocale($data, string $locale, bool $fallback = false)
    {
        foreach ($data as $key => &$content) {
            if ($content->langcode === $locale) {
                continue;
            }
            /** @var Content $content */
            $content = $content->getTranslationIfExists($locale);
            if ($fallback === false && $content->langcode !== $locale) {
                unset($data[$key]);
            } else {
                $data[$key] = $content;
            }
        }

        return $data;
    }

    /**
     * @param  Request  $request
     * @param  string  $key
     * @return mixed|null
     */
    public function getAndRemoveQuery(Request &$request, string $key): mixed
    {
        $inputs = $request->all();
        if (! array_key_exists($key, $inputs)) {
            return null;
        }
        $value = $inputs[$key];
        unset($inputs[$key]);

        $request->replace($inputs);

        return $value;
    }

    /**
     * Remove $keyName from request $group
     *
     * @param  Request  $request
     * @param  string  $group
     * @param  string  $keyName
     * @return Request
     */
    public function removeInputFromRequest(Request $request, string $group, string $keyName): Request
    {
        $inputs = $request->all();
        if (! isset($inputs[$group])) {
            return $request;
        }
        $values = json_decode($inputs[$group], true) ?? [];
        if ($values === null) {
            return $request;
        }
        foreach ($values as $key => $value) {
            if (is_int($key)) {
                if ((is_string($value) && $value == $keyName)
                    || (is_array($value) && ! empty($value) && Arr::first($value) == $keyName)
                ) {
                    unset($values[$key]);
                }
            } else {
                if ($key == $keyName) {
                    unset($values[$keyName]);
                }
            }
        }
        if (is_object($values)) {
            unset($values->{$keyName});
        } elseif (is_array($values)) {
            foreach ($values as $key => $value) {
                if (is_object($value) && isset($value->{$keyName})) {
                    unset($value->{$keyName});
                } else {
                    if ($value == $keyName) {
                        unset($values[$key]);
                    }
                }
            }
        }
        if (is_int(Arr::first(array_keys($values)))) {
            $values = array_values((array) $values);
        }

        $inputs[$group] = json_encode($values);

        $request->replace($inputs);

        return $request;
    }

    /**
     * Extract operator (eq, gte, gt...)
     *
     * @param  string  $operator
     * @return string
     */
    public function getOperatorFromString(string $operator): ?string
    {
        switch ($operator) {
            case 'in':
                return 'in';
            case 'notin':
                return 'not in';
            case 'ne':
                return '!=';
            case 'eq':
                return '=';
            case 'gt':
                return '>';
            case 'gte':
                return '>=';
            case 'lt':
                return '<';
            case 'lte':
                return '<=';
            case 'like':
                return 'like';
            case 'null':
                return 'null';
            case 'notnull':
                return 'notnull';
        }

        return '=';
    }

    public function find(string $type, string $uuid): ?Content
    {
        if (! Schema::hasContentType($type)) {
            throw ModelSchemaNotFoundException::named($type);
        }

        return call_user_func(type_to_class($type).'::find', $uuid);
    }

    public function findOrFail(string $type, string $uuid): Content|Section
    {
        if (! Schema::hasContentType($type) && ! Schema::hasSectionType($type)) {
            throw ModelSchemaNotFoundException::named($type);
        }

        return call_user_func(
            (Schema::isContentType($type) ? type_to_class($type) : section_type_to_class($type)).'::findOrFail',
            $uuid
        );
    }

    public function all(string $type): \Illuminate\Database\Eloquent\Collection
    {
        if (! Schema::hasContentType($type)) {
            throw ModelSchemaNotFoundException::named($type);
        }

        return call_user_func(type_to_class($type).'::get');
    }
}
