<?php

declare(strict_types=1);

namespace Inside\Content\Services\Queries;

use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Inside\Content\Contracts\ContentExternalList as ContentExternalListContract;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Transformers\ContentTransformer;

final class ContentExternalList implements ContentExternalListContract
{
    public function __construct(
        private ContentTransformer $transformer
    ) {
    }

    public function __invoke(string $type, Request $request, array $fields): LengthAwarePaginator|array
    {
        $page = (int) $request->get('page', '1');
        $limit = $this->getLimit($request);
        $paginate = $this->shouldPaginate($request);

        $first = $limit * ($page - 1);
        $last = $limit * $page;

        $models = $this
            ->getModels($type, $request)
            ->map(fn ($model, int $index) => $index >= $first && $index < $last
                ? $this->transformer->transform($model, $fields)
                : []
            );

        return $paginate ?
            ContentHelper::getPaginationFromCollection(
                collection: $models,
                limit: $limit,
                page: $page
            ) :
            $models->toArray();
    }

    protected function getModels(string $type, Request $request): Collection
    {
        $filters = config('external_api.inside_external_api_filters');

        if (! is_null($filters) && ! is_callable($filters)) {
            throw new \Exception('The filters config must be a callable');
        }

        return type_to_class($type)::query()
            ->when(is_callable($filters), fn (Builder $query) => $filters($query, $request))
            ->get();
    }

    protected function getLimit(Request $request): int
    {
        if ($request->has('limit')) {
            return (int) $request->get('limit');
        }

        return (int) config('external_api.inside_external_api_pagination_limit', '20');
    }

    protected function shouldPaginate(Request $request): bool
    {
        if ($request->has('paginate')) {
            return (bool) $request->get('paginate');
        }

        return (bool) config('external_api.inside_external_api_paginate_result', true);
    }
}
