<?php

namespace Inside\Content\Services;

use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Inside\Content\Models\InsidePages;

class InsidePagesService
{
    public static function listAllByPath(string $path, array $filters = []): Collection
    {
        $query = InsidePages::query();
        collect($filters)->each(fn ($value, $key) => $query->where($key, $value));

        return $query->where('path', $path)
            ->select('id', 'title', 'status', 'path', 'created_at', 'updated_at', 'json', 'langcode')
            ->orderByDesc('status')
            ->get();
    }

    public static function getPageById(int $id): InsidePages
    {
        return InsidePages::findOrFail($id);
    }

    public static function getPublishedPageByPath(string $path, string $langcode): ?InsidePages
    {
        return InsidePages::query()
            ->where('path', $path)
            ->where('status', 1)
            ->where('langcode', $langcode)
            ->first();
    }

    public static function deleteDraftPageById(int $id): ?bool
    {
        $page = InsidePages::findOrFail($id);

        if ($page->status === true) {
            throw new \Exception('You cannot delete a published page.');
        }

        return $page->delete();
    }

    public static function publishPage(int $id): InsidePages
    {
        $page = InsidePages::findOrFail($id);

        InsidePages::query()->where('path', $page->path)
            ->where('status', true)
            ->where('langcode', $page->langcode)
            ->update(['status' => false]);

        $page->status = true;
        $page->save();

        return $page;
    }

    public static function createOrUpdate(array $data): InsidePages
    {
        $status = (bool) ($data['status'] ?? false);
        $path = $data['path'];
        $langcode = $data['langcode'];
        $id = $data['id'] ?? null;

        if (is_null($id) && $status === false && self::isLastPublishedPageForPath($path, $langcode)) {
            throw new \Exception('You cannot set status to false. At least one page must remain published.');
        }

        $page = $id ? InsidePages::findOrFail($id) : new InsidePages();

        if (empty($page->title) && empty($data['title'])) {
            $count = InsidePages::query()
                ->where('path', $path)
                ->where('langcode', $langcode)
                ->count();
            $title = 'Page '.($count + 1);
        } else {
            $title = $data['title'] ?? $page->title;
        }

        $payload = [
            'title'  => $title,
            'status' => $status,
            'path'   => $path,
            'langcode'   => $langcode,
        ];

        if (array_key_exists('json', $data)) {
            $payload['json'] = $data['json'];
        } elseif (! $id) {
            $payload['json'] = (object) [];
        }

        $page->fill($payload)->save();

        return $page;
    }

    public static function duplicatePage(int $id): InsidePages
    {
        $originalPage = InsidePages::findOrFail($id);

        $duplicatedPage = $originalPage->replicate();

        $originalJson = $duplicatedPage->getAttribute('json');

        $newJson = self::updateJsonIds(json_decode($originalJson, true));

        $duplicatedPage->setAttribute('json', json_encode($newJson));

        $duplicatedPage->setAttribute('status', false);

        $duplicatedPage->setAttribute('title', $originalPage->title.' - '.__('inside.pages.duplicate'));

        $duplicatedPage->save();

        return $duplicatedPage;
    }

    protected static function isLastPublishedPageForPath(string $path, string $langcode): bool
    {
        $totalPages = InsidePages::query()
            ->where('path', $path)
            ->where('langcode', $langcode)
            ->count();

        if ($totalPages === 0) {
            return false;
        }

        $publishedPages = InsidePages::query()
            ->where('path', $path)
            ->where('langcode', $langcode)
            ->where('status', true)
            ->count();

        return $publishedPages === 0;
    }

    public static function getPageCacheKeyFromPath(string $path, string $langcode): string
    {
        $sanitizedPath = str_replace(['/', '\\', ' '], '_', strtolower(trim($path))).'_'.$langcode;

        return 'inside_pages:'.$sanitizedPath;
    }

    public static function updateJsonIds(array $data): array
    {
        return collect($data)->map(function ($item) {
            if (is_array($item)) {
                $item = collect($item)->map(function ($value, $key) {
                    if ($key === 'id') {
                        return '_'.Str::random(9);
                    }

                    if (is_array($value)) {
                        return self::updateJsonIds($value);
                    }

                    return $value;
                })->toArray();
            }

            return $item;
        })->toArray();
    }
}
