<?php

namespace Inside\Content\Services\Importers;

use Closure;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Maatwebsite\Excel\Concerns\Importable;
use Maatwebsite\Excel\Concerns\OnEachRow;
use Maatwebsite\Excel\Concerns\WithCustomCsvSettings;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithProgressBar;
use Maatwebsite\Excel\Row;

/**
 * Class ContentImport
 */
class ContentImport implements OnEachRow, WithProgressBar, WithHeadingRow, WithCustomCsvSettings
{
    use Importable;

    protected BridgeContent $bridge;

    public function __construct(
        protected array $config
    ) {
        $this->bridge = new BridgeContent();
    }

    public function onRow(Row $row): void
    {
        $rowIndex = $row->getIndex();
        $record = $row->toArray();
        try {
            $datas = [
                'type'   => 'node',
                'bundle' => $this->config['content_type'],
            ];

            // Check existence
            $query = call_user_func(type_to_class($this->config['content_type']).'::query');
            $content = $query->where($this->config['key']['inside'], $record[$this->config['key']['import']])->first();
            if ($content) {
                $datas['uuid_host'] = $content->uuid_host;
            }
            foreach ($this->config['fields'] as $fieldName => $fieldEquivalent) {
                if (($fieldEquivalent instanceof Closure)) {
                    $datas[$fieldName] = $fieldEquivalent($record, $this->bridge);
                    if ($datas[$fieldName] === null) {
                        unset($datas[$fieldName]);
                    }
                } else {
                    $fieldEquivalent = Str::slug($fieldEquivalent);
                    if (isset($record[$fieldEquivalent])) {
                        $datas[$fieldName] = $record[$fieldEquivalent];
                    }
                }
            }

            if (empty($datas['author'])) {
                $datas['author'] = Users::where('email', config('app.technical_mail'))->firstOrFail()->uuid;
            }

            if (! isset($datas['status'])) {
                $datas['status'] = true;
            }

            if (null === ($uuid = $this->bridge->contentUpdate($this->config['content_type'], $datas))) {
                Log::error(
                    '[Inside::import] Failed to Create / Insert Drupal Type {'.$this->config['content_type'].'}  ['
                    .json_encode($datas).']'
                );
            }
            $finishing = $this->config['finishing'];
            if ($finishing && is_callable($finishing)) {
                $finishing($record, $this->bridge, $uuid, $this->getConsoleOutput());
            }
        } catch (Exception $e) {
            $this->getConsoleOutput()->error($e->getMessage());
            Log::error($e->getMessage());
        }
    }

    public function getCsvSettings(): array
    {
        return [
            'delimiter'              => $this->config['format']['options']['delimiter'] ?? ';',
            'enclosure'              => $this->config['format']['options']['enclosure'] ?? '"',
            'line_ending'            => $this->config['format']['options']['line_ending'] ?? PHP_EOL,
            'use_bom'                => $this->config['format']['options']['use_bom'] ?? false,
            'include_separator_line' => $this->config['format']['options']['include_separator_line'] ?? false,
            'excel_compatibility'    => $this->config['format']['options']['excel_compatibility'] ?? false,
        ];
    }
}
