<?php

namespace Inside\Content\Services;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Inside\Content\Services\Importers\CsvImporter;
use Inside\Content\Services\Importers\ExcelImporter;
use Inside\Content\Services\Importers\GnumericImporter;
use Inside\Content\Services\Importers\HtmlImporter;
use Inside\Content\Services\Importers\ImporterInterface;
use Inside\Content\Services\Importers\OdsImporter;
use Inside\Content\Services\Importers\SlkImporter;
use Inside\Content\Services\Importers\TsvImporter;
use Inside\Content\Services\Importers\XlsxImporter;
use Inside\Content\Services\Importers\XmlImporter;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Finder\Finder;

class Importer
{
    protected array $drivers = [];

    protected array $importers = [];

    public function __construct()
    {
        $this->loadDrivers();
    }

    protected function loadDrivers(): void
    {
        // Load buildin exporters
        $this->extend('csv', CsvImporter::class);
        $this->extend('excel', ExcelImporter::class);
        $this->extend('gnumeric', GnumericImporter::class);
        $this->extend('html', HtmlImporter::class);
        $this->extend('ods', OdsImporter::class);
        $this->extend('slk', SlkImporter::class);
        $this->extend('tsv', TsvImporter::class);
        $this->extend('xlsx', XlsxImporter::class);
        $this->extend('xml', XmlImporter::class);

        // Load custom exporters
        $customDirectory = back_path('src/Importers');
        if (File::exists($customDirectory)) {
            foreach (
                Finder::create()->files()->name('*Exporter.php')->in(back_path('src/Importers')) as $file
            ) {
                $this->extend(
                    Str::lower(Str::before($file->getBasename(), 'Importer.php')),
                    'Inside\\'.Str::upper(env('APP_CODE')).'\\Importers\\'.$file->getBasename(
                        '.'.$file->getExtension()
                    )
                );
            }
        }
    }

    public function hasDriver(string $driver): bool
    {
        return isset($this->drivers[Str::lower($driver)]);
    }

    public function getDrivers(): array
    {
        return array_keys($this->drivers);
    }

    public function extend(string $name, string $class): void
    {
        $this->drivers[Str::lower($name)] = $class;
    }

    protected function getImporter(string $driver): mixed
    {
        $driver = Str::lower($driver);
        if (isset($this->importers[$driver])) {
            return $this->importers[$driver];
        }

        if (! $this->hasDriver($driver)) {
            return $this->importers[$driver] = $this->getDefaultImporter();
        }

        return $this->importers[$driver] = new $this->drivers[$driver]();
    }

    protected function getDefaultImporter(): ImporterInterface
    {
        return new CsvImporter();
    }

    public function import(string $driver, array $config, OutputInterface $console = null): bool
    {
        return $this->getImporter($driver)->import($config, $console);
    }
}
