<?php

namespace Inside\Content\Services;

use Illuminate\Contracts\Filesystem\Filesystem;
use Illuminate\Database\Query\Builder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Models\Contents\Documents;
use Inside\Content\Models\Contents\Folders;
use Inside\Content\Models\Contents\Tools;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use PharData;
use ZipArchive;

/**
 * Class GedExportService
 */
class GedExportService
{
    const FOLDER = 'folder';

    private Filesystem $disk;

    private string $exportPath;

    private string $format = 'zip';

    private ?string $rootFolder;

    private array $metadata = [];

    private array $authors = [];

    public function execute(): void
    {
        $this->disk = Storage::disk('local');
        $this->addPath($this->exportPath);
        $this->loadAuthors();

        $gedTree = $this->gedTree();
        $metadata = [];

        $gedTree->each(function ($folder) use (&$metadata) {
            $folderPath = $this->exportPath.'/'.str_replace('/', '-', $folder->title);
            $this->addPath($folderPath);
            $this->exportFolders($folder, $folderPath);
            $metadata = [
                'status' => $folder->status ? 'published' : 'unpublished',
                'author_name' => $this->authors[$folder->author]['name'],
                'author_email' => $this->authors[$folder->author]['email'],
                'created_at' => get_date($folder->created_at)?->format('d-m-Y H:i:s'),
                'updated_at' => get_date($folder->updated_at)?->format('d-m-Y H:i:s'),
                'type' => self::FOLDER,
                'intranet_name' => $folder->title,
                'original_name' => $folder->title,
            ];
        });

        $this->disk->put($this->exportPath.'/metadata.json', (string) json_encode($metadata, JSON_PRETTY_PRINT));

        $this->format === 'zip' ? $this->archiveDirectoryAsZip() : $this->archiveDirectoryAsTar();
    }

    public function setExportPath(string $exportPath): self
    {
        $this->exportPath = $exportPath;

        return $this;
    }

    public function setRootFolder(?string $rootFolder): self
    {
        $this->rootFolder = $rootFolder;

        return $this;
    }

    public function setFormat(string $format): self
    {
        $this->format = $format;

        return $this;
    }

    private function gedTree(): Collection
    {
        /**
         * @var Builder $foldersQuery
         */
        $foldersQuery = type_to_class('folders')::query()
            ->where([
                'pid' => null,
                'status' => true,
            ]);

        if (! is_null($this->rootFolder)) {
            $foldersQuery->where('uuid', $this->rootFolder);
        }

        return $foldersQuery->get();
    }

    private function exportFolders(Folders $parentFolder, string $folderPath): void
    {
        $subFolders = $parentFolder->getChildrenIfExist();
        $this->metadata = [];

        $subFolders->each(function ($folder) use ($folderPath) {
            $folderPath = $folderPath.'/'.str_replace('/', '-', $folder->title);
            $this->addPath($folderPath);
            $this->exportFolders($folder, $folderPath);
            $this->extractFolderMetadata($folder);
        });

        $this->exportDocuments($parentFolder, $folderPath);

        $this->disk->put($folderPath.'/metadata.json', (string) json_encode($this->metadata, JSON_PRETTY_PRINT));
    }

    private function exportDocuments(Folders $parentFolder, string $folderPath): void
    {
        $documents = $parentFolder->reverseDocuments();
        if ($documents->count() < 1) {
            return;
        }

        $documents->each(function ($document) use ($folderPath) {
            $copyPath = $folderPath.'/'.pathinfo($document->file, PATHINFO_BASENAME);
            if (! $this->disk->exists($copyPath)) {
                $this->disk->copy($document->file, $copyPath);
            }
            $this->extractDocumentMetadata($document);
        });
    }

    private function extractFolderMetadata(Folders $folder): void
    {
        $this->metadata[] = [
            'status' => $folder->status ? 'published' : 'unpublished',
            'author_name' => $this->authors[$folder->author]['name'],
            'author_email' => $this->authors[$folder->author]['email'],
            'created_at' => get_date($folder->created_at)?->format('d-m-Y H:i:s'),
            'updated_at' => get_date($folder->updated_at)?->format('d-m-Y H:i:s'),
            'type' => self::FOLDER,
            'intranet_name' => $folder->title,
            'original_name' => $folder->title,
        ];
    }

    private function extractDocumentMetadata(Documents $document): void
    {
        $this->metadata[] = [
            'status' => $document->status ? 'published' : 'unpublished',
            'author_name' => $this->authors[$document->author]['name'],
            'author_email' => $this->authors[$document->author]['email'],
            'created_at' => get_date($document->created_at)?->format('d-m-Y H:i:s'),
            'updated_at' => get_date($document->updated_at)?->format('d-m-Y H:i:s'),
            'type' => pathinfo($document->file, PATHINFO_EXTENSION),
            'intranet_name' => $document->title,
            'original_name' => pathinfo($document->file, PATHINFO_BASENAME),
        ];
    }

    private function loadAuthors(): void
    {
        $this->authors = Users::all()->mapWithKeys(fn ($user) => [
            $user->uuid => ['email' => $user->email, 'name' => $user->firstname.' '.$user->lastname],
        ])->toArray();
    }

    private function addPath(string $path): void
    {
        if (! $this->disk->exists($path)) {
            $this->disk->makeDirectory($path);
        }
    }

    private function archiveDirectoryAsTar(): void
    {
        $outputPath = $this->disk->path('/');
        $tarFile = $this->disk->path('ged_export.tar');
        $phar = new PharData($tarFile);
        $phar->buildFromDirectory($this->disk->path($this->exportPath));
        $phar->compress(\Phar::GZ);
        unlink($tarFile);
    }

    private function archiveDirectoryAsZip(): void
    {
        $sourcePath = $this->disk->path($this->exportPath);
        $archivePath = $this->disk->path('ged_export.zip');
        $zip = new ZipArchive();
        if ($zip->open($archivePath, ZipArchive::CREATE | ZipArchive::OVERWRITE) === true) {
            $files = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator($sourcePath),
                \RecursiveIteratorIterator::SELF_FIRST
            );

            foreach ($files as $file) {
                if (in_array($file->getFilename(), ['.', '..'])) {
                    continue;
                }

                $relativePath = substr($file->getRealPath(), strlen($sourcePath) + 1);

                if ($file->isDir()) {
                    $zip->addEmptyDir($relativePath);
                } else {
                    $zip->addFile($file->getRealPath(), $relativePath);
                }
            }

            $zip->close();
        }
    }
}
