<?php

namespace Inside\Content\Services\Exporters;

use Exception;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Exceptions\FieldSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Contents\Users;
use PhpOffice\PhpSpreadsheet\Shared\Date;

trait ContentExportable
{
    /**
     * @throws Exception
     */
    protected function addReferenceColumn(Model $information, int $cardinality, string $type, string $column, array $fieldOptions, array &$mapped): void
    {
        if ($cardinality == 1) {
            $data = $information->{Str::camel($column)};
            if ($data instanceof Collection) {
                $target = $this->getTarget($type, $column);
                $targetClass = type_to_class($target);
                $this->addColumns(
                    $data->isNotEmpty() ? $data->first() : new $targetClass(),
                    $fieldOptions['fields'],
                    $mapped,
                    $target
                );
            } else {
                $this->addColumns($data, $fieldOptions['fields'], $mapped, $this->getTarget($type, $column));
            }
        } else {
            // Special count
            if (isset($fieldOptions['count'])) {
                $mapped[$fieldOptions['count']['order'] ?? 1] = $information->{Str::camel($column)}->count() ?? 0;
            }

            $collectMapped = [];
            if ($information->{Str::camel($column)} instanceof Collection) {
                foreach ($information->{Str::camel($column)} as $data) {
                    $collected = [];
                    $this->addColumns(
                        $data,
                        $fieldOptions['fields'],
                        $collected,
                        $this->getTarget($type, $column)
                    );

                    // Prepare glue
                    foreach ($collected as $k => $c) {
                        if (! isset($collectMapped[$k])) {
                            $collectMapped[$k] = [];
                        }
                        $collectMapped[$k][] = $c;
                    }
                }
                if (empty($collectMapped)) {
                    $mapped[] = null;
                } else {
                    // Glue columns
                    foreach ($collectMapped as $key => $value) {
                        $mapped[$key] = implode($fieldOptions['separator'] ?? ' - ', $value);
                    }
                }
                if (empty($collectMapped)) {
                    $mapped[] = '';
                }
            }
        }
    }

    /**
     * Add a magic column ( only available on command line exports )
     */
    protected function addMagicColumn(Model $information, array $fieldOptions, array &$mapped): void
    {
        foreach ($fieldOptions['fields'] as $magicColumn) {
            if (is_array($magicColumn)) {
                $magicFieldOptions = Arr::first($magicColumn);
                $magicColumn = array_key_first($magicColumn);
                if (isset($magicFieldOptions['callback'])
                    && is_callable(
                        $magicFieldOptions['callback']
                    )
                ) {
                    $mapped[$magicFieldOptions['order'] ?? 1] = $magicFieldOptions['callback']($information);
                }
            }
        }
    }

    /**
     * add column value on row
     *
     * @param mixed $information
     * @param array $columns
     * @param array $mapped
     * @throws \Inside\Content\Exceptions\ModelSchemaNotFoundException|Exception
     */
    protected function addColumns($information, array $columns, array &$mapped, string $type): void
    {
        if ($information == null) {
            $mapped[1] = null;

            return;
        }
        foreach ($columns as $column) {
            if (is_array($column)) {
                $fieldOptions = Arr::first($column);
                $column = (string) array_key_first($column);
                if ($this->isSystemColumn($column)) {
                    $mapped[$fieldOptions['order'] ?? 1] = $this->getValue($information, $type, $column);
                } elseif (Schema::hasField($type, $column)) {
                    $options = Schema::getFieldOptions($type, $column);

                    if ($options['type'] == 'reference') {
                        $this->addReferenceColumn(
                            $information,
                            $options['cardinality'],
                            $type,
                            $column,
                            $fieldOptions,
                            $mapped
                        );
                    } else {
                        if ($this->isExportable($type, $column)) {
                            if (isset($fieldOptions['fields'])) {
                                $this->addMagicColumn($information, $fieldOptions, $mapped);
                            } else {
                                $mapped[$fieldOptions['order'] ?? 1] = $this->getValue($information, $type, $column);
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * Prepare headings
     *
     * @param array $columns
     * @param array $headings
     * @param string $type
     * @param string $prefix
     * @throws Exception
     */
    protected function loadHeadings(array $columns, array &$headings, string $type, string $prefix = ''): void
    {
        foreach ($columns as $column) {
            if (is_array($column)) {
                $fieldOptions = Arr::first($column);
                $column = (string) array_key_first($column);
                if (! $this->isSystemColumn($column) && ! $this->isExportable($type, $column)) {
                    continue;
                }
                if ($this->isSystemColumn($column)) {
                    $headings[$fieldOptions['order'] ?? 1] = $prefix.Lang::get('export.titles.'.$column);
                } elseif (Schema::hasField($type, $column)) {
                    $options = Schema::getFieldOptions($type, $column);

                    if ($options['type'] == 'reference') {
                        $modelOptions = Schema::getModelOptions($this->getTarget($type, $column));

                        if (isset($fieldOptions['count'])) {
                            $headings[$fieldOptions['count']['order'] ?? 1] =
                                $prefix.($options['title'][config('app.locale')] ?? $column).' : '
                                .Lang::get(
                                    'export.titles.count'
                                );
                        } else {
                            $this->loadHeadings(
                                $fieldOptions['fields'],
                                $headings,
                                $this->getTarget($type, $column),
                                $prefix.($options['title'][config('app.locale')] ?? $column).' : '
                            );
                        }
                    } else {
                        if (isset($fieldOptions['fields'])) {
                            $this->loadHeadings(
                                $fieldOptions['fields'],
                                $headings,
                                $type,
                                $prefix.($options['title'][config('app.locale')] ?? $column).' : '
                            );
                        } else {
                            $headings[$fieldOptions['order'] ?? 1] =
                                $prefix.($options['title'][config('app.locale')] ?? $column);
                        }
                    }
                } elseif ($type === 'users' && $column === 'fullname') {
                    $headings[$fieldOptions['order'] ?? 1] = $prefix.Lang::get('export.titles.fullname');
                } elseif (isset($fieldOptions['callback'])) {
                    $headings[$fieldOptions['order'] ?? 1] = $prefix.Lang::get('export.titles.'.$column);
                }
            }
        }
    }

    /**
     * Prepare column formats
     *
     * @param array  $columns
     * @param array  $formats
     * @param string $type
     * @throws \Inside\Content\Exceptions\ModelSchemaNotFoundException|Exception
     */
    protected function loadColumnFormats(array $columns, array &$formats, string $type): void
    {
        foreach ($columns as $column) {
            if (is_array($column)) {
                $fieldOptions = Arr::first($column);
                $column = (string) array_key_first($column);
                if ($this->isSystemColumn($column)) {
                    continue;
                }
                if (Schema::hasField($type, $column)) {
                    $options = Schema::getFieldOptions($type, $column);

                    if ($options['type'] != 'reference') {
                        $formats[$fieldOptions['order'] ?? 1] = $this->getFormat($type, $column);
                    } else {
                        if ($this->isExportable($type, $column)) {
                            $this->loadColumnFormats(
                                $fieldOptions['fields'],
                                $formats,
                                $this->getTarget($type, $column)
                            );
                        }
                    }
                } else {
                    if ($this->isExportable($type, $column)) {
                        $formats[$fieldOptions['order'] ?? 1] = null;
                    }
                }
            }
        }
    }

    /**
     * @param string $column
     * @return bool
     */
    protected function isSystemColumn(string $column): bool
    {
        return in_array($column, ['uuid', 'created_at', 'roles']);
    }

    /**
     * @param string $type
     * @param string $fieldName
     * @return bool
     */
    protected function isExportable(string $type, string $fieldName): bool
    {
        if ($type === 'users' && $fieldName === 'fullname') {
            return true;
        }
        try {
            $options = Schema::getFieldOptions($type, $fieldName);
        } catch (FieldSchemaNotFoundException $e) {
            return true; // Not a real field assume config is correct and it will have a callback
        }

        return ! isset($options['non_exportable']) || ! $options['non_exportable'];
    }

    /**
     * @param string $type
     * @param string $fieldName
     * @return string
     * @throws Exception
     */
    protected function getTarget(string $type, string $fieldName): string
    {
        $options = Schema::getFieldOptions($type, $fieldName);
        if ($options['type'] != 'reference') {
            throw new Exception('Can not load target on non reference field ['.$type.']('.$fieldName.')');
        }

        return Arr::first($options['target']);
    }

    /**
     * @param User|Users $information
     * @param string $type
     * @param string $fieldName
     * @return float|int|string|void
     */
    protected function getValue($information, string $type, string $fieldName)
    {
        if ($fieldName === 'fullname' && $type === 'users') {
            return trim(Str::ucfirst(Str::lower($information->firstname ?? '')).' '.Str::upper($information->lastname ?? ''));
        }

        if ($fieldName === 'roles') {
            if ($information instanceof User) {
                return $information->roles->filter(function ($role) {
                    return $role->type == null;
                })->pluck('name')->implode(', ');
            }

            if ($information instanceof Users) {
                $user = User::where('uuid', $information->uuid)->first();
                if ($user && $user->roles) {
                    return $user->roles->filter(function ($role) {
                        return $role->type == null;
                    })->pluck('name')->implode(', ');
                }
            }

            return '';
        }

        $value = $information->{$fieldName};

        if ($fieldName === 'created_at') {
            return get_date($value)?->format('d/m/Y H:i');
        }

        try {
            $options = Schema::getFieldOptions($type, $fieldName);
        } catch (FieldSchemaNotFoundException $e) {
            return $value;
        }
        switch ($options['type']) {
            case 'section':
                return $this->getSectionsValue($information->uuid, $options);
            case 'file':
            case 'image':
            case 'link':
                // TODO
                break;
            case 'timestamp':
                return $value ? Date::dateTimeToExcel(get_date($value)?->toDateTime()) : '-';
            case 'text':
            case 'textarea':
            default:
                return $value;
        }
    }

    /**
     * @param string $type
     * @param string $fieldName
     * @return string|void|null
     */
    protected function getFormat(string $type, string $fieldName)
    {
        try {
            $options = Schema::getFieldOptions($type, $fieldName);
        } catch (FieldSchemaNotFoundException $e) {
            return null;
        }
        switch ($options['type']) {
            case 'file':
            case 'image':
            case 'link':
                // TODO
                break;
            case 'timestamp':
                return 'dd/mm/yyyy hh:mm';
            case 'text':
            case 'textarea':
            default:
                return null;
        }
    }

    protected function getSectionsValue(string $contentUuid, array $options): string
    {
        $sectionValue = '';
        collect($options['target'])->each(function ($sectionName) use ($contentUuid, &$sectionValue) {
            call_user_func(section_type_to_class($sectionName).'::query')->where('sectionable_uuid', $contentUuid)->each(function ($section) use ($sectionName, &$sectionValue) {
                $sectionValue .= match ($sectionName) {
                    'text' => $this->cleanText($section['body']),
                    'text_two_columns' => strip_tags($section['text_left']).' '.strip_tags($section['text_right']),
                    default => $section,
                };
            });
        });

        return $sectionValue;
    }

    protected function cleanText(string $text): string
    {
        $text = html_entity_decode(strip_tags($text), ENT_QUOTES | ENT_HTML5, 'UTF-8');
        $text = preg_replace('/\x{00A0}/u', ' ', $text);
        $text = preg_replace('/&nbsp;/', ' ', $text ?? '');

        return trim(preg_replace('/\s+/', ' ', $text ?? ''));
    }
}
