<?php

namespace Inside\Content\Services;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Inside\Content\Services\Exporters\CsvExporter;
use Inside\Content\Services\Exporters\ExcelExporter;
use Inside\Content\Services\Exporters\ExporterInterface;
use Inside\Content\Services\Exporters\HtmlExporter;
use Inside\Content\Services\Exporters\OdsExporter;
use Inside\Content\Services\Exporters\PdfExporter;
use Inside\Content\Services\Exporters\TsvExporter;
use Inside\Content\Services\Exporters\XlsxExporter;
use Inside\Content\Services\Exporters\XmlExporter;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Finder\Finder;

class Exporter
{
    protected array $drivers = [];

    protected array $exporters = [];

    public function __construct()
    {
        $this->loadDrivers();
    }

    protected function loadDrivers(): void
    {
        // Load buildin exporters
        $this->extend('csv', CsvExporter::class);
        $this->extend('excel', ExcelExporter::class);
        $this->extend('html', HtmlExporter::class);
        $this->extend('ods', OdsExporter::class);
        $this->extend('pdf', PdfExporter::class);
        $this->extend('tsv', TsvExporter::class);
        $this->extend('xlsx', XlsxExporter::class);
        $this->extend('xml', XmlExporter::class);

        // Load custom exporters
        $customDirectory = back_path('src/Exporters');
        if (File::exists($customDirectory)) {
            foreach (
                Finder::create()->files()->name('*Exporter.php')->in(back_path('src/Exporters')) as $file
            ) {
                $this->extend(
                    Str::lower(Str::before($file->getBasename(), 'Exporter')),
                    'Inside\\'.Str::upper(env('APP_CODE')).'\\Exporters\\'.$file->getBasename(
                        '.'.$file->getExtension()
                    )
                );
            }
        }
    }

    public function hasDriver(string $driver): bool
    {
        return isset($this->drivers[Str::lower($driver)]);
    }

    public function getDrivers(): array
    {
        return array_keys($this->drivers);
    }

    public function extend(string $name, string $class): void
    {
        $this->drivers[Str::lower($name)] = $class;
    }

    protected function getExporter(string $driver): mixed
    {
        $driver = Str::lower($driver);
        if (isset($this->exporters[$driver])) {
            return $this->exporters[$driver];
        }

        if (! $this->hasDriver($driver)) {
            return $this->exporters[$driver] = $this->getDefaultExporter();
        }

        return $this->exporters[$driver] = new $this->drivers[$driver]();
    }

    protected function getDefaultExporter(): ExporterInterface
    {
        return new CsvExporter();
    }

    public function export(string $driver, array $config, OutputInterface $console = null): bool
    {
        return $this->getExporter($driver)->export($config, $console);
    }
}
