<?php

namespace Inside\Content\Services;

use Inside\Content\Contracts\ContentTree as ContentTreeContract;
use Inside\Content\Exceptions\FieldSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;

class ContentTree implements ContentTreeContract
{
    private array $treesConfigurations;

    public function __construct()
    {
        $this->treesConfigurations = config('content-trees.contents', []);
    }

    public function getContentParent(Content $content, int $depth = 1): ?Content
    {
        if (! $this->hasValidTreeConfiguration($content->content_type)) {
            return null;
        }

        $parentField = $this->treesConfigurations[$content->content_type]['parent_field'];
        $parent = $content->{$parentField}->first();

        if ($parent instanceof Content && $depth > 1) {
            $parent = $this->getContentParent($parent, $depth - 1);
        }

        return $parent;
    }

    public function getContentRoot(Content $content): ?Content
    {
        $root = $this->getContentParent($content);

        while ($root && ($rootParent = $this->getContentParent($root)) && $rootParent->uuid !== $root->uuid) {
            $root = $rootParent;
        }

        return $root;
    }

    public function hasValidTreeConfiguration(string $contentType): bool
    {
        if (! array_key_exists($contentType, $this->treesConfigurations)) {
            return false;
        }

        $treeConfiguration = $this->treesConfigurations[$contentType];

        if (! is_array($treeConfiguration) || ! array_key_exists('parent_type', $treeConfiguration) || ! array_key_exists('parent_field', $treeConfiguration)) {
            return false;
        }

        $parentType = $treeConfiguration['parent_type'];
        $parentField = $treeConfiguration['parent_field'];

        try {
            $options = Schema::getFieldOptions($contentType, $parentField);
        } catch (FieldSchemaNotFoundException $e) {
            return false;
        }

        if ($options['type'] !== 'reference' || ! in_array($parentType, $options['target'])) {
            return false;
        }

        return true;
    }

    public function formatContent(?Content $content): ?array
    {
        if ($content === null) {
            return null;
        }

        return [
            'uuid' => $content->uuid,
            'content_type' => $content->content_type,
            'slug' => $content->slug[0] ?? null,
        ];
    }
}
