<?php

namespace Inside\Content\Services;

use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Host\Bridge\BridgeContent;
use League\Csv\Reader;
use League\Csv\Writer;
use Symfony\Component\Console\Style\OutputStyle;

/**
 * Class ContentImportService
 */
class ContentImportService
{
    private const DATETIME_FIELDS = ['published_at', 'created_at', 'updated_at', 'start_date', 'end_date'];

    private array $availableFields;

    private array $createdContents;

    private array $config;

    private array $headers;

    private bool $reverseFlag;

    private Collection $contents;

    private string $contentType;

    private string $filePath;

    private string $disk = 'local';

    private string $importPath = 'import/';

    private OutputStyle $outPut;

    public function execute(string $filename)
    {
        $this->filePath = $this->resolveFilePath($filename);
        $this->contentType = $this->detectContentType($filename);
        $this->availableFields = $this->getAvailableFields();
        $this->readContentsFromCsvFile();
        $this->importContents();

        if (! empty($this->createdContents)) {
            $this->createLogFile($filename);
            $this->renameImportedFile($filename);
        }
    }

    public function setReverseImportOrder($flag = false)
    {
        $this->reverseFlag = $flag;

        return $this;
    }

    public function setConfig(array $config): self
    {
        $this->config = $config;

        return $this;
    }

    public function setOutput(OutputStyle $outputStyle): self
    {
        $this->outPut = $outputStyle;

        return $this;
    }

    private function resolveFilePath(string $filename): string
    {
        $filePath = $this->importPath.$filename;

        if (! Storage::disk($this->disk)->exists($filePath)) {
            throw new \InvalidArgumentException("File not found: {$filePath}");
        }

        return $filePath;
    }

    private function detectContentType(string $filename): string
    {
        $contentType = pathinfo($filename, PATHINFO_FILENAME);

        if (! InsideSchema::isModel($contentType)) {
            throw new \UnexpectedValueException("Content type not exists: {$contentType}");
        }

        if (! isset($this->config['allowed_content_types'][$contentType])) {
            throw new \UnexpectedValueException("Content type not allowed: {$contentType}");
        }

        return $contentType;
    }

    private function getAvailableFields()
    {
        return array_merge(
            InsideSchema::getFieldListing($this->contentType),
            ['published_at', 'created_at', 'updated_at']
        );
    }

    private function readContentsFromCsvFile(): void
    {
        $file = Storage::disk($this->disk)->path($this->filePath);
        $csv = Reader::createFromPath($file, 'r');
        $csv->setDelimiter(',');
        $csv->setEnclosure('"');
        $csv->setHeaderOffset(0);

        $records = collect($csv->getRecords());
        $this->headers = $csv->getHeader();
        $this->contents = $records->values();

        if ($this->reverseFlag && $records->isNotEmpty()) {
            $rows = $records->reverse();
            $this->contents = collect($rows->values());
        }
    }

    private function importContents(): void
    {
        if (empty($this->headers) || $this->contents->isEmpty()) {
            $this->outPut->warning('Aucune donnée à importer.');

            return;
        }

        $contents = $this->contents
            ->filter(function ($row) {
                return count(array_filter($row, fn ($value) => $value !== null && $value !== '')) > 0;
            })
            ->map(function ($row, $index) {
                $lineNumber = $index + 2;
                try {
                    $data = $this->prepareData($row);

                    if (! $data) {
                        $this->outPut->warning("Ligne $lineNumber ignorée: préparation des données échouée");

                        return null;
                    }

                    return $data;
                } catch (Exception $e) {
                    $this->outPut->warning("Ligne $lineNumber ignorée: ".$e->getMessage());

                    return null;
                }
            })
            ->filter();

        $validCount = $contents->count();
        $invalidCount = $this->contents->count() - $validCount;
        $this->contents = $contents;

        $this->outPut->success("Lignes valides: $validCount");
        $this->outPut->success("Lignes invalides: $invalidCount");

        if ($contents->isEmpty()) {
            $this->outPut->success('Aucun contenu valide à importer');

            return;
        }

        $this->insertContents();
    }

    private function insertContents(): void
    {
        $bridge = new BridgeContent();
        $insertedCount = 0;
        $skippedCount = 0;

        foreach ($this->contents as $content) {
            try {
                $cleanContent = $this->resolveReferences($content);

                if (! $cleanContent) {
                    $skippedCount++;
                    continue;
                }

                if (empty($cleanContent['title'] ?? null)) {
                    $this->outPut->warning('Contenu ignoré : titre manquant');
                    $skippedCount++;
                    continue;
                }

                if ($this->contentAlreadyExists($cleanContent)) {
                    $this->outPut->warning('Contenu ignoré : contenu identique déjà existant');
                    $skippedCount++;
                    continue;
                }

                $uuid = $bridge->contentInsert(
                    type: $this->contentType,
                    data: $cleanContent,
                    creation: true,
                    fromCli: true
                );

                if ($uuid) {
                    $insertedCount++;
                    $this->outPut->success("Contenu inséré: $uuid");

                    $this->createdContents[] = [
                        'title' => $cleanContent['title'],
                        'uuid' => $uuid,
                    ];
                }
            } catch (Exception $e) {
                $this->outPut->error("Erreur d'insertion: ".$e->getMessage());
            }
        }

        $this->outPut->success("$insertedCount / {$this->contents->count()} contenus insérés pour '{$this->contentType}'");
        if ($skippedCount > 0) {
            $this->outPut->success("$skippedCount / {$this->contents->count()} contenus ignorés pour '{$this->contentType}'");
        }
    }

    private function resolveReferences(array $content): ?array
    {
        $data = $content;
        $referenceFieldMappings = $this->config['reference_content_types'];
        foreach ($referenceFieldMappings as $refField) {
            $refKey = "{$refField}_ref";
            if (! isset($data[$refKey])) {
                continue;
            }

            $refValue = $this->findReferenceByName($data[$refKey], $refField);

            if (! $refValue) {
                $this->outPut->warning("Référence non trouvée: {$data[$refKey]} pour $refField - Contenu ignoré");

                return null;
            }
        }

        return $data;
    }

    private function contentAlreadyExists(array $data): bool
    {
        try {
            $table = sprintf('inside_content_%s', $this->contentType);

            $query = DB::table($table);

            $columns = DB::getSchemaBuilder()->getColumnListing($table);

            foreach ($data as $field => $value) {
                if (in_array($field, $columns) &&
                    ! in_array($field, ['uuid', 'uuid_host', 'created_at', 'updated_at', 'published_at'])) {
                    if ($value === null) {
                        $query->whereNull($field);
                    } else {
                        $query->where($field, $value);
                    }
                }
            }

            return $query->exists();
        } catch (Exception $e) {
            $this->outPut->error("Erreur lors de la vérification d'existence: ".$e->getMessage());

            return false;
        }
    }

    private function findReferenceByName(string $name, string $type): ?string
    {
        $table = "inside_content_$type";
        $field = 'title';

        try {
            if (! InsideSchema::hasContentType($type) && ! InsideSchema::hasModel($type)) {
                $this->outPut->error("Le type de contenu $type n'existe pas");

                return null;
            }

            $result = DB::table($table)
                ->where($field, $name)
                ->first();

            if ($result instanceof \stdClass && isset($result->uuid)) {
                return $result->uuid;
            }

            return null;
        } catch (Exception $e) {
            $this->outPut->error('Erreur lors de la recherche de référence: '.$e->getMessage());

            return null;
        }
    }

    private function prepareData(array $data): ?array
    {
        $result = collect($data)
            ->mapWithKeys(function ($value, $key) {
                if (empty($value)) {
                    return [];
                }

                if (in_array($key, self::DATETIME_FIELDS)) {
                    try {
                        $value = $this->formatDate($value);
                    } catch (Exception $exception) {
                        throw new Exception($exception->getMessage());
                    }
                }

                $referenceFieldMappings = $this->config['reference_content_types'];
                if (in_array($key, $referenceFieldMappings)) {
                    $referenceValue = $this->findReferenceByName($value, $key);
                    if ($referenceValue) {
                        return [$key => $referenceValue];
                    } else {
                        return [$key.'_ref' => $value];
                    }
                }

                return in_array($key, $this->availableFields) ? [$key => $value] : [];
            })
            ->toArray();

        if (in_array('icon', $this->availableFields) && ! isset($result['icon'])) {
            $result['icon'] = '{"name":"users"}';
        }

        return ! empty($result) ? $result : null;
    }

    private function formatDate(string $value): ?string
    {
        if ($value === 'en cours') {
            return get_date_in_user_timezone(now())?->format('Y-m-d H:i:s');
        }

        try {
            $date = get_date_in_user_timezone($value, 'd/m/Y');
            if ($date) {
                return $date->format('Y-m-d H:i:s');
            }
        } catch (Exception $e) {
        }

        try {
            $formatter = new \IntlDateFormatter(
                locale: 'fr_FR',
                dateType: \IntlDateFormatter::FULL,
                timeType: \IntlDateFormatter::NONE,
                pattern: 'dd MM yyyy'
            );

            $formattedDate = $formatter->parse($value);
            if (! is_numeric($formattedDate)) {
                throw new Exception("Impossible de parser la date: $value");
            }

            return get_date_in_user_timezone($formattedDate)?->format('Y-m-d H:i:s');
        } catch (Exception $e) {
            throw new Exception("Impossible de parser la date: $value");
        }
    }

    private function createLogFile(string $originalFileName): void
    {
        $directory = 'archived_import/';
        $disk = Storage::disk('local');

        if (! $disk->exists($directory)) {
            $disk->makeDirectory($directory);
        }
        $fileInfo = pathinfo($originalFileName);
        $baseName = $fileInfo['filename'].'_log-';
        $extension = '.'.($fileInfo['extension'] ?? 'csv');

        $logFileName = $baseName.date('YmdHis').$extension;
        $counter = 1;

        $fullPath = storage_path('app/'.$directory.$logFileName);
        $csv = Writer::createFromPath($fullPath, 'w+');
        $csv->insertOne(['title', 'uuid']);

        foreach ($this->createdContents as $content) {
            $csv->insertOne([
                $content['title'] ?? 'Sans titre',
                $content['uuid'] ?? '',
            ]);
        }

        $this->outPut->success("Fichier de log créé: $logFileName avec ".count($this->createdContents).' contenus');
    }

    private function renameImportedFile(string $originalFileName): void
    {
        if (empty($this->createdContents)) {
            return;
        }
        $directory = 'archived_import/';
        $fileInfo = pathinfo($originalFileName);
        $baseName = $fileInfo['filename'].'-';
        $extension = '.'.($fileInfo['extension'] ?? 'csv');

        $newFileName = $baseName.date('YmdHis').$extension;
        $counter = 1;

        $disk = Storage::disk('local');

        $source = $this->importPath.$originalFileName;
        $destination = $directory.$newFileName;

        if ($disk->exists($source) && $disk->move($source, $destination)) {
            $this->outPut->success("Fichier renommé en: $newFileName");
        } else {
            $this->outPut->success('Impossible de renommer le fichier original');
        }
    }
}
