<?php

declare(strict_types=1);

namespace Inside\Content\Services;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Inside\Content\Contracts\ContentCollectionOrderManager as ContentCollectionOrderManagerContract;
use Inside\Content\DTO\ContentOrderedItemDTO;
use Inside\Host\Bridge\BridgeContent;

final class ContentCollectionOrderManager implements ContentCollectionOrderManagerContract
{
    private ?BridgeContent $bridge = null;

    public function getBridge(): BridgeContent
    {
        return $this->bridge ??= new BridgeContent();
    }

    public function updateContentsOrder(Collection $orderedItems, ?ContentOrderedItemDTO $parent = null): Collection
    {
        return $orderedItems->values()->each(function (ContentOrderedItemDTO $item, int $orderValue) use ($parent) {
            $this->updateContentsOrder($item->children, $item);

            $parentAttribute = Str::camel($item->parentField ?? '');

            $exists = type_to_class($item->contentType)::query()
                ->where('uuid', $item->uuid)
                ->where(
                    fn (Builder $query) => $query
                        ->where($item->orderField, $orderValue)
                        ->when($orderValue === 0, fn (Builder $query) => $query->orWhereNull($item->orderField))
                )
                ->when(
                    $parentAttribute && $parent?->uuid,
                    fn (Builder $query) => $query->whereHas($parentAttribute, fn (Builder $query) => $query->where('uuid', $parent?->uuid))
                )
                ->exists();

            if ($exists) {
                return;
            }

            $data = [
                'uuid' => $item->uuid,
                $item->orderField => $orderValue,
            ];

            if ($item->parentField && $parent?->uuid) {
                $data[$item->parentField] = $parent->uuid;
            }

            $this->getBridge()->contentUpdate($item->contentType, $data);
        });
    }
}
