<?php

namespace Inside\Content\Services\ConfigurationGenerator;

use Illuminate\Support\Collection;
use Inside\Content\Services\ConfigurationGenerator\Bundle\BaseBundle;
use Inside\Content\Services\ConfigurationGenerator\Bundle\Core\BaseFieldOverride;
use Inside\Content\Services\ConfigurationGenerator\Bundle\Field\Field\BaseField;
use Inside\Content\Services\ConfigurationGenerator\Bundle\Field\Storage\BaseStorage;
use Inside\Content\Services\ConfigurationGenerator\Bundle\Language\ContentSettingsBundle;
use Inside\Content\Services\ConfigurationGenerator\Bundle\Paragraphs\ParagraphsTypeBundle;

class ParagraphsGenerator
{
    protected array $fields = [];

    protected array $available = [];

    protected array $unavailable = [];

    protected function __construct(array $configuration)
    {
        $this->fields = $configuration['default.fields'];
        $this->available = $this->initializeAvailableParagraphs($configuration);
    }

    public static function load(): self
    {
        return new self(config('generator.paragraphs'));
    }

    public function getAvailable(): array
    {
        return $this->available;
    }

    public function getUnvailable(): array
    {
        return $this->unavailable;
    }

    public function enable(string $paragraph): self
    {
        $unavailable = collect($this->getUnvailable());

        if ($unavailable->has($paragraph)) {
            $this->available = collect($this->available)->merge($unavailable->get($paragraph))->toArray();
        }

        return $this;
    }

    protected function initializeAvailableParagraphs(array $configuration): array
    {
        return collect($this->initializeParagraphs($configuration['default.paragraphs']))
            ->merge($this->initializeSpecificParagraphs($configuration['specific']))
            ->merge($this->initializeDeprecatedParagraphs($configuration['deprecated']))
            ->merge($this->initializeParagraphs($configuration['custom']))
            ->toArray();
    }

    protected function initializeSpecificParagraphs(array $paragraphs): array
    {
        return collect($this->initializeParagraphs($paragraphs, false))->reject(function ($configuration, $name) {
            $isNew = in_array(true, collect($configuration)->map->isNew()->toArray());

            if ($isNew) {
                $this->unavailable[$name] = $configuration;
            }

            return $isNew;
        })->flatten()->toArray();
    }

    protected function initializeDeprecatedParagraphs(array $paragraphs): array
    {
        return collect($this->initializeParagraphs($paragraphs))
            ->reject(fn (BaseBundle $paragraph) => $paragraph->isNew())
            ->toArray();
    }

    protected function initializeParagraphs(array $paragraphs, bool $flatten = true): array
    {
        return collect($paragraphs)
            ->map(fn (array $configuration, string $id) => $this->initializeParagraph($id, $configuration['label'], $configuration['fields'] ?? null)
            )
            ->when($flatten, fn (Collection $collection) => $collection->flatten())
            ->toArray();
    }

    protected function initializeParagraph(string $id, string $label, ?array $fields): array
    {
        $bundle = ParagraphsTypeBundle::create($id, $label);
        $contentSettings = ContentSettingsBundle::create($bundle, $id);
        $defaultFields = $this->initializeDefaultParagraphFields($bundle, $id);

        $fields = collect($fields)->map(fn (array $configuration, string $fieldName) => [
            $this->initializeField(
                bundle: $bundle,
                id: $id,
                fieldName: $fieldName,
                configuration: collect($configuration)
            ),
            $this->initializeStorage(
                bundle: $bundle,
                fieldName: $fieldName,
                configuration: collect($configuration)
            ),
        ]);

        return collect([$bundle])->add($contentSettings)->merge($fields->flatten())->merge($defaultFields)->toArray();
    }

    protected function initializeField(BaseBundle $bundle, string $id, string $fieldName, Collection $configuration): BaseField
    {
        return $configuration->get('class')::create($bundle, $id, $fieldName, $configuration);
    }

    protected function initializeStorage(BaseBundle $bundle, string $fieldName, Collection $configuration): BaseStorage
    {
        return $configuration->get('class')::getStorage()::create($bundle, $fieldName, $configuration);
    }

    protected function initializeDefaultParagraphFields(BaseBundle $bundle, string $id): array
    {
        return collect($this->fields)->map(fn (array $configuration, string $fieldName) => BaseFieldOverride::create(
            bundle: $bundle,
            id: $id,
            fieldName: $fieldName,
            configuration: collect($configuration))
        )->toArray();
    }
}
