<?php

namespace Inside\Content\Services;

use Carbon\Carbon;
use Illuminate\Support\Facades\URL;
use Inside\Content\Calendars\Google;
use Inside\Content\Calendars\Ics;
use Inside\Content\Calendars\Office;
use Inside\Content\Calendars\Outlook;
use Inside\Content\Calendars\Yahoo;
use Inside\Content\Contracts\CalendarInterface;
use Inside\Content\Models\Contents\Events;
use Inside\Content\Models\Sections\Invite;
use Inside\Slug\Facades\ContentUrlGenerator;
use InvalidArgumentException;

class CalendarService
{
    private function getCalendar(string $calendarType): CalendarInterface
    {
        return match ($calendarType) {
            'google' => new Google(),
            'outlook' => new Outlook(),
            'office' => new Office(),
            'yahoo' => new Yahoo(),
            'ics' => new Ics(),
            default => throw new InvalidArgumentException('Calendar type '.$calendarType.' is not allowed'),
        };
    }

    public function getEventCalendarExport(string $calendarType, Events $event): string
    {
        $calendar = $this->getCalendar($calendarType);

        $eventUrl = ContentUrlGenerator::generateUrl($event);
        $eventUrl = isset($eventUrl) ? URL::to($eventUrl) : null;

        if (empty($event->end_date)) {
            $event->end_date = get_date($event->date)?->addHour();
        }

        return $calendar->generate(
            $event->title,
            $event->place,
            $event->date,
            $event->end_date,
            $event->description,
            $eventUrl
        );
    }

    public function getInviteCalendarExport(string $calendarType, Invite $invite): string
    {
        $calendar = $this->getCalendar($calendarType);

        return $calendar->generate(
            $invite->title,
            $invite->place,
            $invite->date,
            $invite->end_date,
            $invite->description,
            $invite->invite_link
        );
    }
}
