<?php

namespace Inside\Content\Services\Caches;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Inside\Authentication\Models\User;

/**
 * Content Cache
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class ContentCache
{
    /**
     * Check if given parameters have stored cache
     */
    public function has(string $types, array $parameters = null): ?array
    {
        if (! $this->canCache($parameters)) {
            return null;
        }

        $key = $this->getKey($parameters);
        $tags = $this->getTags($types);

        if ($cached = Cache::tags($tags)->get($key)) {
            return $cached;
        }

        return null;
    }

    public function forget(array $tags): void
    {
        if (! $this->canCache()) {
            return;
        }

        Cache::tags($tags)->flush();
    }

    public function store(string $types, array $parameters = null, array $data = []): void
    {
        if (! $this->canCache($parameters)) {
            return;
        }

        $key = $this->getKey($parameters);
        $tags = $this->getTags($types);

        Cache::tags($tags)->forever($key, $data);
    }

    /**
     * Check if we can cache this request
     */
    protected function canCache(?array $parameters = null): bool
    {
        if (! env('APP_CACHE', true)) {
            return false;
        }

        $fields = json_decode($parameters['fields'] ?? '[]', true);

        if ($fields === null) {
            return false;
        }

        foreach ($fields as $field) {
            if (is_array($field)) {
                foreach ($field as $reference) {
                    if (! $this->canCache(['fields' => json_encode(is_array($reference) ? $reference : [$reference])])) {
                        return false;
                    }
                }
                continue;
            }

            if (in_array($field, ['admin'])) {
                return false;
            }
        }

        return true;
    }

    protected function getKey(array $parameters = null): string
    {
        $appKey = env('APP_KEY');
        $key = 'inside.'.$appKey.'.content';

        if ($parameters) {
            return $key.'.'.sha1((string) json_encode($parameters));
        }

        return $key;
    }

    public function getTags(string $types = ''): array
    {
        $appKey = env('APP_KEY');
        $tags = array_merge([$appKey, 'all'], explode(',', $types));

        $authUser = Auth::user();

        if (! $authUser instanceof User) {
            return $tags;
        }

        $user = User::findOrFail($authUser->uuid);

        return array_merge($user->roles()->pluck('name')->toArray(), $tags);
    }
}
