<?php

declare(strict_types=1);

namespace Inside\Content\Services\AdvancedTable;

use DOMDocument;
use Maatwebsite\Excel\Concerns\FromArray;

class Export implements FromArray
{
    protected array $jsonData;

    public function __construct(array $jsonData)
    {
        $this->jsonData = $jsonData;
    }

    public function array(): array
    {
        $headers = [];

        foreach ($this->jsonData['headers'] as $header) {
            $headers[] = $header['content'] ? $this->extractPlainText($header['content']) : '';
        }

        $rows = [];

        foreach ($this->jsonData['rows'] as $row) {
            $rowData = [];
            foreach ($row['data'] as $cell) {
                $rowData[] = $cell['content'] ? $this->extractPlainText($cell['content']) : '';
            }
            $rows[] = $rowData;
        }

        return array_merge([$headers], $rows);
    }

    private function extractPlainText(string $htmlContent): string
    {
        $doc = new DOMDocument();
        libxml_use_internal_errors(true);
        $doc->loadHTML(mb_convert_encoding($htmlContent, 'HTML-ENTITIES', 'UTF-8'));
        libxml_clear_errors();

        $node = $doc->getElementsByTagName('body')->item(0);
        if ($node === null) {
            return '';
        }

        return $this->extractTextRecursively($node);
    }

    private function extractTextRecursively(\DOMNode $node): string
    {
        $orderedContent = [];

        foreach ($node->childNodes as $child) {
            if ($child instanceof \DOMElement && $child->nodeName === 'a' && $child->hasAttribute('href')) {
                $href = $child->getAttribute('href');
                $text = trim($child->textContent);

                if (! empty($text)) {
                    $orderedContent[] = "($href) $text";
                } else {
                    $orderedContent[] = $href;
                }
                continue;
            }

            $elements = $this->extractElements($child);
            if (! empty($elements)) {
                $orderedContent[] = $elements;
                continue;
            }

            if ($child instanceof \DOMElement && $child->nodeName === 'ged-documents' && $child->hasAttribute('documents')) {
                $documents = json_decode(urldecode($child->getAttribute('documents')));

                $orderedContent[] = collect($documents)->map(function ($document) {
                    return url($document->file->path);
                })->implode("\n");
            }

            if ($child->hasChildNodes()) {
                $childContent = $this->extractTextRecursively($child);
                $orderedContent[] = $childContent;
            }
        }

        return implode("\n", array_filter($orderedContent));
    }

    public function extractElements(\DOMNode $node): string
    {
        $allowedTags = ['strong', 'span', 'p', 'li', 'u', 'em', 's'];
        $orderedContent = [];

        $isHeading = preg_match('/^h[1-6]$/', $node->nodeName);
        $isAllowedTag = in_array($node->nodeName, $allowedTags);

        if (! $isAllowedTag && ! $isHeading) {
            return '';
        }

        foreach ($node->childNodes as $child) {
            if ($child instanceof \DOMText) {
                $orderedContent[] = $child->nodeValue;
            } elseif ($child instanceof \DOMElement) {
                if ($child->nodeName === 'a') {
                    $href = $child->getAttribute('href');
                    $text = trim($child->textContent);
                    $orderedContent[] = "($href) $text";
                } else {
                    $orderedContent[] = $this->extractElements($child);
                }
            }
        }

        return implode('', array_filter($orderedContent));
    }
}
