<?php

declare(strict_types=1);

namespace Inside\Content\Providers;

use Doctrine\DBAL\Exception;
use Doctrine\DBAL\Types\Type;
use Fomvasss\Punycode\PunycodeServiceProvider;
use Hashids\Hashids;
use Illuminate\Support\Composer;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Log;
use Inside\Application;
use Inside\Content\Console\AddMissingWysiwygImagesCommand;
use Inside\Content\Console\CleanRevisionOfDeletedContentCommand;
use Inside\Content\Console\CleanUploadsCommand;
use Inside\Content\Console\CleanUsersProfileImageCommand;
use Inside\Content\Console\DeleteContentCommand;
use Inside\Content\Console\DuplicateGedCommand;
use Inside\Content\Console\ExportContentCommand;
use Inside\Content\Console\ExportContentTypeCommand;
use Inside\Content\Console\ExportGedCommand;
use Inside\Content\Console\ExportSectionTypeCommand;
use Inside\Content\Console\FindMissingFilesCommand;
use Inside\Content\Console\GenerateAllImageCommand;
use Inside\Content\Console\ImportContentCommand;
use Inside\Content\Console\ImportContentFromCsvCommand;
use Inside\Content\Console\ImportContentFromInside1Command;
use Inside\Content\Console\ImportContentTypeFromCsvCommand;
use Inside\Content\Console\ImportStockPricesCommand;
use Inside\Content\Console\ImportWTTJJobOffersCommand;
use Inside\Content\Console\InitRevisionCommand;
use Inside\Content\Console\MakeContentTypeCommand;
use Inside\Content\Console\Paragraphs\EnableParagraph;
use Inside\Content\Console\Paragraphs\FixMultilingualLabels;
use Inside\Content\Console\Paragraphs\GenerateConfigurations;
use Inside\Content\Console\Paragraphs\ImportConfigurations;
use Inside\Content\Console\Paragraphs\ListUnavailableParagraphs;
use Inside\Content\Console\Paragraphs\MakeRattachedBundlesTranslatable;
use Inside\Content\Console\PurgeOldContentsCommand;
use Inside\Content\Console\RebuildModelsCommand;
use Inside\Content\Console\RepairFileExtensionsCommand;
use Inside\Content\Console\RepairImageExtensionsCommand;
use Inside\Content\Console\RepairPublicationFields;
use Inside\Content\Console\RepairUploadLimitCommand;
use Inside\Content\Console\RepairVideoExtensionsCommand;
use Inside\Content\Console\RepairWysiwygImagesCommand;
use Inside\Content\Console\ReplaceUuidsWithHostInRolesTableCommand;
use Inside\Content\Console\SetContentTypesAsTranslatableCommand;
use Inside\Content\Console\SyncPivotsCommand;
use Inside\Content\Contracts\Alphavantage;
use Inside\Content\Contracts\ContentBlueprints as ContentBlueprintsContract;
use Inside\Content\Contracts\ContentCollectionOrderManager as ContentCollectionOrderManagerContract;
use Inside\Content\Contracts\ContentExternalList as ContentExternalListContract;
use Inside\Content\Contracts\ContentList as ContentListContract;
use Inside\Content\Contracts\ContentTree as ContentTreeContract;
use Inside\Content\Contracts\DrupalSchema as DrupalSchemaContract;
use Inside\Content\Contracts\ModelForm as ModelFormContract;
use Inside\Content\Contracts\QueryHelper;
use Inside\Content\Contracts\RevisionService as RevisionServiceContract;
use Inside\Content\Contracts\SchemaService as SchemaServiceContract;
use Inside\Content\Contracts\Serializer;
use Inside\Content\Contracts\Transformer as TransformerContract;
use Inside\Content\Contracts\WelcomeToTheJungleApiClient as WelcomeToTheJungleApiClientContract;
use Inside\Content\Contracts\WelcomeToTheJungleContentsManager as WelcomeToTheJungleContentsManagerContract;
use Inside\Content\Contracts\WysiwygImageService;
use Inside\Content\Doctrine\DBAL\Types\TimestampType;
use Inside\Content\Events\BirthdayWishesSendEvent;
use Inside\Content\Events\CommentDeletedEvent;
use Inside\Content\Events\ContentCacheIsNotValidAnyMoreEvent;
use Inside\Content\Events\ContentCreatedEvent;
use Inside\Content\Events\ContentDeletedEvent;
use Inside\Content\Events\ContentFullyInsertedEvent;
use Inside\Content\Events\ContentFullyUpdatedEvent;
use Inside\Content\Events\ContentPublishedEvent;
use Inside\Content\Events\ContentSynchedEvent;
use Inside\Content\Events\ContentUpdatedEvent;
use Inside\Content\Events\ContentUpdatingEvent;
use Inside\Content\Events\ProcessingUpdate;
use Inside\Content\Events\SeniorityAnniversaryWishesSendEvent;
use Inside\Content\Facades\AlphabetContent;
use Inside\Content\Facades\Calendar;
use Inside\Content\Facades\ScopeLogic;
use Inside\Content\Http\Middlewares\AddWysiwygImagesOnContentForm;
use Inside\Content\Http\Middlewares\SaveWysiwygImages;
use Inside\Content\Http\Middlewares\UserMentionRequest;
use Inside\Content\Listeners\AlphabetContentCreatedListener;
use Inside\Content\Listeners\AlphabetContentUpdatedListener;
use Inside\Content\Listeners\BirthdayWishesSendListener;
use Inside\Content\Listeners\CleanupOldFiles;
use Inside\Content\Listeners\CommentDeletedListener;
use Inside\Content\Listeners\ContentCreatedListener;
use Inside\Content\Listeners\ContentDeletedListener;
use Inside\Content\Listeners\ContentFullyUpdatedListener;
use Inside\Content\Listeners\ContentInsertedListener;
use Inside\Content\Listeners\ContentPublishedListener;
use Inside\Content\Listeners\ContentUpdatingListener;
use Inside\Content\Listeners\DocumentTitleUpdatedListener;
use Inside\Content\Listeners\ImageStylesSubscriber;
use Inside\Content\Listeners\InvalidateCache;
use Inside\Content\Listeners\JobOffersDeletedListener;
use Inside\Content\Listeners\JobOffersFormTemplateUpdatedListener;
use Inside\Content\Listeners\JobOffersUpdatedListener;
use Inside\Content\Listeners\NewsCategoriesUpdatedListener;
use Inside\Content\Listeners\SeniorityAnniversaryWishesSendListener;
use Inside\Content\Listeners\SetIsMaintenanceListener;
use Inside\Content\Listeners\UserDisabledListener;
use Inside\Content\Listeners\WarnNewsFollowersListener;
use Inside\Content\Services\AlphabetContentService;
use Inside\Content\Services\Caches\ContentCache;
use Inside\Content\Services\CalendarService;
use Inside\Content\Services\CleanContentService;
use Inside\Content\Services\ContentBlueprints;
use Inside\Content\Services\ContentCollectionOrderManager;
use Inside\Content\Services\ContentTree;
use Inside\Content\Services\Exporter;
use Inside\Content\Services\Forms\ModelForm;
use Inside\Content\Services\Importer;
use Inside\Content\Services\OpenWeatherService;
use Inside\Content\Services\Queries\ContentExternalList;
use Inside\Content\Services\Queries\ContentList;
use Inside\Content\Services\Queries\ContentQuery;
use Inside\Content\Services\Queries\ContentQueryHelper;
use Inside\Content\Services\Queries\DrupalSchema;
use Inside\Content\Services\Revision\RevisionService;
use Inside\Content\Services\Schema\ClassesService;
use Inside\Content\Services\Schema\SchemaService;
use Inside\Content\Services\ScopeLogicService;
use Inside\Content\Services\Serializer\SerializerService;
use Inside\Content\Services\Stocks\AlphavantageService;
use Inside\Content\Services\WelcomeToTheJungle\WelcomeToTheJungleApiClient;
use Inside\Content\Services\WelcomeToTheJungle\WelcomeToTheJungleContentsManager;
use Inside\Content\Services\WysiwygImage;
use Inside\Content\Support\ShortUrlManager;
use Inside\Content\Support\UniqueKeyGenerator;
use Inside\Content\Transformers\ContentTransformer;
use Inside\Host\Event\Entity\User\UserInsertEvent;
use Inside\Support\EventServiceProvider;
use Inside\Support\Filesystem;
use Inside\User\Events\UserDisabledEvent;
use Mews\Purifier\PurifierServiceProvider;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Ramsey\Uuid\Doctrine\UuidType;

/**
 * Inside package content service.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class ContentServiceProvider extends EventServiceProvider
{
    protected array $listen = [
        CommentDeletedEvent::class => [
            CommentDeletedListener::class,
        ],
        ContentCreatedEvent::class => [
            ContentCreatedListener::class,
            SetIsMaintenanceListener::class,
        ],
        ContentFullyInsertedEvent::class => [
            ContentInsertedListener::class,
            WarnNewsFollowersListener::class,
            AlphabetContentCreatedListener::class,
        ],
        ContentUpdatedEvent::class => [
            NewsCategoriesUpdatedListener::class,
            DocumentTitleUpdatedListener::class,
        ],
        ContentFullyUpdatedEvent::class => [
            JobOffersUpdatedListener::class,
            JobOffersFormTemplateUpdatedListener::class,
            ContentFullyUpdatedListener::class,
            AlphabetContentUpdatedListener::class,
        ],
        ContentPublishedEvent::class => [
            ContentPublishedListener::class,
        ],
        ContentDeletedEvent::class => [
            ContentDeletedListener::class,
            JobOffersDeletedListener::class,
        ],
        BirthdayWishesSendEvent::class => [
            BirthdayWishesSendListener::class,
        ],
        SeniorityAnniversaryWishesSendEvent::class => [
            SeniorityAnniversaryWishesSendListener::class,
        ],
        ContentUpdatingEvent::class => [
            ContentUpdatingListener::class,
            InvalidateCache::class,
            SetIsMaintenanceListener::class,
        ],
        ContentSynchedEvent::class => [
            InvalidateCache::class,
        ],
        ProcessingUpdate::class => [
            CleanupOldFiles::class,
        ],
        ContentCacheIsNotValidAnyMoreEvent::class => [
            InvalidateCache::class,
        ],
        UserDisabledEvent::class => [
            UserDisabledListener::class,
        ],
    ];

    protected array $subscribe = [
        ImageStylesSubscriber::class,
    ];

    public array $bindings = [
        ModelFormContract::class => ModelForm::class,
        TransformerContract::class => ContentTransformer::class,
        Serializer::class => SerializerService::class,
        Calendar::class => CalendarService::class,
        ContentTreeContract::class => ContentTree::class,
        WelcomeToTheJungleApiClientContract::class => WelcomeToTheJungleApiClient::class,
        WelcomeToTheJungleContentsManagerContract::class => WelcomeToTheJungleContentsManager::class,
        AlphabetContent::class => AlphabetContentService::class,
        ContentCollectionOrderManagerContract::class => ContentCollectionOrderManager::class,
        Alphavantage::class => AlphavantageService::class,
    ];

    public array $singletons = [
        WysiwygImageService::class => WysiwygImage::class,
        SchemaServiceContract::class => SchemaService::class,
        DrupalSchemaContract::class => DrupalSchema::class,
        CleanContentService::class => CleanContentService::class,
        'content.cache' => ContentCache::class,
        'inside.content.importer' => Importer::class,
        'inside.content.exporter' => Exporter::class,
        ScopeLogic::class => ScopeLogicService::class,
        RevisionServiceContract::class => RevisionService::class,
        ContentBlueprintsContract::class => ContentBlueprints::class,
    ];

    protected array $commands = [
        RebuildModelsCommand::class,
        SyncPivotsCommand::class,
        ExportContentTypeCommand::class,
        ExportGedCommand::class,
        DuplicateGedCommand::class,
        ExportSectionTypeCommand::class,
        FindMissingFilesCommand::class,
        ImportContentCommand::class,
        ExportContentCommand::class,
        MakeContentTypeCommand::class,
        GenerateAllImageCommand::class,
        CleanUploadsCommand::class,
        RepairWysiwygImagesCommand::class,
        RepairPublicationFields::class,
        DeleteContentCommand::class,
        InitRevisionCommand::class,
        CleanRevisionOfDeletedContentCommand::class,
        RepairUploadLimitCommand::class,
        RepairFileExtensionsCommand::class,
        RepairImageExtensionsCommand::class,
        RepairVideoExtensionsCommand::class,
        MakeRattachedBundlesTranslatable::class,
        GenerateConfigurations::class,
        EnableParagraph::class,
        ListUnavailableParagraphs::class,
        ImportConfigurations::class,
        ImportWTTJJobOffersCommand::class,
        CleanUsersProfileImageCommand::class,
        FixMultilingualLabels::class,
        ImportContentFromInside1Command::class,
        PurgeOldContentsCommand::class,
        ReplaceUuidsWithHostInRolesTableCommand::class,
        SetContentTypesAsTranslatableCommand::class,
        ImportStockPricesCommand::class,
        AddMissingWysiwygImagesCommand::class,
        ImportContentFromCsvCommand::class,
    ];

    public function register(): void
    {
        try {
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/allowed_mimetypes.php', 'allowed_mimetypes');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/importable_content_types.php', 'importable_content_types');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/external_api.php', 'external_api');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/scheduler.php', 'scheduler');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/resource.php', 'resource');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/imports.php', 'imports');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/contents.php', 'contents');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/events.php', 'events');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/content-trees.php', 'content-trees');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/blueprints.php', 'blueprints');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/wttj.php', 'wttj');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/generator/config.php', 'generator.config');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/generator/files.php', 'generator.files');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/generator/languages.php', 'generator.languages');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/generator/paragraphs.php', 'generator.paragraphs');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/user-seen.php', 'user-seen');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/news_followers.php', 'news_followers');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/duplication.php', 'duplication');
        } catch (NotFoundExceptionInterface | ContainerExceptionInterface) {
        }

        // Add special exception for wysiwyg images path
        $this->app['config']->set(
            'authentication',
            array_merge(
                $this->app['config']->get('authentication', []),
                [
                    'direct_access_exceptions' => ['#^wysiwyg/images/.*$#'],
                ]
            )
        );

        $this->app->middleware([
            UserMentionRequest::class,
            AddWysiwygImagesOnContentForm::class,
            SaveWysiwygImages::class,
        ]);

        $this->app->bind(ContentListContract::class, function () {
            return new ContentList(new ContentQuery());
        });

        $this->app->bind(ContentExternalListContract::class, function () {
            return new ContentExternalList(new ContentTransformer());
        });

        $this->app->singleton(
            ClassesService::class,
            function () {
                return new ClassesService(app(SchemaService::class), new Filesystem(), app(Composer::class));
            }
        );

        if (config('app.debugging_events', false) === true) {
            Event::listen('*', function ($eventName, $arguments) {
                if (str_starts_with($eventName, 'Inside\\Content\\Events')) {
                    $info = '['.$eventName.']';
                    if (count($arguments) !== 1) {
                        return;
                    }
                    $event = $arguments[0];
                    if (property_exists($event, 'model') && ! is_null($event->model)) {
                        $info .= ' on ['.get_class($event->model).']('.$event->model->uuid.')';
                    }

                    Log::debug($info);
                }
            });
        }

        $this->app->singleton(
            ShortUrlManager::class,
            function ($app) {
                return new ShortUrlManager(
                    new UniqueKeyGenerator(
                        new Hashids(
                            $app['config']['app.secret'],
                            $app['config']['content.short_url_length']
                        )
                    )
                );
            }
        );

        $this->app->singleton(
            QueryHelper::class,
            function (Application $app) {
                return new ContentQueryHelper($app->getMiddlewares());
            }
        );

        $this->app->register(PurifierServiceProvider::class);

        // Register uuid to doctrine
        try {
            Type::addType('uuid', UuidType::class);
        } catch (Exception $exception) {
            Log::error('[ContentServiceProvider::handle] failed to add uuid type do Doctrine DBAL');
        }

        // Register timestamp to doctrine
        try {
            Type::addType('timestamp', TimestampType::class);
        } catch (Exception $exception) {
            Log::error('[ContentServiceProvider::handle] failed to add timestamp type do Doctrine DBAL');
        }

        // Punicode
        if (class_exists('\\Fomvasss\\Punycode\\PunycodeServiceProvider')) {
            $this->app->register(PunycodeServiceProvider::class);
        }

        if ($this->app->runningInConsole()) {
            $this->app->singleton(
                'command.content.csv',
                function ($app) {
                    return new ImportContentTypeFromCsvCommand(new Filesystem());
                }
            );
        }
    }

    public function boot(): void
    {
        parent::boot();

        $this->mergeConfigFrom(__DIR__.'/../../config/openweathermap.php', 'openweathermap');

        $this->app->singleton(OpenWeatherService::class, function () {
            return new OpenWeatherService(
                apiKey: config('openweathermap.api_key') ?? '',
                apiUrl: config('openweathermap.api_url')
            );
        });
    }
}
