<?php

declare(strict_types=1);

namespace Inside\Content\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder as EloquentBuilder;
use Illuminate\Database\Eloquent\Model as EloquentModel;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Facades\Storage;
use Inside\Support\HasMimeTypes;

/**
 * @property string $filename
 * @property string $hash
 * @property string $path
 * @property string $url
 * @property string $relative_url
 * @property string|null $mimetype
 * @property Carbon $created_at
 * @property-read int $id
 * @property string $disk
 * @property string $base_path
 * @property string $internal
 * @property string $internal_path
 * @property ?string $imageable_id
 * @property ?string $imageable_type
 * @property int $size
 *
 * @method static WysiwygImage create(array $attributes = [])
 * @method static EloquentBuilder|WysiwygImage whereYear($column, $operator, $value = null, $boolean = 'and')
 * @method static EloquentBuilder|WysiwygImage whereMonth($column, $operator, $value = null, $boolean = 'and')
 *
 * @mixin EloquentBuilder
 */
#[\AllowDynamicProperties]
final class WysiwygImage extends EloquentModel
{
    use HasMimeTypes;

    protected $fillable = [
        'filename',
        'mimetype',
        'size',
        'disk',
        'internal',
        'internal_path',
        'hash',
        'imageable_type',
        'imageable_id',
    ];

    protected static function boot()
    {
        parent::boot();

        self::deleted(function (self $image) {
            $disk = Storage::disk($image->disk);
            // Delete file
            if ($disk->exists($image->base_path.$image->internal_path.$image->internal)) {
                $disk->delete($image->base_path.$image->internal_path.$image->internal);
                if (empty($disk->files($image->base_path))) {
                    $disk->deleteDirectory($image->base_path);
                }
            }
        });
    }

    public function content(): MorphTo
    {
        return $this->morphTo('imageable');
    }

    public function getBasePathAttribute(): string
    {
        if (is_null($this->imageable_id)) {
            return 'temp';
        } else {
            return class_to_type((string) $this->imageable_type).'/'.$this->imageable_id;
        }
    }

    public function getPathAttribute(): string
    {
        return Storage::disk($this->disk)->path($this->base_path.$this->internal_path.$this->internal);
    }

    public function getUrlAttribute(): string
    {
        if (! is_null($this->mimetype)) {
            $extension = $this->getExtensions($this->mimetype);
            if (empty($extension)) {
                throw (new ModelNotFoundException())->setModel(get_class($this));
            }

            return Storage::disk($this->disk)->url($this->created_at->year.
                '/'.$this->created_at->month.'/'.$this->filename);
        }

        return '';
    }

    public function getRelativeUrlAttribute(): string
    {
        $appUrl = config('app.url');
        if (! str_ends_with($appUrl, '/')) {
            $appUrl .= '/';
        }

        return str_replace($appUrl, '', $this->url);
    }
}
