<?php

namespace Inside\Content\Models\Traits;

use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Users;
use Inside\Content\Models\Section;
use Inside\Permission\Models\Role;
use Inside\Support\Str;

trait HasAuthor
{
    public static function bootHasAuthor(): void
    {
        static::creating(function ($model) {
            if (Str::endsWith(class_to_type($model), '_menus')) {
                return; // Menu don't have author_id it's full drupal stuff
            }
            if (is_subclass_of($model, Section::class)) {
                $model->author = $model->author_id = $model->sectionable->author;
            } else {
                $model->author_id = $model->author;
            }
            // In some circonstances where author is not relevant, we don't have one, let's give it one so that
            // foreign key system is respected
            if (is_null($model->author)) {
                $model->author = $model->author_id = Role::where('name', 'super_administrator')
                    ->firstOrFail()
                    ->users()
                    ->first()
                    ->id;
            }
        });
    }

    /**
     * Author of this entity
     *
     * @return BelongsTo
     * @deprecated author is column name ... waiting for author to be removed
     */
    public function author(): BelongsTo
    {
        return $this->belongsTo('\Inside\Content\Models\Contents\Users', 'author_id', 'uuid');
    }

    /**
     * Author of this entity as an auth user
     *
     * @return BelongsTo
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'author_id', 'uuid');
    }

    /**
     * @return BelongsTo
     * @deprecated use creator
     * @see        creator
     */
    public function authors(): BelongsTo
    {
        return $this->belongsTo(Users::class, 'author_id', 'uuid');
    }

    /**
     * @return BelongsTo
     */
    public function updateAuthor(): BelongsTo
    {
        return $this->belongsTo(Users::class, 'update_author', 'uuid');
    }

    /**
     * @return BelongsTo
     */
    public function modificator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'update_author', 'uuid');
    }
}
