<?php

namespace Inside\Content\Models;

use Closure;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model as BaseModel;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Event;
use Inside\Content\Events\ContentFieldCreatedEvent;
use Inside\Content\Events\ContentFieldDeletedEvent;
use Inside\Content\Events\ContentFieldUpdatedEvent;

/**
 * @property int $id
 * @property int $model_id
 * @property string $name
 * @property string $type
 * @property string $class
 * @property array|null $options
 * @property bool $displayed
 * @property-read \Illuminate\Database\Eloquent\Collection|Field[] $fields
 * @property-read int|null $fields_count
 * @property-read Model $model
 * @method static Builder|Field newModelQuery()
 * @method static Builder|Field newQuery()
 * @method static Builder|Field query()
 * @method static Builder|Field whereDisplayed($value)
 * @method static Builder|Field whereId($value)
 * @method static Builder|Field whereModelId($value)
 * @method static Builder|Field whereName($value)
 * @method static Builder|Field whereOptions($value)
 * @method static Builder|Field whereType($value)
 * @method static Field select($columns = ['*'])
 * @method static Builder where($column, $operator = null, $value = null, $boolean = 'and')
 * @method static Builder whereHas($relation, Closure $callback = null, $operator = '>=', $count = 1)
 * @method static \Inside\Database\Eloquent\Builder whereNotIn(string $column, mixed $values, string $boolean = 'and')
 *
 * @mixin Builder
 * @mixin \Eloquent
 */
class Field extends BaseModel
{
    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'inside_fields';

    /**
     * The attributes that are guarded.
     *
     * @var array
     */
    protected $fillable = ['model_id', 'name', 'type', 'title', 'options', 'displayed'];

    /**
     * The attributes that are guarded.
     *
     * @var array
     */
    protected $guarded = ['id'];

    /**
     * The attributes that are hidden.
     *
     * @var array
     */
    protected $hidden = ['id'];

    /**
     * Remove the timestamps.
     *
     * @var bool
     */
    public $timestamps = false;

    /**
     * Fields to cast.
     *
     * @var array
     */
    protected $casts = [
        'options'   => 'array',
        'displayed' => 'boolean',
    ];

    protected $dispatchesEvents = [
        'updating' => \Inside\Content\Events\ContentFieldUpdatingEvent::class,
    ];

    public static function boot()
    {
        parent::boot();

        self::created(
            function ($field) {
                Event::dispatch(new ContentFieldCreatedEvent($field));
            }
        );

        self::updated(
            function ($field) {
                Event::dispatch(new ContentFieldUpdatedEvent($field));
            }
        );

        self::deleted(
            function ($field) {
                Event::dispatch(new ContentFieldDeletedEvent($field));
            }
        );
    }

    /**
     * @return BelongsTo
     */
    public function model(): BelongsTo
    {
        return $this->belongsTo(Model::class, 'model_id', 'id');
    }
}
