<?php

namespace Inside\Content\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Contents\Users;
use Inside\Content\Models\Traits\CanRunWithoutEvents;
use Inside\Content\Models\Traits\HasAuthor;
use Inside\Content\Models\Traits\HasSection;
use Inside\Content\Models\Traits\HasStatus;
use Inside\Database\Eloquent\WithEnhancedBuilder;
use Inside\Host\Exodus\Traits\HasThirdPartySettingsConfiguration;
use Inside\Permission\Exodus\Dto\Indexes\PermissibleContentIndexDto;
use Inside\Permission\Exodus\Dto\Privileges\ContentPrivilegeDto;
use Inside\Permission\Exodus\Models\Capability;
use Inside\Permission\Exodus\Models\Privileges\CategorizableContentPrivilege;
use Inside\Permission\Exodus\Models\Privileges\ContentPrivilege;
use Inside\Permission\Exodus\Models\Privileges\ContentSpecificPrivilege;
use Inside\Permission\Exodus\Models\Traits\HasPrivileges;
use Inside\Permission\Exodus\Models\ViewModels\CategorizableContentTranslationIndexes;
use Inside\Reaction\Models\Reaction;

/**
 * Class Content
 *
 * Base Model class for all Content Models
 *
 * @property-read ?User $creator
 * @property-read Users $authors
 * @property-read ?User $modificator
 * @property-read Users $update_author
 * @property-read string $content_type
 * @property bool $status
 * @property string $author
 * @property string $author_id
 * @property-read Field[] $fields
 * @property-read Collection|Reaction[] $reactions
 * @property-read int|null $reactions_count
 * @property int|null $created_at
 * @property int|null $updated_at
 * @property-read int $published_at
 * @property string|null $uuid
 * @property string $uuid_host
 * @property string|null $pid
 * @property string $langcode
 * @property string $title
 * @method static \Inside\Database\Eloquent\Builder|Content newModelQuery()
 * @method static \Inside\Database\Eloquent\Builder|Content newQuery()
 * @method static \Inside\Database\Eloquent\Builder|Content query()
 * @method Content getTranslationIfExists(string $langcode)
 * @method Collection getChildrenIfExist(array $filters = [])
 */
class Content extends Model
{
    use CanRunWithoutEvents, HasAuthor, HasSection, HasStatus, HasPrivileges, HasThirdPartySettingsConfiguration, WithEnhancedBuilder;

    public function reactions(): MorphMany
    {
        return $this->morphMany(
            '\Inside\Reaction\Models\Reaction',
            'reactionable',
            'reactionable_type',
            'reactionable_uuid',
            'uuid'
        );
    }

    /**
     * @param  string  $langcode
     * @return mixed
     * @deprecated use getTranslationIfExists
     * @see        getTranslationIfExists
     */
    public function getTranslationIfExist(string $langcode)
    {
        return $this->getTranslationIfExists($langcode);
    }

    /**
     * Magic call for sluggable construction
     *
     * @return array<string, array<string, string>>
     */
    public function sluggable(): array
    {
        return [
            'title' => [
                'source' => 'title',
            ],
        ];
    }

    /**
     * Get the content type for this content
     */
    public function getContentTypeAttribute(): string
    {
        return class_to_type($this);
    }

    /**
     * Fields for this model
     *
     * @return Field[]|Builder[]|Collection|\Illuminate\Support\Collection
     */
    public function getFieldsAttribute()
    {
        return Field::whereHas(
            'model',
            function ($query) {
                $query->whereClass(get_class($this));
            }
        )->get();
    }

    /**
     * @return array
     */
    public function getTranslationIds(): array
    {
        return $this->getTranslations(['uuid', 'langcode']);
    }

    public function getTranslations(array $fields): array
    {
        $data = [];
        $languages = list_languages();
        $query = call_user_func(get_class($this).'::query');

        $prefixWithTable = fn (string $field) => str($this->getTable())
            ->append('.')
            ->finish($field)
            ->toString;

        $querySelectors = collect(array_diff($fields, ['slug']))
            ->map(fn (string $fieldName) => $prefixWithTable($fieldName))
            ->all();

        $translations = $query
            ->select($querySelectors)
            ->where('uuid_host', $this->uuid_host)
            ->when(in_array('slug', $fields), fn (Builder $query) => $query
                ->leftJoin('inside_slugs', $prefixWithTable('uuid'), '=', 'inside_slugs.uuid')
                ->addSelect('inside_slugs.slug')
            )
            ->get()
            ->toArray();

        foreach ($translations as $translation) {
            $translationsByLang[$translation['langcode']] = collect($fields)->map(function ($fieldName) use ($translation) {
                return [$fieldName => $translation[$fieldName]];
            })->collapse()->toArray();
        }

        foreach ($languages as $langCode) {
            $data[$langCode]['uuid'] = '';
            if (isset($translationsByLang[$langCode])) {
                $data[$langCode] = $translationsByLang[$langCode];
            }
        }

        return $data;
    }

    /**
     * @param mixed $query
     * @param  array  $filters
     * @param string|null $table
     * @return mixed
     */
    protected function applyFilterToChildren($query, array $filters = [], string $table = null)
    {
        if (! $table) {
            $table = $this->getTable();
        }

        foreach ($filters as $field => $value) {
            if (false !== strpos($field, ':')) {
                [$sanitizedField, $operator] = explode(':', $field, 2);
            } else {
                $sanitizedField = $field;
                $operator = null;
            }

            if ($table == 'inside_content_users' && $sanitizedField == 'langcode') {
                continue;
            }

            switch ($sanitizedField) {
                case 'limit':
                case 'offset':
                case 'sort':
                case 'paginate':
                    break;
                case 'parent_uuid':
                    $query->join(
                        DB::raw('`inside_pivots` AS `parent_pivots`'),
                        $table.'.uuid',
                        '=',
                        'parent_pivots.parent_uuid'
                    )->where('parent_pivots.parent_uuid', '=', $value);
                    break;
                case 'slug':
                    $exploded = explode('/', $value);
                    $uuids = DB::select(
                        'SELECT inside_slugs.uuid FROM inside_slugs '.'INNER JOIN '.$table
                        .' ON inside_slugs.uuid = '.$table.'.uuid '.'WHERE '.$table.'.uuid_host IN ('
                        .'SELECT content.uuid_host FROM '.$table.' AS content '
                        .'INNER JOIN inside_slugs AS slug ON slug.uuid = content.uuid '.'WHERE content.uuid_host = '
                        .$table.'.uuid_host '.'AND slug.slug = "'.$exploded[0].'"'.')'
                    );
                    $uuids = array_map(
                        function ($item) {
                            return $item->uuid;
                        },
                        $uuids
                    );

                    $query->join('inside_slugs', $table.'.uuid', '=', 'inside_slugs.uuid')->whereIn(
                        'inside_slugs.uuid',
                        $uuids
                    );
                    break;
                default:
                    if (! InsideSchema::hasField(class_to_type($this), $sanitizedField) && ! is_array($value)) {
                        continue 2;
                    }

                    $operator = isset($operator) ? $this->getOperator($operator) : '=';

                    if ($operator == 'not in') {
                        $query->whereNotIn($table.'.'.$sanitizedField, $value);
                        break;
                    }

                    if ($operator == 'in') {
                        $query->whereIn($table.'.'.$sanitizedField, $value);
                        break;
                    }

                    if ($operator == 'null') {
                        $query->whereNull($table.'.'.$sanitizedField);
                        break;
                    }

                    if ($operator == 'notnull') {
                        $query->whereNotNull($table.'.'.$sanitizedField);
                        break;
                    }

                    if (is_array($value) && count(array_filter(array_keys($value), 'is_string'))) {
                        if (InsideSchema::hasField(class_to_type($query->getModel()), $sanitizedField)) {
                            $options =
                                InsideSchema::getFieldOptions(class_to_type($query->getModel()), $sanitizedField);
                            $target = $options['target'][0] ?? null;
                            if ($target !== null) {
                                $query->whereHas(
                                    Str::camel($sanitizedField),
                                    function ($query) use ($value, $target) {
                                        return $this->applyFilterToChildren(
                                            $query,
                                            $value,
                                            type_to_table($target)
                                        );
                                    }
                                );
                            }
                        }
                        break;
                    }

                    $query->where($table.'.'.$sanitizedField, $operator, $value);
                    break;
            }
        }

        return $query;
    }

    /**
     * @param  string  $operator
     * @return string|null
     */
    protected function getOperator(string $operator): ?string
    {
        return match ($operator) {
            'in' => 'in',
            'notin' => 'not in',
            'ne' => '!=',
            'eq' => '=',
            'gt' => '>',
            'gte' => '>=',
            'lt' => '<',
            'lte' => '<=',
            'like' => 'like',
            'null' => 'null',
            'notnull' => 'notnull',
            default => '=',
        };
    }

    /**
     * Follow parents to the root & return all parents
     *
     * @return array
     */
    public function followParents(): array
    {
        if ($this->parent) {
            return array_merge([$this->parent], $this->parent->followParents());
        }

        return [];
    }

    /**
     * Get all parents by type from any relation fields. $reversed is used to get reversed relation field parents.
     *
     * @param  array  $filters
     * @param  bool  $reversed
     * @return array
     */
    public function getParentsFromRelationFields(array $filters = [], bool $reversed = false): array
    {
        // Prepare field prefixes
        $fieldPrefix = $reversed ? 'related' : 'parent';
        $relatedFieldPrefix = $reversed ? 'parent' : 'related';

        // Use pivot table to get parents
        $query = DB::table('inside_pivots')->where('inside_pivots.'.$fieldPrefix.'_uuid', $this->getKey())->where(
            'inside_pivots.'.$fieldPrefix.'_type',
            get_class($this)
        );
        if ($reversed) {
            // We only want content types ( no sections ) and not current type
            $query->whereIn(
                'inside_pivots.'.$relatedFieldPrefix.'_type',
                Arr::where(
                    array_map(
                        function ($type) {
                            return type_to_class($type);
                        },
                        InsideSchema::getContentTypes()
                    ),
                    function ($class) {
                        return $class != get_class($this);
                    }
                )
            );
        }

        if ($this->langcode) {
            $query->where($relatedFieldPrefix.'_langcode', $this->langcode);
        }
        if (isset($filters['content_type'])) {
            if (is_array($filters['content_type'])) {
                $classes = [];
                foreach ($filters['content_type'] as $type) {
                    $class = type_to_class($type);
                    if (class_exists($class)) {
                        $classes[] = $class;
                    }
                }
                $query->whereIn('inside_pivots.'.$relatedFieldPrefix.'_type', $classes);
            } elseif (is_string($filters['content_type'])) {
                $class = type_to_class($filters['content_type']);
                if (class_exists($class)) {
                    $query->where('inside_pivots.'.$relatedFieldPrefix.'_type', $class);
                }
            }
        }
        $parents = $query->get();
        $results = [];
        foreach ($parents as $parent) {
            $parentEntity = call_user_func(
                $parent->{$relatedFieldPrefix.'_type'}.'::find',
                $parent->{$relatedFieldPrefix.'_uuid'}
            );
            if ($parentEntity) {
                // Only get existing entity ( if $parentEntity is null, pivot is broken )
                $results[class_to_type($parentEntity)] = array_merge([$parentEntity], $parentEntity->followParents());
            }
        }
        // Menu & comment content type don't use pivots, they use parent instead
        if ($this->isMenuType() || class_to_type($this) === 'comments') {
            if (! isset($results[class_to_type($this)])) {
                $results[class_to_type($this)] = $this->followParents();
            }
        }

        return $results;
    }

    /**
     * Get Parents if exists filtered by filters
     *
     * @param  array  $filters
     * @return array
     */
    public function getParentsIfExist(array $filters = []): array
    {
        $results = [];

        // Get all parents from both reversed and direction relation fields
        $results = array_merge($results, $this->getParentsFromRelationFields($filters));
        $results = array_merge($results, $this->getParentsFromRelationFields($filters, true));

        // Menu content type don't use pivots, they use parent instead
        if ($this->isMenuType()) {
            if (! isset($results[class_to_type($this)])) {
                $results[class_to_type($this)] = $this->followParents();
            }
        }

        return $results;
    }

    /**
     * Get content options
     *
     * @param  string  $key
     * @return bool|mixed
     */
    public function __get($key)
    {
        if (str($key)->startsWith('is') && method_exists(static::class, $key)) {
            return $this::{$key}();
        }

        if (! InsideSchema::hasModel(class_to_type($this)) || ! InsideSchema::isModel(class_to_type($this))) {
            return parent::__get($key);
        }

        try {
            $options = InsideSchema::getModelOptions(class_to_type($this));
            $optionKey = Str::camel(Str::after($key, 'is'));
            if (isset($options[$optionKey])) {
                if ($options[$optionKey] === 1 || $options[$optionKey] === true) {
                    return true;
                } elseif ($options[$optionKey] === 0 || $options[$optionKey] === false) {
                    return false;
                }
            }
        } catch (ModelSchemaNotFoundException $exception) {
            Log::warning('Model ['.get_class($this).'] options is missing! ', [
                'message' => $exception->getMessage(),
            ]);
        }

        return parent::__get($key);
    }

    public function images(): MorphMany
    {
        return $this->morphMany(WysiwygImage::class, 'imageable');
    }

    /**
     * Says if this content type is a menu or not
     *
     * @return bool
     */
    public function isMenuType(): bool
    {
        return Str::endsWith(class_to_type($this), '_menus');
    }

    public function translation(): MorphOne
    {
        return $this->morphOne(CategorizableContentTranslationIndexes::class, 'translatable', 'translatable_type', 'translatable_uuid', 'uuid');
    }

    public function hasSpecificPrivilege(): bool
    {
        if (! config('permission.specific_permissions_enabled')) {
            return false;
        }

        return ContentSpecificPrivilege::where('uuid', $this->uuid)->whereHas('roles')->exists();
    }

    public static function getCategorizableRequiredRelations(): array
    {
        $relations = collect(Schema::getRelationsWithOption(class_to_type(static::class)))
            ->filter(fn ($relation) => $relation['required'])
            ->map(fn ($relation) => $relation['target'][0])
            ->filter(fn ($relation) => type_to_class($relation)::isCategorizable())
            ->toArray();

        return $relations;
    }

    /**
     * For Permissions V2, this method is used to get the restriction requirements for this content.
     * It will return an array with the content type and the relations that are required for this content.
     * When it's not categorizable, you can use the content type to get the privileges.
     * When it's categorizable, you can use the relations to get the privileges on the specific content.
     *
     * @return array
     */
    public function getRestrictionRequirements(): array
    {
        $requirements = [
            'categorizables' => [],
            'content_types' => [],
        ];

        if (! $this::isPermissible()) {
            return $requirements;
        }

        $relations = collect(Schema::getRelationsWithOption($this->content_type))
            ->filter(fn ($relation) => $relation['required'])
            ->each(function ($relation, $method) use (&$requirements) {
                $type = $relation['target'][0];
                $classType = type_to_class($type);

                if ($classType::isCategorizable()) {
                    $requirements['categorizables'][$classType] = $this->{$method}()->get()->pluck('uuid_host')->toArray();
                } elseif ($classType !== Users::class && $classType::isPermissible()) {
                    $requirements['content_types'][] = $classType;
                }
            });

        return $requirements;
    }

    public function getPrivileges(): array
    {
        $table = $this::isCategorizable() ? CategorizableContentPrivilege::TABLE : ContentPrivilege::TABLE;

        return DB::table($table)
            ->join(Capability::TABLE, $table.'.capability_id', Capability::TABLE.'.id')
            ->select($table.'.id', Capability::TABLE.'.name')
            ->when($this::isCategorizable(),
                fn ($query) => $query->where('uuid_host', $this->uuid_host),
                fn ($query) => $query->where('uuid', $this->uuid),
            )->pluck('id', 'name')->toArray();
    }

    public function isOwnedBy(User $user): bool
    {
        return $this->author_id === $user->id;
    }
}
