<?php

namespace Inside\Content\Mail;

use Illuminate\Bus\Queueable;
use Illuminate\Mail\Mailable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Lang;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Slug\Facades\ContentUrlGenerator;

class ContentReported extends Mailable
{
    use Queueable;
    use SerializesModels;

    protected string $subjectMail = 'report.mail.subject';

    protected string $text = 'report.mail.content';

    protected string $button = 'report.mail.button';

    public function __construct(
        protected User $user,
        protected Content $content,
        protected string $reason,
        protected User $reporter
    ) {
    }

    public function build(): self
    {
        $replacements = [
            'model.body' => $this->content->title,
            'author.fullname' => trim(trim((string) $this->reporter->firstname).' '.trim((string) $this->reporter->lastname)),
            'author.email' => $this->reporter->email,
            'reason' => trim($this->reason),
        ];
        $url = $this->buildUrl();

        if (class_to_type($this->content) === 'comments') {
            $replacements['model.body'] = $this->content->body;
            $this->subjectMail = 'report.comments.mail.subject';
            $this->text = 'report.comments.mail.content';
            $this->button = 'report.comments.mail.button';
        }

        return $this->markdown('emails.reported-content')
            ->subject(
                Lang::get($this->subjectMail)
            )
            ->with(
                [
                    'lines' => explode(
                        "\n",
                        trim(Lang::get($this->text, $replacements))
                    ),
                    'actionUrl' => env('APP_URL').'/'.$url,
                    'actionText' => Lang::get($this->button),
                ]
            );
    }

    /**
     * Build callback url
     *
     * @return string
     */
    public function buildUrl()
    {
        if (class_to_type($this->content) !== 'comments') {
            return 'edit/'.class_to_type($this->content).'/'.$this->content->uuid;
        }

        $parents = $this->content->getParentsIfExist();
        $scrollComment = $this->content;

        // Merge parents
        if ($this->content->parent) {
            $parents['comments'] = [$this->content->parent];
        }

        $model = $parent = null;
        $parent = Arr::last($parents['comments']) ?? null;
        if (! empty($parent)) {
            $scrollComment = Arr::first($parents['comments']) ?? null;
            foreach ($parents as $content_type => $contentParents) {
                if ($content_type != 'comments' && ! empty($contentParents)) {
                    $model = Arr::first($contentParents);
                    break;
                }
            }
        } else {
            foreach ($parents as $content_type => $contentParents) {
                if ($content_type != 'comments' && ! empty($contentParents)) {
                    $model = $parent = Arr::first($contentParents);
                    break;
                }
            }
        }

        $url = (ContentUrlGenerator::generateUrl($model) ?? '').'#&d/'.$scrollComment->uuid;

        return $url;
    }
}
