<?php

namespace Inside\Content\Listeners;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Events\ContentDeletedEvent;
use Inside\Content\Events\ContentFieldDeletedEvent;
use Inside\Content\Events\ContentFieldUpdatingEvent;
use Inside\Content\Events\ContentFullyInsertedEvent;
use Inside\Content\Events\ContentUpdatingEvent;
use Inside\Content\Facades\Schema;
use Inside\Content\Jobs\GenerateStylizedImage;
use Inside\Content\Models\Contents\ImageStyles;

/**
 * Class ImageStylesSubscriber
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class ImageStylesSubscriber
{
    /**
     * Manage content creation
     *
     * A content has been fully inserted
     * We must create or update stylised image if needed
     *
     * @param ContentFullyInsertedEvent $event
     */
    public function handleContentFullyInserted(ContentFullyInsertedEvent $event): void
    {
        $model = $event->model;

        if (get_class($model) != ImageStyles::class) {
            // Any content type
            $type = class_to_type($model);

            $imageFields = Schema::getFieldListingOfType($type, 'image');

            foreach ($imageFields as $fieldName) {
                $image = $model->{$fieldName};
                if (! $image) {
                    continue; // No image set!
                }

                $options = Schema::getFieldOptions($type, $fieldName);

                if (isset($options['image_styles']) && ! empty($options['image_styles'])) {
                    $styles = ImageStyles::query()->whereIn('title', $options['image_styles'])->get();
                    foreach ($styles as $style) {
                        GenerateStylizedImage::dispatch($image, $style)->onQueue(get_images_queue_name());
                    }
                }
            }
        }
    }

    /**
     * Manage image styles changing ! or content image field changing
     *
     * @param ContentUpdatingEvent $event
     */
    public function handleContentUpdating(ContentUpdatingEvent $event): void
    {
        $model = $event->model;
        if (! $model->isDirty()) {
            return;
        }
        if (get_class($model) == ImageStyles::class) {
            // Image style title changed, rename directory
            if ($model->isDirty('title')) {
                Storage::disk('local')->move(
                    'styles/'.$this->getStyleDirectoryName($model, true),
                    'styles/'.$this->getStyleDirectoryName($model)
                );
            }

            // Regenerate all content image for this style
            $imageFields = Schema::getAllFieldsListingOfType('image');

            // Prepare needes types
            $styles = [];
            foreach ($imageFields as $modelName => $fields) {
                foreach ($fields as $fieldName) {
                    $options = Schema::getFieldOptions($modelName, $fieldName);
                    if (! isset($options['image_styles'])) {
                        continue;
                    }
                    $styles[$fieldName] = $model;
                }
            }
            // Rebuild images
            foreach ($imageFields as $modelName => $fields) {
                $query = call_user_func(
                    (Schema::isSectionType($modelName) ? section_type_to_class($modelName) : type_to_class($modelName))
                    .'::query'
                );
                $contents = $query->where('status', true)->get();
                foreach ($contents as $content) {
                    foreach ($fields as $fieldName) {
                        $image = $content->{$fieldName};
                        if ($image && isset($styles[$fieldName])) {
                            GenerateStylizedImage::dispatch($image, $styles[$fieldName])->onQueue(
                                get_images_queue_name()
                            );
                        }
                    }
                }
            }
        } else {
            // This is not an image style, could be any content
            $type = class_to_type($model);

            // Let's check if a stylized image has changed
            $imageFields = Schema::getFieldListingOfType($type, 'image');

            $disk = Storage::disk('local');

            foreach ($imageFields as $fieldName) {
                if ($model->isDirty($fieldName) && $model->{$fieldName} !== null) {
                    $options = Schema::getFieldOptions($type, $fieldName);

                    if (isset($options['image_styles']) && ! empty($options['image_styles'])) {
                        $styles = ImageStyles::query()->whereIn('title', $options['image_styles'])->get();
                        foreach ($styles as $style) {
                            $disk->delete(
                                'styles/'.$this->getStyleDirectoryName($style).'/'.$model->getOriginal(
                                    $fieldName
                                )
                            );
                            GenerateStylizedImage::dispatch($model->{$fieldName}, $style)->onQueue(
                                get_images_queue_name()
                            );
                        }
                    }
                }
            }
        }
    }

    /**
     * When deleting ImageStyles, we should delete style diretory
     *
     * @param ContentDeletedEvent $event
     */
    public function handleContentDeleted(ContentDeletedEvent $event): void
    {
        $model = $event->model;

        if (get_class($model) == ImageStyles::class) {
            // Delete style directory
            Storage::disk('local')->deleteDirectory(
                'styles/'.$this->getStyleDirectoryName($model)
            );
        } else {
            $type = class_to_type($model);

            // Delete stylised image from this content
            $imageFields = Schema::getFieldListingOfType($type, 'image');

            foreach ($imageFields as $fieldName) {
                $image = $model->{$fieldName};
                if (! $image) {
                    continue; // No image set!
                }
                $options = Schema::getFieldOptions($type, $fieldName);

                if (isset($options['image_styles']) && ! empty($options['image_styles'])) {
                    $styles = ImageStyles::whereIn('title', $options['image_styles'])->get();
                    foreach ($styles as $style) {
                        Storage::disk('local')->delete(
                            'styles/'.$this->getStyleDirectoryName($style).'/'.$image
                        );
                    }
                }
            }
        }
    }

    /**
     * Regnerate styles on image field update
     *
     * @param ContentFieldUpdatingEvent $event
     */
    public function handleContentFieldUpdating(ContentFieldUpdatingEvent $event): void
    {
        $field = $event->field;
        if ($field->type == 'image' && $field->isDirty('options')) {
            if (isset($field->options['image_styles'])) {
                $newStyles = $field->options['image_styles'];
                $oldOptions = json_decode($field->getOriginal('options'), true);
                $oldStyles = [];
                if (isset($oldOptions['image_styles'])) {
                    $oldStyles = $oldOptions['image_styles'];
                }

                foreach (ImageStyles::all() as $style) {
                    if (in_array($style->title, $oldStyles) && in_array($style->title, $newStyles)) {
                        // this style was in our styles and it still is there nothing to do!
                    } elseif (! in_array($style->title, $oldStyles) && in_array($style->title, $newStyles)) {
                        // It's a new style, generate images for this new style !!
                        $field->model()->each(
                            function ($model) use ($field, $style) {
                                $contents = call_user_func($model->class.'::query')->get();

                                foreach ($contents as $content) {
                                    if ($content->{$field->name} !== null && ! empty($content->{$field->name})) {
                                        Log::debug(
                                            '[GenerateStylizedImage] dispatch ['.$content->{$field->name}.'] style {'
                                            .$style->title.'}'
                                        );
                                        GenerateStylizedImage::dispatch($content->{$field->name}, $style)->onQueue(
                                            get_images_queue_name()
                                        );
                                    }
                                }
                            }
                        );
                    } elseif (in_array($style->title, $oldStyles) && ! in_array($style->title, $newStyles)) {
                        // Old style removed !
                        $field->model()->each(
                            function ($model) use ($field, $style) {
                                $contents = call_user_func($model->class.'::query')->get();

                                foreach ($contents as $content) {
                                    if ($content->{$field->name} !== null && ! empty($content->{$field->name})) {
                                        $disk = Storage::disk('local');
                                        $path = 'styles/'.$this->getStyleDirectoryName($style).'/'
                                            .$content->{$field->name};
                                        if ($disk->exists($path)) {
                                            Log::debug(
                                                '[GenerateStylizedImage] delete ['.'styles/'
                                                .$this->getStyleDirectoryName(
                                                    $style
                                                ).'/'.$content->{$field->name}.'] style {'.$style->title.'}'
                                            );

                                            $disk->delete($path);
                                        }
                                    }
                                }
                            }
                        );
                    } else {
                        // Not a new one, nothing to do
                    }
                }
            }
        }
    }

    /**
     * Field deletion, if type is image we should delete generated image
     *
     * @param ContentFieldDeletedEvent $event
     */
    public function handleContentFieldDeleted(ContentFieldDeletedEvent $event): void
    {
        $field = $event->field;

        if ($field->type == 'image') {
            if (isset($field->options['image_styles'])) {
                foreach (ImageStyles::query()->whereIn('title', $field->options['image_styles'])->get() as $style) {
                    $field->class::each(
                        function ($content) use ($field, $style) {
                            if ($content->{$field->name}) {
                                Storage::disk('local')->delete(
                                    'styles/'.$this->getStyleDirectoryName($style).'/'.$content->{$field->name}
                                );
                            }
                        }
                    );
                }
            }
        }
    }

    /**
     * Get unique image style
     *
     * @param ImageStyles $style
     * @param bool                                        $original use original title instead of current title
     * @return string
     */
    protected function getStyleDirectoryName(ImageStyles $style, bool $original = false): string
    {
        return $original ? $style->getOriginal('title') : $style->title;
    }

    /**
     * Event subscription
     *
     * @param mixed $events
     */
    public function subscribe($events): void
    {
        if (! Schema::hasModel('image_styles')) {
            return;
        }

        $events->listen(
            'Inside\Content\Events\ContentFullyInsertedEvent',
            'Inside\Content\Listeners\ImageStylesSubscriber@handleContentFullyInserted'
        );

        $events->listen(
            'Inside\Content\Events\ContentUpdatingEvent',
            'Inside\Content\Listeners\ImageStylesSubscriber@handleContentUpdating'
        );

        $events->listen(
            'Inside\Content\Events\ContentDeletedEvent',
            'Inside\Content\Listeners\ImageStylesSubscriber@handleContentDeleted'
        );

        $events->listen(
            'Inside\Content\Events\ContentFieldUpdatingEvent',
            'Inside\Content\Listeners\ImageStylesSubscriber@handleContentFieldUpdating'
        );

        $events->listen(
            'Inside\Content\Events\ContentFieldDeletedEvent',
            'Inside\Content\Listeners\ImageStylesSubscriber@handleContentFieldDeleted'
        );
    }
}
