<?php

declare(strict_types=1);

namespace Inside\Content\Listeners;

use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Content\Events\ContentFullyInsertedEvent;
use Inside\Content\Events\ContentFullyUpdatedEvent;
use Inside\Content\Events\ContentSavedWithImages;
use Inside\Content\Models\Content;
use Inside\Content\Models\Section;

abstract class ContentFullySavedWithImagesListener
{
    abstract protected function handleContent(Content $content): void;

    protected function checkContent(Content $content): bool
    {
        return true;
    }

    public function __invoke(ContentFullyInsertedEvent|ContentFullyUpdatedEvent|ContentSavedWithImages $event): void
    {
        $content = $this->getContentFromEvent($event);
        if (! $content instanceof Content) {
            return;
        }

        if (! $this->checkContent($content)) {
            return;
        }

        if ($this->isLocked($event)) {
            Log::debug(__(
                '[:class] Got event :event but locked => listener is already working',
                [
                    'class' => get_class($this),
                    'event' => get_class($event),
                    'key' => $this->getKey($content),
                ]
            ));

            return;
        }
        $this->startWorking($event);
        Log::debug(__(
            '[:class] Got event :event => start working (:key) => <:state>',
            [
                'class' => get_class($this),
                'event' => get_class($event),
                'key' => $this->getKey($content),
                'state' => 'starting_'.$this->getContentState($event),
            ]
        ));
        $startAt = now();
        while ($this->needToWait($event) && $startAt->diffInMinutes() <= 5) {
            Log::debug(__(
                '[:class] Got event :event => waiting for concurrent listener to finish(:key) since :minute min',
                [
                    'class' => get_class($this),
                    'event' => get_class($event),
                    'key' => $this->getKey($content),
                    'minute' => $startAt->diffInMinutes(),
                ]
            ));
            sleep(1);
        }

        if (! $this->contentIsReady($event)) {
            Log::debug(__(
                '[:class] Got event :event but not ready to handle content (:key) => <:state>',
                [
                    'class' => get_class($this),
                    'event' => get_class($event),
                    'key' => $this->getKey($content),
                    'state' => $this->getContentState($event),
                ]
            ));
            $this->saveContentState($event);

            return;
        }

        Log::debug(__(
            '[:class] Got event :event and is now ready to handle content (:key) => <:state>',
            [
                'class' => get_class($this),
                'event' => get_class($event),
                'key' => $this->getKey($content),
                'state' => $this->getContentState($event),
            ]
        ));

        $this->lock($event);
        $this->handleContent($content);

        $this->resetContentState($event);
    }

    protected function contentIsReady(
        ContentFullyInsertedEvent|ContentFullyUpdatedEvent|ContentSavedWithImages $event
    ): bool {
        $content = $this->getContentFromEvent($event);
        if (! $content instanceof Content) {
            return false;
        }
        $key = $this->getKey($content);

        return ! is_null($content->uuid)
            && Cache::has($key)
            && Cache::get($key) === $this->getWantedEventValue($event);
    }

    protected function saveContentState(
        ContentFullyInsertedEvent|ContentFullyUpdatedEvent|ContentSavedWithImages $event
    ): void {
        $content = $this->getContentFromEvent($event);
        if (! $content instanceof Content) {
            return;
        }
        Cache::put($this->getKey($content), $this->getEventValue($event), now()->addMinutes(15));
    }

    protected function getContentState(
        ContentFullyInsertedEvent|ContentFullyUpdatedEvent|ContentSavedWithImages $event
    ): string {
        $content = $this->getContentFromEvent($event);
        if (! $content instanceof Content) {
            return 'not_set';
        }

        return Cache::get($this->getKey($content), 'not_set');
    }

    protected function resetContentState(
        ContentFullyInsertedEvent|ContentFullyUpdatedEvent|ContentSavedWithImages $event
    ): void {
        $content = $this->getContentFromEvent($event);
        if (! $content instanceof Content) {
            return;
        }
        Cache::forget($this->getKey($content));
    }

    protected function isLocked(
        ContentFullyInsertedEvent|ContentFullyUpdatedEvent|ContentSavedWithImages $event
    ): bool {
        $content = $this->getContentFromEvent($event);
        if (! $content instanceof Content) {
            return false;
        }
        $key = $this->getKey($content);

        return ! is_null($content->uuid)
            && Cache::has($key)
            && Cache::get($key) === 'locked';
    }

    protected function lock(
        ContentFullyInsertedEvent|ContentFullyUpdatedEvent|ContentSavedWithImages $event
    ): void {
        $content = $this->getContentFromEvent($event);
        if (! $content instanceof Content) {
            return;
        }
        Cache::put($this->getKey($content), 'lock', now()->addMinutes(15));
    }

    protected function getKey(Content $content): string
    {
        return Str::snake(str_replace('\\', ' ', get_class($this))).'_'.$content->content_type.'_'.$content->uuid;
    }

    protected function getEventValue(
        ContentFullyInsertedEvent|ContentFullyUpdatedEvent|ContentSavedWithImages $event,
        string $prefix = ''
    ): string {
        return $prefix.$event instanceof ContentSavedWithImages ? 'image' : 'content';
    }

    protected function getWantedEventValue(
        ContentFullyInsertedEvent|ContentFullyUpdatedEvent|ContentSavedWithImages $event,
        string $prefix = ''
    ): string {
        return $prefix.$event instanceof ContentSavedWithImages ? 'content' : 'image';
    }

    protected function getContentFromEvent(
        ContentFullyInsertedEvent|ContentFullyUpdatedEvent|ContentSavedWithImages $event
    ): Content|Section {
        return $event instanceof ContentSavedWithImages ? $event->content : $event->model;
    }

    protected function startWorking(ContentFullyUpdatedEvent|ContentFullyInsertedEvent|ContentSavedWithImages $event): void
    {
        $content = $this->getContentFromEvent($event);
        if (! $content instanceof Content) {
            return;
        }
        Cache::put(
            $this->getKey($content),
            $this->getEventValue($event, 'starting_'),
            now()->addMinutes(15)
        );
    }

    protected function needToWait(ContentFullyUpdatedEvent|ContentFullyInsertedEvent|ContentSavedWithImages $event): bool
    {
        $content = $this->getContentFromEvent($event);
        if (! $content instanceof Content) {
            return false;
        }
        $key = $this->getKey($content);

        return ! is_null($content->uuid)
            && Cache::has($key)
            && Cache::get($key) === $this->getWantedEventValue($event, 'starting_');
    }
}
